#!/bin/bash

# BayarCash Payment Gateway Module - Package Script
# This script creates a distribution-ready package of the BayarCash module

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuration
MODULE_NAME="BayarCash"
VERSION="2.0.0"
PACKAGE_NAME="bayarcash-payment-gateway-v${VERSION}"
BUILD_DIR="build"
TEMP_DIR="${BUILD_DIR}/temp"
PACKAGE_DIR="${BUILD_DIR}/${PACKAGE_NAME}"

# Functions
print_header() {
    echo -e "\n${BLUE}=== $1 ===${NC}"
}

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

cleanup() {
    if [ -d "$TEMP_DIR" ]; then
        rm -rf "$TEMP_DIR"
        print_success "Cleaned up temporary files"
    fi
}

# Main packaging function
package_module() {
    print_header "BayarCash Payment Gateway Module Packager v${VERSION}"

    # Cleanup any previous builds
    cleanup

    # Create build directories
    print_header "Setting up build environment"
    mkdir -p "$BUILD_DIR"
    mkdir -p "$TEMP_DIR"
    mkdir -p "$PACKAGE_DIR"
    print_success "Created build directories"

    # Copy module files
    print_header "Copying module files"

    # Core files
    cp -r Http "$PACKAGE_DIR/"
    cp -r Services "$PACKAGE_DIR/"
    cp -r Providers "$PACKAGE_DIR/"
    cp -r Routes "$PACKAGE_DIR/"
    cp -r resources "$PACKAGE_DIR/"
    cp -r Listeners "$PACKAGE_DIR/"
    print_success "Copied core module files"

    # Configuration files
    cp module.json "$PACKAGE_DIR/"
    cp composer.json "$PACKAGE_DIR/"
    if [ -f "Config/config.php" ]; then
        mkdir -p "$PACKAGE_DIR/Config"
        cp Config/config.php "$PACKAGE_DIR/Config/"
    fi
    print_success "Copied configuration files"

    # Documentation
    cp README.md "$PACKAGE_DIR/"
    cp INSTALLATION.md "$PACKAGE_DIR/"
    print_success "Copied documentation"

    # Create LICENSE file if it doesn't exist
    if [ ! -f "LICENSE" ]; then
        cat > "$PACKAGE_DIR/LICENSE" << 'EOF'
MIT License

Copyright (c) 2024 BayarCash Payment Gateway Module

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
EOF
        print_success "Created LICENSE file"
    else
        cp LICENSE "$PACKAGE_DIR/"
        print_success "Copied LICENSE file"
    fi

    # Create archive
    print_header "Creating distribution archive"

    cd "$BUILD_DIR"
    zip -r "${PACKAGE_NAME}.zip" "$MODULE_NAME/" -x "*.DS_Store" "*/.*" > /dev/null
    print_success "Created ZIP archive: ${PACKAGE_NAME}.zip"

    # Create tar.gz archive
    tar -czf "${PACKAGE_NAME}.tar.gz" "$MODULE_NAME/" --exclude="*.DS_Store" --exclude="*/.*"
    print_success "Created TAR.GZ archive: ${PACKAGE_NAME}.tar.gz"

    cd ..

    # Generate checksums
    print_header "Generating checksums"

    cd "$BUILD_DIR"
    sha256sum "${PACKAGE_NAME}.zip" > "${PACKAGE_NAME}.zip.sha256"
    sha256sum "${PACKAGE_NAME}.tar.gz" > "${PACKAGE_NAME}.tar.gz.sha256"
    print_success "Generated SHA256 checksums"

    cd ..

    # Create installation script
    print_header "Creating installation script"

    cat > "$BUILD_DIR/install.sh" << 'EOF'
#!/bin/bash

# BayarCash Payment Gateway Module - Quick Install Script

set -e

# Colors
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m'

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

# Check if we're in a Laravel project
if [ ! -f "artisan" ]; then
    print_error "This doesn't appear to be a Laravel project (artisan not found)"
    exit 1
fi

# Check if Modules directory exists
if [ ! -d "Modules" ]; then
    print_warning "Modules directory not found, creating..."
    mkdir -p Modules
fi

# Extract and install module
if [ -f "bayarcash-payment-gateway-*.zip" ]; then
    PACKAGE_FILE=$(ls bayarcash-payment-gateway-*.zip | head -n 1)
    print_success "Found package: $PACKAGE_FILE"

    unzip -o "$PACKAGE_FILE" -d Modules/ > /dev/null
    print_success "Extracted module to Modules/BayarCash"

    # Install composer dependencies
    if command -v composer &> /dev/null; then
        print_success "Installing composer dependencies..."
        composer require webimpian/bayarcash-php-sdk:^2.0
        print_success "Dependencies installed"
    else
        print_warning "Composer not found. Please install dependencies manually:"
        echo "  composer require webimpian/bayarcash-php-sdk:^2.0"
    fi

    # Clear cache
    php artisan cache:clear > /dev/null 2>&1
    php artisan config:clear > /dev/null 2>&1
    php artisan route:clear > /dev/null 2>&1
    print_success "Cleared application cache"

    echo ""
    print_success "BayarCash Payment Gateway module installed successfully!"
    echo ""
    echo "Next steps:"
    echo "1. Configure your BayarCash credentials in the admin panel"
    echo "2. Enable the payment gateway"
    echo "3. Test the payment flow"
    echo ""
    echo "For detailed instructions, see: Modules/BayarCash/INSTALLATION.md"

else
    print_error "Package file not found. Please ensure the ZIP file is in the current directory."
    exit 1
fi
EOF

    chmod +x "$BUILD_DIR/install.sh"
    print_success "Created installation script"

    # Create package info file
    cat > "$BUILD_DIR/PACKAGE_INFO.md" << EOF
# BayarCash Payment Gateway Module - Package Information

## Package Details
- **Module Name**: $MODULE_NAME
- **Version**: $VERSION
- **Package Date**: $(date '+%Y-%m-%d %H:%M:%S')
- **Package Size**: $(du -h "$BUILD_DIR/${PACKAGE_NAME}.zip" | cut -f1)

## Contents
- Core module files
- Documentation (README.md, INSTALLATION.md)
- Configuration files
- License
- Installation script

## Installation
1. Extract the package to your Laravel project's Modules directory
2. Run the installation script: \`bash install.sh\`
3. Configure the module in your admin panel

## Checksums
- ZIP: \`$(cat "$BUILD_DIR/${PACKAGE_NAME}.zip.sha256" | cut -d' ' -f1)\`
- TAR.GZ: \`$(cat "$BUILD_DIR/${PACKAGE_NAME}.tar.gz.sha256" | cut -d' ' -f1)\`

## Support
For installation help and troubleshooting, refer to the INSTALLATION.md file included in the package.
EOF

    print_success "Created package information file"

    # Final summary
    print_header "Package Summary"
    echo "Package created successfully!"
    echo ""
    echo "Files created:"
    echo "  📦 ${BUILD_DIR}/${PACKAGE_NAME}.zip"
    echo "  📦 ${BUILD_DIR}/${PACKAGE_NAME}.tar.gz"
    echo "  🔍 ${BUILD_DIR}/${PACKAGE_NAME}.zip.sha256"
    echo "  🔍 ${BUILD_DIR}/${PACKAGE_NAME}.tar.gz.sha256"
    echo "  📋 ${BUILD_DIR}/PACKAGE_INFO.md"
    echo "  🔧 ${BUILD_DIR}/install.sh"
    echo ""
    echo "Package size: $(du -h "$BUILD_DIR/${PACKAGE_NAME}.zip" | cut -f1)"
    echo ""
    print_success "Ready for distribution! 🚀"
}

# Check dependencies
check_dependencies() {
    if ! command -v zip &> /dev/null; then
        print_error "zip is required but not installed. Please install zip first."
        exit 1
    fi
}

# Main execution
main() {
    check_dependencies
    package_module
    cleanup
}

# Run main function
main "$@"