# BayarCash Payment Gateway Module

A comprehensive payment gateway module for Laravel applications that integrates with BayarCash to support multiple Malaysian payment methods including FPX, DuitNow QR, Credit Cards, and E-Wallets.

## 🚀 Features

- **Multiple Payment Methods**: FPX, DuitNow QR, Credit Cards, E-Wallets
- **Subscription Support**: Automatic subscription activation after successful payments
- **Webhook Processing**: Real-time payment status updates
- **Multi-Currency**: Support for MYR, USD, SGD
- **Comprehensive Logging**: Detailed logging for debugging and monitoring
- **Security**: HMAC SHA256 signature validation for webhooks
- **Error Handling**: Robust error handling with detailed logging
- **Laravel Integration**: Native Laravel module architecture

## 📋 Requirements

- PHP 8.1 or higher
- Laravel 10.0 or 11.0
- BayarCash PHP SDK 2.0+
- MySQL/PostgreSQL database

## 🛠 Installation

### 1. Install via Composer

```bash
composer require modules/bayarcash
```

### 2. Manual Installation

1. Copy the `BayarCash` module directory to your `Modules/` folder
2. Install the BayarCash SDK:
   ```bash
   composer require webimpian/bayarcash-php-sdk:^2.0
   ```

### 3. Register the Module

Add the service provider to your `config/app.php`:

```php
'providers' => [
    // Other providers...
    Modules\BayarCash\Providers\BayarCashServiceProvider::class,
],
```

### 4. Publish Configuration

```bash
php artisan vendor:publish --provider="Modules\BayarCash\Providers\BayarCashServiceProvider"
```

### 5. Configure Environment Variables

Add these to your `.env` file:

```env
BAYARCASH_PORTAL_KEY=your_portal_key
BAYARCASH_PAT_KEY=your_pat_key
BAYARCASH_SECRET_KEY=your_secret_key
BAYARCASH_SANDBOX=true
BAYARCASH_ENABLED=true
```

## ⚙️ Configuration

### BayarCash Credentials

1. **Portal Key**: Your BayarCash portal key
2. **PAT Key**: Personal Access Token from BayarCash
3. **Secret Key**: Secret key for webhook signature validation
4. **Sandbox Mode**: Set to `true` for testing, `false` for production

### Payment Settings

Configure the payment gateway in your admin panel:

1. Navigate to **Settings > Payment > BayarCash**
2. Enter your credentials
3. Enable the gateway
4. Test the configuration

## 🔄 Usage

### Basic Payment Flow

1. **Customer initiates payment**: Customer clicks pay on an invoice
2. **Redirect to BayarCash**: System redirects to BayarCash payment page
3. **Customer completes payment**: Customer selects payment method and pays
4. **Return to application**: Customer is redirected back with payment status
5. **Webhook confirmation**: BayarCash sends webhook for payment confirmation
6. **Invoice activation**: Invoice is marked as paid and subscription is activated

### Programmatic Usage

```php
// Get the BayarCash gateway
$gateway = app('bayarcash.gateway');

// Initialize payment
$paymentResult = $gateway->initializePayment($invoice, [
    'final_amount' => 100.00,
    'remaining_credit' => 0,
]);

// Verify payment
$transaction = Transaction::find($transactionId);
$result = $gateway->verify($transaction);

if ($result->isDone()) {
    // Payment successful
    echo "Payment completed successfully!";
}
```

### Webhook Handling

The module automatically handles webhooks at:
- `{subdomain}/payment/bayarcash/webhook`
- `{subdomain}/payment/bayarcash/callback`

Configure these URLs in your BayarCash dashboard.

## 🔐 Security

### Webhook Signature Validation

All webhooks are validated using HMAC SHA256 signatures to ensure authenticity:

```php
$isValid = $gateway->validateWebhookSignature($payload, $signature);
```

### Rate Limiting

Webhook endpoints are rate-limited to prevent abuse:
- 100 requests per minute per IP

## 📊 Logging

The module provides comprehensive logging for monitoring and debugging:

```php
// Payment initialization
Log::info('BayarCash payment initialization', [
    'invoice_id' => $invoice->id,
    'amount' => $finalAmount,
]);

// Payment verification
Log::info('BayarCash payment verified', [
    'transaction_id' => $transaction->id,
    'status' => $paymentStatus,
]);

// Webhook processing
Log::info('BayarCash webhook received', [
    'status' => $webhookStatus,
    'transaction_id' => $transactionId,
]);
```

## 🧪 Testing

### Sandbox Mode

Enable sandbox mode for testing:

```env
BAYARCASH_SANDBOX=true
```

### Test Payment Flow

1. Create a test invoice
2. Navigate to payment page
3. Use BayarCash test credentials
4. Complete test payment
5. Verify webhook reception

### Test Webhook

You can test webhooks using tools like ngrok:

```bash
# Install ngrok
npm install -g ngrok

# Expose local server
ngrok http 80

# Use the ngrok URL for webhook configuration
```

## 🔧 Troubleshooting

### Common Issues

**1. Payment gateway not showing**
- Check if `BAYARCASH_ENABLED=true` in `.env`
- Verify credentials are correct
- Check logs for initialization errors

**2. Webhook not receiving**
- Verify webhook URL is accessible
- Check signature validation
- Review webhook logs in BayarCash dashboard

**3. Payment not activating subscription**
- Check if webhook is being received
- Verify invoice marking logic
- Review payment verification logs

### Debug Mode

Enable debug logging:

```php
Log::debug('BayarCash Debug', [
    'gateway_active' => $gateway->isActive(),
    'credentials_set' => !empty($portalKey),
]);
```

## 📈 Monitoring

### Key Metrics to Monitor

1. **Payment Success Rate**: Track successful vs failed payments
2. **Webhook Delivery**: Monitor webhook reception and processing
3. **Response Times**: Track payment processing times
4. **Error Rates**: Monitor error logs for issues

### Health Checks

```php
// Check gateway health
if ($gateway->isActive()) {
    echo "BayarCash gateway is operational";
} else {
    echo "BayarCash gateway is not configured";
}
```

## 🔄 Updates

### Updating the Module

1. Backup your current configuration
2. Update via Composer:
   ```bash
   composer update modules/bayarcash
   ```
3. Clear cache:
   ```bash
   php artisan cache:clear
   php artisan config:clear
   ```

## 🤝 Support

### Getting Help

1. **Documentation**: Review this README and inline code documentation
2. **Logs**: Check Laravel logs for detailed error information
3. **BayarCash Support**: Contact BayarCash for API-related issues
4. **Module Issues**: Create an issue in the module repository

### Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests
5. Submit a pull request

## 📄 License

This module is open-sourced software licensed under the [MIT license](LICENSE).

## 🙏 Credits

- [BayarCash](https://bayarcash.com) for the payment gateway API
- [Webimpian](https://github.com/webimpian) for the BayarCash PHP SDK
- Laravel community for the framework

## 📋 Changelog

### v2.0.0
- ✅ Fixed invoice payment automation
- ✅ Improved error handling and logging
- ✅ Enhanced webhook processing
- ✅ Added comprehensive documentation
- ✅ Optimized code structure
- ✅ Added proper signature validation

### v1.0.0
- 🎉 Initial release
- ✅ Basic payment processing
- ✅ Webhook support
- ✅ Multi-currency support