<?php

namespace Modules\BayarCash\Providers;

use Illuminate\Support\ServiceProvider;

class BayarCashServiceProvider extends ServiceProvider
{
    /**
     * The module name.
     *
     * @var string
     */
    protected $moduleName = 'BayarCash';

    /**
     * Boot the application events.
     *
     * @return void
     */
    public function boot()
    {
        $this->registerTranslations();
        $this->registerConfig();
        $this->registerViews();
        $this->loadMigrationsFrom(module_path($this->moduleName, 'Database/Migrations'));
        $this->registerRoutes();
        $this->registerEventListeners();
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        // Register the BayarCash payment gateway
        $this->app->bind('bayarcash.gateway', function ($app) {
            $settings = app(\App\Settings\PaymentSettings::class);

            // Ensure values are properly cast to expected types
            $portalKey = (string) ($settings->bayarcash_portal_key ?? '');
            $patKey = (string) ($settings->bayarcash_pat_key ?? '');
            $apiSecretKey = (string) ($settings->bayarcash_api_secret_key ?? '');
            $sandbox = (bool) ($settings->bayarcash_sandbox ?? true);

            return new \Modules\BayarCash\Services\BayarCashPaymentGateway(
                $portalKey,
                $patKey,
                $apiSecretKey,
                $sandbox
            );
        });

        // Register payment gateway singleton
        $this->app->singleton(\Modules\BayarCash\Services\BayarCashPaymentGateway::class, function ($app) {
            return $app->make('bayarcash.gateway');
        });
    }

    /**
     * Register translations.
     *
     * @return void
     */
    protected function registerTranslations()
    {
        $langPath = resource_path('lang/modules/' . strtolower($this->moduleName));

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, $this->moduleName);
        } else {
            $this->loadTranslationsFrom(module_path($this->moduleName, 'resources/lang'), $this->moduleName);
        }
    }

    /**
     * Register config.
     *
     * @return void
     */
    protected function registerConfig()
    {
        $this->publishes([
            module_path($this->moduleName, 'Config/config.php') => config_path($this->moduleName . '.php'),
        ], 'config');
        $this->mergeConfigFrom(
            module_path($this->moduleName, 'Config/config.php'), $this->moduleName
        );
    }

    /**
     * Register views.
     *
     * @return void
     */
    protected function registerViews()
    {
        $viewPath = resource_path('views/modules/' . strtolower($this->moduleName));

        $sourcePath = module_path($this->moduleName, 'resources/views');

        $this->publishes([
            $sourcePath => $viewPath
        ], 'views');

        $this->loadViewsFrom(array_merge([$sourcePath], [
            $viewPath
        ]), $this->moduleName);
    }

    /**
     * Register routes.
     *
     * @return void
     */
    protected function registerRoutes()
    {
        $this->loadRoutesFrom(module_path($this->moduleName, 'Routes/web.php'));
        $this->loadRoutesFrom(module_path($this->moduleName, 'Routes/api.php'));
    }

    /**
     * Register event listeners.
     *
     * @return void
     */
    protected function registerEventListeners()
    {
        // Register event listener for extending payment settings
        $this->app['events']->listen(
            \App\Events\PaymentSettingsExtending::class,
            \Modules\BayarCash\Listeners\ExtendPaymentSettings::class
        );

        // Register event listener for adding BayarCash to payment settings view
        $this->app['events']->listen(
            \App\Events\PaymentSettingsViewRendering::class,
            \Modules\BayarCash\Listeners\AddBayarCashToPaymentSettings::class
        );

        // Register event listener for payment gateway registration
        $this->app['events']->listen(
            \App\Events\PaymentGatewayRegistration::class,
            \Modules\BayarCash\Listeners\RegisterBayarCashPaymentGateway::class
        );
    }

    /**
     * Get the services provided by the provider.
     *
     * @return array
     */
    public function provides()
    {
        return [];
    }
}
