# BayarCash Production 500 Error - Troubleshooting Guide

## 🚨 **Issue**: Works on `wao.test` but 500 error on `wao.my`

This is a common production vs local environment issue. Here's how to diagnose and fix:

## 🔍 **Step 1: Check Production Logs**

SSH into your production server and check Laravel logs:

```bash
# Check today's log
tail -50 /var/www/wao.my/storage/logs/laravel-$(date +%Y-%m-%d).log

# Or check the general log
tail -50 /var/www/wao.my/storage/logs/laravel.log

# Look for errors related to BayarCash
grep -i "bayarcash\|500\|error" /var/www/wao.my/storage/logs/laravel-$(date +%Y-%m-%d).log
```

## 🔧 **Step 2: Common Production Issues**

### A. Missing Dependencies
```bash
# On production server
cd /var/www/wao.my
composer require webimpian/bayarcash-php-sdk:^2.0
```

### B. Autoloader Issues
```bash
# Regenerate autoloader
composer dump-autoload --optimize

# Clear all caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### C. Module Not Active
```bash
# Check module status
php artisan module:list | grep BayarCash

# Activate if needed
php artisan module:activate BayarCash
```

### D. File Permissions
```bash
# Fix permissions
sudo chown -R www-data:www-data /var/www/wao.my/Modules/BayarCash/
sudo chmod -R 755 /var/www/wao.my/Modules/BayarCash/
```

## 🔍 **Step 3: Specific Checks**

### Check if Routes Exist
```bash
php artisan route:list --name=admin.settings.payment.bayarcash
```

### Check if Module Files Exist
```bash
ls -la /var/www/wao.my/Modules/BayarCash/
ls -la /var/www/wao.my/Modules/BayarCash/Http/Controllers/Admin/
```

### Check PaymentSettings Class
```bash
php artisan tinker
```
In tinker:
```php
$settings = app(\App\Settings\PaymentSettings::class);
// Check if this works without error
dd(method_exists($settings, 'isDynamicProperty'));
```

## 🐛 **Step 4: Common Production Errors**

### Error: "Class not found"
**Solution:**
```bash
composer dump-autoload --optimize
php artisan cache:clear
```

### Error: "Route not defined"
**Solution:**
```bash
php artisan route:clear
php artisan cache:clear
```

### Error: "Property does not exist"
**Solution:**
Check if production `PaymentSettings` class has dynamic properties support.

### Error: "Module not found"
**Solution:**
```bash
# Ensure module.json exists and is valid
cat /var/www/wao.my/Modules/BayarCash/module.json

# Re-activate module
php artisan module:activate BayarCash
```

## 🔧 **Step 5: Production Environment Differences**

### Check PHP Version
```bash
php -v
# Ensure PHP 8.1+
```

### Check Laravel Version
```bash
php artisan --version
# Ensure Laravel 10.0+
```

### Check Environment Settings
```bash
# Check if production has correct settings
php artisan config:show | grep -i module
```

## 📋 **Step 6: Manual Debug Process**

Create a test route to debug:

**File**: `/var/www/wao.my/routes/web.php`
Add temporarily:
```php
Route::get('/debug-bayarcash', function() {
    try {
        // Test 1: Check if module exists
        $moduleExists = file_exists(base_path('Modules/BayarCash/module.json'));

        // Test 2: Check if settings class works
        $settings = app(\App\Settings\PaymentSettings::class);

        // Test 3: Check if gateway can be resolved
        $gateway = app('bayarcash.gateway');

        return response()->json([
            'module_exists' => $moduleExists,
            'settings_class' => get_class($settings),
            'gateway_name' => $gateway->getName(),
            'status' => 'success'
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ], 500);
    }
});
```

Then visit: `https://wao.my/debug-bayarcash`

## 🎯 **Most Likely Solutions**

Based on common production issues:

1. **Missing SDK**: `composer require webimpian/bayarcash-php-sdk:^2.0`
2. **Cache Issues**: `php artisan cache:clear && php artisan config:clear`
3. **Autoloader**: `composer dump-autoload --optimize`
4. **Module Inactive**: `php artisan module:activate BayarCash`

## 📞 **Next Steps**

Run these commands on production and share the output:

```bash
# 1. Check module status
php artisan module:list | grep BayarCash

# 2. Check routes
php artisan route:list --name=bayarcash

# 3. Check Laravel logs
tail -20 storage/logs/laravel-$(date +%Y-%m-%d).log

# 4. Test gateway
php artisan tinker --execute="echo app('bayarcash.gateway')->getName();"
```

This will help identify the specific production issue!