<?php

namespace Modules\BayarCash\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Invoice\Invoice;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Modules\BayarCash\Services\BayarCashDuitNowQrPaymentGateway;

/**
 * BayarCash DuitNow QR Payment Controller
 *
 * Handles DuitNow QR payment processing through BayarCash.
 *
 * @package Modules\BayarCash\Http\Controllers
 */
class BayarCashDuitNowQrController extends Controller
{
    protected BayarCashDuitNowQrPaymentGateway $gateway;

    public function __construct()
    {
        $this->gateway = app('bayarcash.duitnow.gateway');
    }

    /**
     * Handle DuitNow QR checkout process
     */
    public function checkout(Request $request, string $subdomain, Invoice $invoice): RedirectResponse
    {
        try {
            // Validate gateway availability
            if (!$this->gateway->isActive()) {
                return redirect()->route('tenant.invoices.show', [$subdomain, $invoice])
                    ->with('error', 'BayarCash DuitNow QR payment gateway is not available.');
            }

            // Validate invoice status
            if ($invoice->isPaid()) {
                return redirect()->route('tenant.invoices.show', [$subdomain, $invoice])
                    ->with('info', 'This invoice has already been paid.');
            }

            // Validate payment amount
            $paymentData = $this->validatePaymentAmount($request, $invoice);
            if (!$paymentData['valid']) {
                return redirect()->route('tenant.invoices.show', [$subdomain, $invoice])
                    ->with('error', $paymentData['message']);
            }

            // Process DuitNow QR payment
            $paymentResult = $this->processPayment($invoice, $paymentData);

            // Redirect to BayarCash DuitNow QR payment page
            return redirect($paymentResult['payment_url']);

        } catch (\Exception $e) {
            \Log::error('BayarCash DuitNow QR checkout error', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
            ]);

            return redirect()->route('tenant.invoices.show', [$subdomain, $invoice])
                ->with('error', 'Unable to process DuitNow QR payment. Please try again.');
        }
    }

    /**
     * Validate payment amount and requirements
     */
    protected function validatePaymentAmount(Request $request, Invoice $invoice): array
    {
        $remainingCredit = $request->get('remaining_credit', 0);
        $finalAmount = $invoice->total() - $remainingCredit;
        $currency = $invoice->currency->code ?? 'MYR';
        $minimumAmount = $this->gateway->getMinimumChargeAmount($currency);

        if ($finalAmount < $minimumAmount) {
            return [
                'valid' => false,
                'message' => "Minimum DuitNow QR payment amount is {$minimumAmount} {$currency}."
            ];
        }

        return [
            'valid' => true,
            'remaining_credit' => $remainingCredit,
            'final_amount' => $finalAmount,
            'currency' => $currency
        ];
    }

    /**
     * Process DuitNow QR payment with BayarCash
     */
    protected function processPayment(Invoice $invoice, array $paymentData): array
    {
        // Create pending transaction
        $transaction = $invoice->createPendingTransaction($this->gateway, $invoice->tenant_id);

        // Initialize DuitNow QR payment with BayarCash
        $paymentResult = $this->gateway->initializePayment($invoice, [
            'transaction_id' => $transaction->id,
            'remaining_credit' => $paymentData['remaining_credit'],
            'final_amount' => $paymentData['final_amount'],
        ]);

        // Update transaction with BayarCash DuitNow QR details
        $transaction->update([
            'idempotency_key' => $paymentResult['transaction_id'],
            'metadata' => [
                'order_id' => $paymentResult['order_id'],
                'bayarcash_payment_intent_id' => $paymentResult['transaction_id'],
                'bayarcash_order_id' => $paymentResult['order_id'],
                'payment_url' => $paymentResult['payment_url'],
                'payment_method' => 'duitnow_qr',
                'payment_channel' => 'DuitNow QR',
                'remaining_credit' => $paymentData['remaining_credit'],
                'final_amount' => $paymentData['final_amount'],
            ],
        ]);

        return $paymentResult;
    }
}