<?php

namespace Modules\BayarCash\Services;

use App\Events\TransactionCreated;
use App\Models\Invoice\Invoice;
use App\Models\Transaction;
use App\Services\Billing\TransactionResult;
use App\Services\PaymentGateways\Contracts\PaymentGatewayInterface;
use App\Settings\PaymentSettings;
use Exception;
use Illuminate\Support\Facades\Log;
use Webimpian\BayarcashSdk\Bayarcash;

/**
 * BayarCash Payment Gateway
 *
 * A comprehensive payment gateway integration for BayarCash that supports
 * subscription payments, webhooks, and multiple payment methods for Malaysian market.
 *
 * Features:
 * - FPX, DuitNow QR, Credit Cards, E-Wallets support
 * - Subscription payment handling with automatic activation
 * - Webhook processing for real-time payment updates
 * - Multi-currency support (MYR, USD, SGD)
 * - Comprehensive error handling and logging
 * - Secure signature validation
 *
 * @package Modules\BayarCash\Services
 * @version 3.0.0
 */
class BayarCashPaymentGateway implements PaymentGatewayInterface
{
    /**
     * Gateway type identifier
     */
    public const TYPE = 'bayarcash_fpx';

    /**
     * Payment status constants
     */
    public const STATUS_PENDING = '1';
    public const STATUS_FAILED = '2';
    public const STATUS_SUCCESSFUL = '3';

    /**
     * BayarCash credentials
     */
    protected string $portalKey;
    protected string $patKey;
    protected string $secretKey;
    protected bool $sandbox;
    protected bool $active = false;

    /**
     * BayarCash SDK instance
     */
    protected ?Bayarcash $bayarCash = null;

    /**
     * Supported currencies and minimum amounts
     */
    protected array $minimumAmounts = [
        'MYR' => 1.00,
        'USD' => 0.50,
        'SGD' => 1.00,
    ];

    /**
     * Create a new BayarCash payment gateway instance.
     */
    public function __construct(string $portalKey, string $patKey, string $secretKey, bool $sandbox = true)
    {
        $this->portalKey = $portalKey;
        $this->patKey = $patKey;
        $this->secretKey = $secretKey;
        $this->sandbox = $sandbox;

        $this->validate();
        $this->initializeSdk();
    }

    /**
     * Validate gateway configuration
     */
    public function validate(): void
    {
        $hasCredentials = !empty($this->portalKey) && !empty($this->patKey) && !empty($this->secretKey);
        $settings = app(PaymentSettings::class);
        $isEnabled = $settings->bayarcash_enabled ?? false;

        $this->active = $hasCredentials && $isEnabled;
    }

    /**
     * Initialize BayarCash SDK
     */
    protected function initializeSdk(): void
    {
        if (!$this->active) {
            return;
        }

        try {
            $this->bayarCash = new Bayarcash($this->patKey);

            if ($this->sandbox) {
                $this->bayarCash->useSandbox();
            }

            $this->bayarCash->setApiVersion('v3');

            Log::info('BayarCash SDK initialized successfully', [
                'sandbox' => $this->sandbox,
                'portal_key' => $this->portalKey,
            ]);
        } catch (Exception $e) {
            Log::error('BayarCash SDK initialization failed', [
                'error' => $e->getMessage(),
                'sandbox' => $this->sandbox,
            ]);
            $this->active = false;
        }
    }

    /**
     * Get the gateway name
     */
    public function getName(): string
    {
        return 'BayarCash FPX';
    }

    /**
     * Get the gateway type
     */
    public function getType(): string
    {
        return self::TYPE;
    }

    /**
     * Get the gateway description
     */
    public function getDescription(): string
    {
        return 'Pay securely using FPX Online Banking - Direct bank transfer from your Malaysian bank account.';
    }

    /**
     * Get the gateway short description
     */
    public function getShortDescription(): string
    {
        return 'FPX Online Banking';
    }

    /**
     * Check if the gateway is active
     */
    public function isActive(): bool
    {
        return $this->active;
    }

    /**
     * Get the settings URL
     */
    public function getSettingsUrl(): string
    {
        return route('admin.settings.payment.bayarcash');
    }

    /**
     * Get the checkout URL for an invoice
     */
    public function getCheckoutUrl(Invoice $invoice): string
    {
        return tenant_route('tenant.payment.bayarcash.checkout', ['invoice' => $invoice->id]);
    }

    /**
     * Check if auto-billing is supported
     */
    public function supportsAutoBilling(): bool
    {
        return false;
    }

    /**
     * Get auto-billing update URL
     */
    public function getAutoBillingDataUpdateUrl(string $returnUrl = '/'): string
    {
        return $returnUrl;
    }

    /**
     * Get minimum charge amount for currency
     */
    public function getMinimumChargeAmount($currency): float
    {
        return $this->minimumAmounts[$currency] ?? 1.00;
    }

    /**
     * Initialize payment with BayarCash
     */
    public function initializePayment(Invoice $invoice, array $metadata = []): array
    {
        if (!$this->active || !$this->bayarCash) {
            throw new Exception('BayarCash gateway is not properly configured');
        }

        $finalAmount = round($metadata['final_amount'] ?? $invoice->total(), 2);
        $remainingCredit = $metadata['remaining_credit'] ?? 0;

        $this->logPaymentInitialization($invoice, $finalAmount, $remainingCredit);

        $orderId = $this->generateOrderId($invoice);
        $paymentData = $this->buildPaymentData($invoice, $orderId, $finalAmount, $metadata);

        try {
            $checksum = $this->generateChecksum($paymentData);
            $paymentData['checksum'] = $checksum;

            $paymentIntent = $this->bayarCash->createPaymentIntent($paymentData);

            if (!$paymentIntent->id) {
                throw new Exception('Failed to create BayarCash payment intent');
            }

            $this->logPaymentSuccess($paymentIntent);

            return [
                'success' => true,
                'payment_url' => $paymentIntent->url,
                'order_id' => $paymentIntent->orderNumber,
                'transaction_id' => $paymentIntent->id,
                'data' => $paymentIntent,
            ];
        } catch (Exception $e) {
            $this->logPaymentError($e, $invoice, $paymentData);
            throw $e;
        }
    }

    /**
     * Verify payment with BayarCash
     */
    public function verifyPayment(string $transactionId): array
    {
        try {
            Log::info('BayarCash payment verification started', [
                'transaction_id' => $transactionId,
            ]);

            $transaction = $this->bayarCash->getTransaction($transactionId);

            if (!$transaction->id) {
                throw new Exception('Failed to retrieve BayarCash transaction');
            }

            return [
                'success' => true,
                'data' => [
                    'id' => $transaction->id,
                    'status' => $transaction->status,
                    'amount' => $transaction->amount,
                    'currency' => $transaction->currency,
                    'order_number' => $transaction->orderNumber,
                    'payer_name' => $transaction->payerName,
                    'payer_email' => $transaction->payerEmail,
                    'metadata' => $transaction->metadata,
                    'created_at' => $transaction->createdAt,
                    'updated_at' => $transaction->updatedAt,
                ],
                'transaction' => $transaction,
            ];
        } catch (Exception $e) {
            Log::error('BayarCash payment verification failed', [
                'transaction_id' => $transactionId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'data' => [
                    'status' => 'error',
                    'error' => $e->getMessage(),
                ],
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Verify transaction and return result
     */
    public function verify(Transaction $transaction): TransactionResult
    {
        try {
            $transactionId = $this->getTransactionIdForVerification($transaction);

            if (empty($transactionId)) {
                return new TransactionResult(TransactionResult::RESULT_FAILED, 'No transaction reference provided');
            }

            $paymentData = $this->verifyPayment($transactionId);

            return $this->processVerificationResult($transaction, $paymentData);
        } catch (Exception $e) {
            Log::error('BayarCash verification failed', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
            ]);

            return new TransactionResult(TransactionResult::RESULT_FAILED, 'Verification failed');
        }
    }

    /**
     * Check if manual review is allowed
     */
    public function allowManualReviewingOfTransaction(): bool
    {
        return true;
    }

    /**
     * Auto charge (not supported)
     */
    public function autoCharge(Invoice $invoice, $remainingCredit = 0): TransactionResult
    {
        return new TransactionResult(
            TransactionResult::RESULT_FAILED,
            'BayarCash does not support automatic charging.'
        );
    }

    /**
     * Validate webhook signature
     */
    public function validateWebhookSignature(string $payload, string $signature): bool
    {
        $expectedSignature = hash_hmac('sha256', $payload, $this->secretKey);
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Handle webhook event
     */
    public function handleWebhook(array $payload): void
    {
        $status = $payload['status'] ?? '';
        $transactionId = $payload['transaction_id'] ?? '';

        Log::info('BayarCash webhook received', [
            'status' => $status,
            'transaction_id' => $transactionId,
        ]);

        switch ($status) {
            case 'completed':
            case self::STATUS_SUCCESSFUL:
                $this->handleSuccessfulPayment($payload);
                break;

            case 'failed':
            case 'cancelled':
            case self::STATUS_FAILED:
                $this->handleFailedPayment($payload);
                break;

            case 'pending':
            case self::STATUS_PENDING:
                $this->handlePendingPayment($payload);
                break;

            default:
                Log::info('Unhandled BayarCash webhook status', ['status' => $status]);
                break;
        }
    }

    /**
     * Build payment data array
     */
    protected function buildPaymentData(Invoice $invoice, string $orderId, float $finalAmount, array $metadata): array
    {
        return [
            'order_number' => $orderId,
            'amount' => $finalAmount,
            'currency' => $invoice->currency->code ?? 'MYR',
            'payment_channel' => Bayarcash::FPX,
            'payer_name' => $invoice->tenant->name ?? 'Customer',
            'payer_email' => $invoice->tenant->email ?? 'customer@example.com',
            'return_url' => tenant_route('tenant.payment.bayarcash.return', ['invoice' => $invoice->id]),
            'callback_url' => tenant_route('tenant.payment.bayarcash.callback'),
            'description' => 'Payment for Invoice #' . $invoice->id,
            'portal_key' => $this->portalKey,
            'metadata' => json_encode([
                'invoice_id' => $invoice->id,
                'tenant_id' => $invoice->tenant_id,
                'transaction_id' => $metadata['transaction_id'] ?? null,
                'remaining_credit' => $metadata['remaining_credit'] ?? 0,
                'final_amount' => $finalAmount,
                'invoice_number' => $invoice->invoice_number,
            ]),
        ];
    }

    /**
     * Generate checksum for payment data
     */
    protected function generateChecksum(array $paymentData): string
    {
        try {
            return $this->bayarCash->createPaymentIntenChecksumValue($this->secretKey, $paymentData);
        } catch (Exception $e) {
            Log::error('BayarCash checksum generation failed', [
                'error' => $e->getMessage(),
                'payment_data_keys' => array_keys($paymentData),
            ]);
            throw new Exception('Checksum generation failed: ' . $e->getMessage());
        }
    }

    /**
     * Generate unique order ID
     */
    protected function generateOrderId(Invoice $invoice): string
    {
        return 'INV_' . $invoice->id . '_' . time() . '_' . substr(md5(uniqid()), 0, 8);
    }

    /**
     * Get transaction ID for verification
     */
    protected function getTransactionIdForVerification(Transaction $transaction): ?string
    {
        return $transaction->metadata['bayarcash_transaction_id'] ?? $transaction->idempotency_key;
    }

    /**
     * Process verification result
     */
    protected function processVerificationResult(Transaction $transaction, array $paymentData): TransactionResult
    {
        if (!$paymentData['success']) {
            return new TransactionResult(TransactionResult::RESULT_FAILED, $paymentData['error'] ?? 'Verification failed');
        }

        $status = $paymentData['data']['status'];

        if ($this->isSuccessfulStatus($status)) {
            $this->markTransactionSuccessful($transaction, $paymentData);
            return new TransactionResult(TransactionResult::RESULT_DONE);
        }

        if ($this->isPendingStatus($status)) {
            return new TransactionResult(TransactionResult::RESULT_PENDING);
        }

        $this->markTransactionFailed($transaction, $paymentData);
        return new TransactionResult(TransactionResult::RESULT_FAILED, 'Payment ' . $status);
    }

    /**
     * Check if status indicates success
     */
    protected function isSuccessfulStatus($status): bool
    {
        return in_array($status, ['successful', 'completed', 'paid', self::STATUS_SUCCESSFUL, 3]);
    }

    /**
     * Check if status indicates pending
     */
    protected function isPendingStatus($status): bool
    {
        return in_array($status, ['pending', self::STATUS_PENDING, 1]);
    }

    /**
     * Mark transaction as successful
     */
    protected function markTransactionSuccessful(Transaction $transaction, array $paymentData): void
    {
        $transaction->update([
            'status' => Transaction::STATUS_SUCCESS,
            'metadata' => array_merge($transaction->metadata ?? [], [
                'bayarcash_response' => $paymentData['data'],
                'verified_at' => now()->toISOString(),
            ]),
        ]);

        $this->markInvoiceAsPaid($transaction);

        Log::info('BayarCash payment marked as successful', [
            'transaction_id' => $transaction->id,
            'bayarcash_status' => $paymentData['data']['status'],
        ]);
    }

    /**
     * Mark transaction as failed
     */
    protected function markTransactionFailed(Transaction $transaction, array $paymentData): void
    {
        $transaction->update([
            'status' => Transaction::STATUS_FAILED,
            'error' => 'Payment ' . ($paymentData['data']['status'] ?? 'unknown'),
            'metadata' => array_merge($transaction->metadata ?? [], [
                'bayarcash_response' => $paymentData['data'],
                'verified_at' => now()->toISOString(),
            ]),
        ]);
    }

    /**
     * Mark invoice as paid if not already paid
     */
    protected function markInvoiceAsPaid(Transaction $transaction): void
    {
        $invoice = $transaction->invoice;

        if ($invoice && !$invoice->isPaid()) {
            $invoice->markAsPaid();

            Log::info('BayarCash invoice marked as paid', [
                'invoice_id' => $invoice->id,
                'transaction_id' => $transaction->id,
                'subscription_id' => $invoice->subscription_id,
            ]);
        }
    }

    /**
     * Handle successful payment webhook
     */
    protected function handleSuccessfulPayment(array $data): void
    {
        $transactionId = $data['transaction_id'] ?? '';
        $metadata = json_decode($data['metadata'] ?? '{}', true);
        $invoiceId = $metadata['invoice_id'] ?? null;

        if (!$invoiceId) {
            Log::warning('BayarCash webhook: No invoice ID in metadata', [
                'transaction_id' => $transactionId
            ]);
            return;
        }

        $invoice = Invoice::find($invoiceId);
        if (!$invoice) {
            Log::warning('BayarCash webhook: Invoice not found', [
                'invoice_id' => $invoiceId
            ]);
            return;
        }

        $transaction = $this->findTransaction($transactionId, $invoice->id);
        if (!$transaction) {
            Log::warning('BayarCash webhook: Transaction not found', [
                'transaction_id' => $transactionId,
                'invoice_id' => $invoiceId,
            ]);
            return;
        }

        if ($transaction->status === Transaction::STATUS_SUCCESS) {
            Log::info('BayarCash webhook: Transaction already processed', [
                'transaction_id' => $transaction->id
            ]);
            return;
        }

        $this->processSuccessfulWebhookPayment($transaction, $invoice, $data, $metadata);
    }

    /**
     * Process successful webhook payment
     */
    protected function processSuccessfulWebhookPayment(Transaction $transaction, Invoice $invoice, array $data, array $metadata): void
    {
        $transaction->update([
            'status' => Transaction::STATUS_SUCCESS,
            'metadata' => array_merge($transaction->metadata ?? [], [
                'bayarcash_response' => $data,
                'webhook_processed_at' => now()->toISOString(),
            ]),
        ]);

        $this->markInvoiceAsPaid($transaction);

        $this->applyCreditIfNeeded($invoice, $metadata);

        event(new TransactionCreated($transaction->id, $invoice->id));
    }

    /**
     * Apply remaining credit if needed
     */
    protected function applyCreditIfNeeded(Invoice $invoice, array $metadata): void
    {
        $remainingCredit = $metadata['remaining_credit'] ?? 0;

        if ($remainingCredit > 0) {
            $invoice->tenant->deductCreditBalance($remainingCredit);
        }
    }

    /**
     * Find transaction by ID
     */
    protected function findTransaction(string $transactionId, int $invoiceId): ?Transaction
    {
        return Transaction::where('idempotency_key', $transactionId)
            ->where('invoice_id', $invoiceId)
            ->first();
    }

    /**
     * Handle failed payment webhook
     */
    protected function handleFailedPayment(array $data): void
    {
        $transactionId = $data['transaction_id'] ?? '';
        $metadata = json_decode($data['metadata'] ?? '{}', true);
        $invoiceId = $metadata['invoice_id'] ?? null;

        if (!$invoiceId) {
            Log::warning('BayarCash webhook: No invoice ID in failed payment', [
                'transaction_id' => $transactionId
            ]);
            return;
        }

        $transaction = $this->findTransaction($transactionId, $invoiceId);
        if (!$transaction) {
            Log::warning('BayarCash webhook: Transaction not found for failed payment', [
                'transaction_id' => $transactionId,
                'invoice_id' => $invoiceId,
            ]);
            return;
        }

        $transaction->update([
            'status' => Transaction::STATUS_FAILED,
            'error' => $data['failure_reason'] ?? 'Payment failed',
            'metadata' => array_merge($transaction->metadata ?? [], [
                'bayarcash_response' => $data,
                'webhook_processed_at' => now()->toISOString(),
            ]),
        ]);
    }

    /**
     * Handle pending payment webhook
     */
    protected function handlePendingPayment(array $data): void
    {
        Log::info('BayarCash webhook: Payment pending', [
            'transaction_id' => $data['transaction_id'] ?? '',
        ]);
    }

    /**
     * Log payment initialization
     */
    protected function logPaymentInitialization(Invoice $invoice, float $finalAmount, float $remainingCredit): void
    {
        Log::info('BayarCash payment initialization', [
            'invoice_id' => $invoice->id,
            'invoice_total' => $invoice->total(),
            'remaining_credit' => $remainingCredit,
            'final_amount' => $finalAmount,
            'currency' => $invoice->currency->code ?? 'MYR',
        ]);
    }

    /**
     * Log payment success
     */
    protected function logPaymentSuccess($paymentIntent): void
    {
        Log::info('BayarCash payment intent created successfully', [
            'payment_intent_id' => $paymentIntent->id,
            'payment_url' => $paymentIntent->url,
        ]);
    }

    /**
     * Log payment error
     */
    protected function logPaymentError(Exception $e, Invoice $invoice, array $paymentData): void
    {
        $errorData = [
            'invoice_id' => $invoice->id,
            'error' => $e->getMessage(),
            'payment_data' => $paymentData,
        ];

        if (method_exists($e, 'getResponse')) {
            $response = $e->getResponse();
            if ($response) {
                $errorData['response_status'] = $response->getStatusCode();
                $responseBody = (string) $response->getBody();
                $errorData['response_body'] = $responseBody;

                $decodedResponse = json_decode($responseBody, true);
                if ($decodedResponse) {
                    $errorData['response_decoded'] = $decodedResponse;
                }
            }
        }

        Log::error('BayarCash payment initialization failed', $errorData);
    }
}