<?php

use Illuminate\Support\Facades\Route;
use Modules\BayarCash\Http\Controllers\BayarCashController;
use Modules\BayarCash\Http\Controllers\BayarCashDuitNowQrController;
use Modules\BayarCash\Http\Controllers\Admin\BayarCashSettingsController;

/*
|--------------------------------------------------------------------------
| BayarCash Payment Gateway Routes
|--------------------------------------------------------------------------
|
| These routes handle BayarCash payment processing for tenant payment flows.
| All routes are secured with tenant middleware and include proper subdomain handling.
|
| Payment Channels:
| - BayarCash FPX (FPX Online Banking) - Original working route
| - BayarCash DuitNow QR (DuitNow QR) - New QR payment method
|
*/

Route::middleware(['web', 'tenant'])->group(function () {
    Route::prefix('{subdomain}/payment/bayarcash')->group(function () {
        // BayarCash FPX payment checkout route
        Route::get('/checkout/{invoice}', [BayarCashController::class, 'checkout'])
            ->name('tenant.payment.bayarcash.checkout')
            ->where('invoice', '[0-9]+')
            ->middleware('throttle:10,1'); // Limit to 10 checkout attempts per minute

        // DuitNow QR checkout route
        Route::get('/duitnow/checkout/{invoice}', [BayarCashDuitNowQrController::class, 'checkout'])
            ->name('tenant.payment.bayarcash.duitnow.checkout')
            ->where('invoice', '[0-9]+')
            ->middleware('throttle:10,1'); // Limit to 10 checkout attempts per minute

        // Payment return route (customer redirect after payment) - shared
        Route::get('/return/{invoice}', [BayarCashController::class, 'returnUrl'])
            ->name('tenant.payment.bayarcash.return')
            ->where('invoice', '[0-9]+');

        // Webhook callback route (server-to-server) - CSRF exempt via bootstrap/app.php
        Route::post('/callback', [BayarCashController::class, 'callback'])
            ->name('tenant.payment.bayarcash.callback')
            ->middleware('throttle:100,1')
            ->withoutMiddleware([\Illuminate\Foundation\Http\Middleware\VerifyCsrfToken::class]);

        // Webhook route (alternative endpoint) - CSRF exempt via bootstrap/app.php
        Route::post('/webhook', [BayarCashController::class, 'webhook'])
            ->name('tenant.payment.bayarcash.webhook')
            ->middleware('throttle:100,1')
            ->withoutMiddleware([\Illuminate\Foundation\Http\Middleware\VerifyCsrfToken::class]);
    });
});

/*
|--------------------------------------------------------------------------
| Admin Routes
|--------------------------------------------------------------------------
|
| Admin panel routes for BayarCash payment gateway configuration.
| These routes are secured with auth and admin middleware.
|
*/

Route::middleware(['web', 'auth', \App\Http\Middleware\AdminMiddleware::class])->prefix('admin/settings/payment')->group(function () {
    // BayarCash admin settings routes
    Route::get('/bayarcash', [BayarCashSettingsController::class, 'index'])
        ->name('admin.settings.payment.bayarcash');

    Route::post('/bayarcash', [BayarCashSettingsController::class, 'update'])
        ->name('admin.settings.payment.bayarcash.update');
});