<?php

namespace Modules\BayarCash\Providers;

use Illuminate\Support\ServiceProvider;

class BayarCashServiceProvider extends ServiceProvider
{
    /**
     * The module name.
     *
     * @var string
     */
    protected $moduleName = 'BayarCash';

    /**
     * Boot the application events.
     *
     * @return void
     */
    public function boot()
    {
        $this->registerTranslations();
        $this->registerConfig();
        $this->registerViews();
        $this->loadMigrationsFrom(module_path($this->moduleName, 'Database/Migrations'));
        $this->registerRoutes();
        $this->registerEventListeners();
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        // Check if BayarCash SDK is available
        $this->checkSdkAvailability();

        // Register the BayarCash FPX payment gateway
        $this->app->bind('bayarcash.gateway', function ($app) {
            $config = $this->getBayarCashConfig();

            return new \Modules\BayarCash\Services\BayarCashPaymentGateway(
                $config['portalKey'],
                $config['patKey'],
                $config['apiSecretKey'],
                $config['sandbox']
            );
        });

        // Register BayarCash DuitNow QR gateway
        $this->app->bind('bayarcash.duitnow.gateway', function ($app) {
            $config = $this->getBayarCashConfig();

            return new \Modules\BayarCash\Services\BayarCashDuitNowQrPaymentGateway(
                $config['portalKey'],
                $config['patKey'],
                $config['apiSecretKey'],
                $config['sandbox']
            );
        });

        // Register payment gateway singletons
        $this->app->singleton(\Modules\BayarCash\Services\BayarCashPaymentGateway::class, function ($app) {
            return $app->make('bayarcash.gateway');
        });

        $this->app->singleton(\Modules\BayarCash\Services\BayarCashDuitNowQrPaymentGateway::class, function ($app) {
            return $app->make('bayarcash.duitnow.gateway');
        });
    }

    /**
     * Check if a class exists in the application.
     *
     * @param string $class
     * @return bool
     */
    protected function classExists(string $class): bool
    {
        return class_exists($class);
    }

    /**
     * Register translations.
     *
     * @return void
     */
    protected function registerTranslations()
    {
        $langPath = resource_path('lang/modules/' . strtolower($this->moduleName));

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, $this->moduleName);
        } else {
            $this->loadTranslationsFrom(module_path($this->moduleName, 'resources/lang'), $this->moduleName);
        }
    }

    /**
     * Register config.
     *
     * @return void
     */
    protected function registerConfig()
    {
        $this->publishes([
            module_path($this->moduleName, 'Config/config.php') => config_path($this->moduleName . '.php'),
        ], 'config');
        $this->mergeConfigFrom(
            module_path($this->moduleName, 'Config/config.php'), $this->moduleName
        );
    }

    /**
     * Register views.
     *
     * @return void
     */
    protected function registerViews()
    {
        $viewPath = resource_path('views/modules/' . strtolower($this->moduleName));

        $sourcePath = module_path($this->moduleName, 'resources/views');

        $this->publishes([
            $sourcePath => $viewPath
        ], 'views');

        $this->loadViewsFrom(array_merge([$sourcePath], [
            $viewPath
        ]), $this->moduleName);
    }

    /**
     * Register routes.
     *
     * @return void
     */
    protected function registerRoutes()
    {
        $this->loadRoutesFrom(module_path($this->moduleName, 'Routes/web.php'));
        $this->loadRoutesFrom(module_path($this->moduleName, 'Routes/api.php'));
    }

    /**
     * Register event listeners.
     *
     * This method checks if event classes exist before registering listeners,
     * ensuring compatibility with different application versions (v1.5.2 and v1.6.0).
     *
     * @return void
     */
    protected function registerEventListeners()
    {
        // Register event listener for extending payment settings (exists in both v1.5.2 and v1.6.0)
        if ($this->classExists(\App\Events\PaymentSettingsExtending::class)) {
            $this->app['events']->listen(
                \App\Events\PaymentSettingsExtending::class,
                \Modules\BayarCash\Listeners\ExtendPaymentSettings::class
            );
        }

        // Register event listener for adding BayarCash to payment settings view
        // This event only exists in v1.5.2, not in v1.6.0
        if ($this->classExists(\App\Events\PaymentSettingsViewRendering::class)) {
            $this->app['events']->listen(
                \App\Events\PaymentSettingsViewRendering::class,
                \Modules\BayarCash\Listeners\AddBayarCashToPaymentSettings::class
            );
        }

        // Register event listener for payment gateway registration (exists in both v1.5.2 and v1.6.0)
        if ($this->classExists(\App\Events\PaymentGatewayRegistration::class)) {
            $this->app['events']->listen(
                \App\Events\PaymentGatewayRegistration::class,
                \Modules\BayarCash\Listeners\RegisterBayarCashPaymentGateway::class
            );
        }
    }

    /**
     * Check if BayarCash SDK is available.
     *
     * @return void
     * @throws \Exception
     */
    protected function checkSdkAvailability()
    {
        if (!class_exists('Webimpian\BayarcashSdk\Bayarcash')) {
            $message = 'BayarCash SDK not found. Please install it using: composer require webimpian/bayarcash-php-sdk:^2.0';

            // Log the error
            if (function_exists('error_log')) {
                error_log($message);
            }

            throw new \Exception($message);
        }
    }

    /**
     * Get BayarCash configuration array.
     *
     * This method is compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (get_setting helper).
     *
     * @return array
     */
    protected function getBayarCashConfig(): array
    {
        // Try to use PaymentSettings class first (v1.5.2 approach)
        if ($this->classExists(\App\Settings\PaymentSettings::class)) {
            try {
                $settings = app(\App\Settings\PaymentSettings::class);
                return [
                    'portalKey' => (string) ($settings->bayarcash_portal_key ?? ''),
                    'patKey' => (string) ($settings->bayarcash_pat_key ?? ''),
                    'apiSecretKey' => (string) ($settings->bayarcash_api_secret_key ?? ''),
                    'sandbox' => (bool) ($settings->bayarcash_sandbox ?? true),
                ];
            } catch (\Exception $e) {
                // Fall through to helper method
            }
        }

        // Fallback to get_setting helper (works in both v1.5.2 and v1.6.0)
        if (function_exists('get_setting')) {
            return [
                'portalKey' => (string) get_setting('payment.bayarcash_portal_key', ''),
                'patKey' => (string) get_setting('payment.bayarcash_pat_key', ''),
                'apiSecretKey' => (string) get_setting('payment.bayarcash_api_secret_key', ''),
                'sandbox' => (bool) get_setting('payment.bayarcash_sandbox', true),
            ];
        }

        // Return empty config if no settings mechanism available
        return [
            'portalKey' => '',
            'patKey' => '',
            'apiSecretKey' => '',
            'sandbox' => true,
        ];
    }

    /**
     * Get the services provided by the provider.
     *
     * @return array
     */
    public function provides()
    {
        return [];
    }
}
