<?php

namespace Modules\BayarCash\Listeners;

use App\Events\PaymentGatewayRegistration;
use Modules\BayarCash\Services\BayarCashPaymentGateway;
use Modules\BayarCash\Services\BayarCashDuitNowQrPaymentGateway;

class RegisterBayarCashPaymentGateway
{
    /**
     * Handle the event.
     *
     * Compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (get_setting helper).
     */
    public function handle(PaymentGatewayRegistration $event): void
    {
        $config = $this->getBayarCashConfig();

        // Check if BayarCash is enabled and configured
        $isConfigured = $config['enabled'] &&
            !empty($config['portalKey']) &&
            !empty($config['patKey']) &&
            !empty($config['apiSecretKey']);

        if ($isConfigured) {
            // Register BayarCash FPX gateway (renamed from original BayarCash)
            $event->billingManager->register('bayarcash_fpx', function () {
                return app(BayarCashPaymentGateway::class);
            });

            // Register BayarCash DuitNow QR gateway
            $event->billingManager->register('bayarcash_duitnow_qr', function () {
                return app(BayarCashDuitNowQrPaymentGateway::class);
            });
        }
    }

    /**
     * Get BayarCash configuration.
     *
     * Compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (get_setting helper).
     *
     * @return array
     */
    protected function getBayarCashConfig(): array
    {
        // Try to use PaymentSettings class first (v1.5.2 approach)
        if (class_exists(\App\Settings\PaymentSettings::class)) {
            try {
                $settings = app(\App\Settings\PaymentSettings::class);
                return [
                    'enabled' => (bool) ($settings->bayarcash_enabled ?? false),
                    'portalKey' => (string) ($settings->bayarcash_portal_key ?? ''),
                    'patKey' => (string) ($settings->bayarcash_pat_key ?? ''),
                    'apiSecretKey' => (string) ($settings->bayarcash_api_secret_key ?? ''),
                    'sandbox' => (bool) ($settings->bayarcash_sandbox ?? true),
                ];
            } catch (\Exception $e) {
                // Fall through to helper method
            }
        }

        // Fallback to get_setting helper (works in both v1.5.2 and v1.6.0)
        if (function_exists('get_setting')) {
            return [
                'enabled' => (bool) get_setting('payment.bayarcash_enabled', false),
                'portalKey' => (string) get_setting('payment.bayarcash_portal_key', ''),
                'patKey' => (string) get_setting('payment.bayarcash_pat_key', ''),
                'apiSecretKey' => (string) get_setting('payment.bayarcash_api_secret_key', ''),
                'sandbox' => (bool) get_setting('payment.bayarcash_sandbox', true),
            ];
        }

        // Return empty config if no settings mechanism available
        return [
            'enabled' => false,
            'portalKey' => '',
            'patKey' => '',
            'apiSecretKey' => '',
            'sandbox' => true,
        ];
    }
}