<?php

namespace Modules\BayarCash\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class BayarCashSettingsController extends Controller
{
    /**
     * Show BayarCash settings page
     *
     * Compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (get_setting helper).
     */
    public function index()
    {
        if (! checkPermission('admin.payment_settings.view')) {
            session()->flash('notification', [
                'type' => 'danger',
                'message' => t('access_denied_note'),
            ]);

            return redirect()->to(route('admin.dashboard'));
        }

        $config = $this->getBayarCashSettings();

        return view('BayarCash::admin.settings', [
            'settings' => null, // Kept for backward compatibility with view
            'bayarcash_enabled' => $config['enabled'],
            'bayarcash_portal_key' => $config['portal_key'],
            'bayarcash_pat_key' => $config['pat_key'],
            'bayarcash_api_secret_key' => $config['api_secret_key'],
            'bayarcash_sandbox' => $config['sandbox'],
        ]);
    }

    /**
     * Update BayarCash settings
     *
     * Compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (set_setting helper).
     */
    public function update(Request $request)
    {
        if (! checkPermission('admin.payment_settings.edit')) {
            session()->flash('notification', [
                'type' => 'danger',
                'message' => t('access_denied_note'),
            ]);

            return redirect()->to(route('admin.dashboard'));
        }

        $validator = Validator::make($request->all(), [
            'bayarcash_enabled' => 'required|boolean',
            'bayarcash_portal_key' => 'required_if:bayarcash_enabled,true|string|max:255',
            'bayarcash_pat_key' => 'required_if:bayarcash_enabled,true|string|max:1000',
            'bayarcash_api_secret_key' => 'required_if:bayarcash_enabled,true|string|max:255',
            'bayarcash_sandbox' => 'required|boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $this->saveBayarCashSettings($request);

        session()->flash('notification', [
            'type' => 'success',
            'message' => t('settings_saved_successfully'),
        ]);

        return redirect()->back();
    }

    /**
     * Get BayarCash settings.
     *
     * Compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (get_setting helper).
     *
     * @return array
     */
    protected function getBayarCashSettings(): array
    {
        // Try to use PaymentSettings class first (v1.5.2 approach)
        if (class_exists(\App\Settings\PaymentSettings::class)) {
            try {
                $settings = app(\App\Settings\PaymentSettings::class);
                return [
                    'enabled' => (bool) ($settings->bayarcash_enabled ?? false),
                    'portal_key' => (string) ($settings->bayarcash_portal_key ?? ''),
                    'pat_key' => (string) ($settings->bayarcash_pat_key ?? ''),
                    'api_secret_key' => (string) ($settings->bayarcash_api_secret_key ?? ''),
                    'sandbox' => (bool) ($settings->bayarcash_sandbox ?? true),
                ];
            } catch (\Exception $e) {
                // Fall through to helper method
            }
        }

        // Fallback to get_setting helper (works in both v1.5.2 and v1.6.0)
        if (function_exists('get_setting')) {
            return [
                'enabled' => (bool) get_setting('payment.bayarcash_enabled', false),
                'portal_key' => (string) get_setting('payment.bayarcash_portal_key', ''),
                'pat_key' => (string) get_setting('payment.bayarcash_pat_key', ''),
                'api_secret_key' => (string) get_setting('payment.bayarcash_api_secret_key', ''),
                'sandbox' => (bool) get_setting('payment.bayarcash_sandbox', true),
            ];
        }

        // Return defaults if no settings mechanism available
        return [
            'enabled' => false,
            'portal_key' => '',
            'pat_key' => '',
            'api_secret_key' => '',
            'sandbox' => true,
        ];
    }

    /**
     * Save BayarCash settings.
     *
     * Compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (set_setting helper).
     *
     * @param Request $request
     * @return void
     */
    protected function saveBayarCashSettings(Request $request): void
    {
        // Try to use PaymentSettings class first (v1.5.2 approach)
        if (class_exists(\App\Settings\PaymentSettings::class)) {
            try {
                $settings = app(\App\Settings\PaymentSettings::class);

                $settings->bayarcash_enabled = $request->boolean('bayarcash_enabled');
                $settings->bayarcash_portal_key = $request->get('bayarcash_portal_key', '');
                $settings->bayarcash_pat_key = $request->get('bayarcash_pat_key', '');
                $settings->bayarcash_api_secret_key = $request->get('bayarcash_api_secret_key', '');
                $settings->bayarcash_sandbox = $request->boolean('bayarcash_sandbox');

                $settings->save();
                return;
            } catch (\Exception $e) {
                // Fall through to helper method
            }
        }

        // Fallback to set_setting helper (works in both v1.5.2 and v1.6.0)
        if (function_exists('set_setting')) {
            set_setting('payment.bayarcash_enabled', $request->boolean('bayarcash_enabled'));
            set_setting('payment.bayarcash_portal_key', $request->get('bayarcash_portal_key', ''));
            set_setting('payment.bayarcash_pat_key', $request->get('bayarcash_pat_key', ''));
            set_setting('payment.bayarcash_api_secret_key', $request->get('bayarcash_api_secret_key', ''));
            set_setting('payment.bayarcash_sandbox', $request->boolean('bayarcash_sandbox'));
        }
    }
}