<?php

namespace Modules\BayarCash;

class BayarCash
{
    /**
     * Module information.
     */
    public static function info(): array
    {
        return [
            'name' => 'BayarCash Payment Gateway',
            'description' => 'Comprehensive BayarCash Payment Gateway Module - Supports FPX and DuitNow QR for Malaysian market',
            'version' => '3.1.0',
            'author' => 'Payment Module Developer',
            'compatibility' => 'v1.5.2, v1.6.0',
        ];
    }

    /**
     * Check if module is properly configured.
     *
     * Compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (get_setting helper).
     */
    public static function isConfigured(): bool
    {
        $config = self::getConfig();

        return $config['enabled'] &&
            !empty($config['portal_key']) &&
            !empty($config['pat_key']) &&
            !empty($config['api_secret_key']);
    }

    /**
     * Get BayarCash configuration.
     *
     * Compatible with both v1.5.2 (PaymentSettings class) and v1.6.0 (get_setting helper).
     *
     * @return array
     */
    public static function getConfig(): array
    {
        // Try to use PaymentSettings class first (v1.5.2 approach)
        if (class_exists(\App\Settings\PaymentSettings::class)) {
            try {
                $settings = app(\App\Settings\PaymentSettings::class);
                return [
                    'enabled' => (bool) ($settings->bayarcash_enabled ?? false),
                    'portal_key' => (string) ($settings->bayarcash_portal_key ?? ''),
                    'pat_key' => (string) ($settings->bayarcash_pat_key ?? ''),
                    'api_secret_key' => (string) ($settings->bayarcash_api_secret_key ?? ''),
                    'sandbox' => (bool) ($settings->bayarcash_sandbox ?? true),
                ];
            } catch (\Exception $e) {
                // Fall through to helper method
            }
        }

        // Fallback to get_setting helper (works in both v1.5.2 and v1.6.0)
        if (function_exists('get_setting')) {
            return [
                'enabled' => (bool) get_setting('payment.bayarcash_enabled', false),
                'portal_key' => (string) get_setting('payment.bayarcash_portal_key', ''),
                'pat_key' => (string) get_setting('payment.bayarcash_pat_key', ''),
                'api_secret_key' => (string) get_setting('payment.bayarcash_api_secret_key', ''),
                'sandbox' => (bool) get_setting('payment.bayarcash_sandbox', true),
            ];
        }

        // Return defaults if no settings mechanism available
        return [
            'enabled' => false,
            'portal_key' => '',
            'pat_key' => '',
            'api_secret_key' => '',
            'sandbox' => true,
        ];
    }
}
