<?php

namespace Modules\BayarCash\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Invoice\Invoice;
use App\Models\Transaction;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Modules\BayarCash\Services\BayarCashPaymentGateway;

/**
 * BayarCash Payment Controller
 *
 * Handles BayarCash payment processing including checkout, return handling,
 * and webhook processing for payment confirmations.
 *
 * @package Modules\BayarCash\Http\Controllers
 */
class BayarCashController extends Controller
{
    protected BayarCashPaymentGateway $gateway;

    public function __construct()
    {
        $this->gateway = app('bayarcash.gateway');
    }

    /**
     * Handle checkout process
     */
    public function checkout(Request $request, string $subdomain, Invoice $invoice): RedirectResponse
    {
        try {
            // Validate gateway availability
            if (!$this->gateway->isActive()) {
                return $this->redirectWithError($subdomain, $invoice, 'BayarCash payment gateway is not available.');
            }

            // Validate invoice status
            if ($invoice->isPaid()) {
                return $this->redirectWithInfo($subdomain, $invoice, 'This invoice has already been paid.');
            }

            // Validate payment amount
            $paymentData = $this->validatePaymentAmount($request, $invoice);
            if (!$paymentData['valid']) {
                return $this->redirectWithError($subdomain, $invoice, $paymentData['message']);
            }

            // Create and process payment
            $paymentResult = $this->processPayment($invoice, $paymentData);

            // Redirect to BayarCash payment page
            return redirect($paymentResult['payment_url']);

        } catch (Exception $e) {
            Log::error('BayarCash checkout error', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return $this->redirectWithError($subdomain, $invoice, 'Unable to process payment. Please try again.');
        }
    }

    /**
     * Handle return from BayarCash payment page
     */
    public function returnUrl(Request $request, string $subdomain, Invoice $invoice): RedirectResponse
    {
        $status = $request->get('status');
        $transactionId = $request->get('transaction_id');
        $orderId = $request->get('order_id');

        Log::info('BayarCash return URL accessed', [
            'invoice_id' => $invoice->id,
            'status' => $status,
            'transaction_id' => $transactionId,
            'order_id' => $orderId,
        ]);

        try {
            // Find the associated transaction
            $transaction = $this->findTransaction($invoice, $transactionId);

            if (!$transaction) {
                Log::warning('BayarCash return: Transaction not found', [
                    'invoice_id' => $invoice->id,
                    'transaction_id' => $transactionId,
                    'order_id' => $orderId,
                ]);

                return $this->redirectWithError($subdomain, $invoice, 'Transaction not found.');
            }

            // Update transaction with BayarCash transaction ID if needed
            $this->updateTransactionId($transaction, $transactionId);

            // Verify payment status
            $verificationResult = $this->gateway->verify($transaction);

            return $this->handleVerificationResult($subdomain, $invoice, $verificationResult);

        } catch (Exception $e) {
            Log::error('BayarCash return URL error', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
            ]);

            return $this->redirectWithError($subdomain, $invoice, 'Unable to verify payment status. Please contact support.');
        }
    }

    /**
     * Handle webhook from BayarCash
     */
    public function webhook(Request $request): JsonResponse
    {
        $startTime = microtime(true);

        try {
            // Validate webhook signature
            $validationResult = $this->validateWebhookSignature($request);
            if (!$validationResult['valid']) {
                return response()->json(['error' => $validationResult['message']], 400);
            }

            // Parse webhook data
            $webhookData = $validationResult['data'];
            $metadata = json_decode($webhookData['metadata'] ?? '{}', true);
            $tenantId = $metadata['tenant_id'] ?? null;

            if (!$tenantId) {
                Log::warning('BayarCash webhook: No tenant ID in metadata');
                return response()->json(['error' => 'No tenant ID found'], 400);
            }

            // Process webhook event
            $this->gateway->handleWebhook($webhookData);

            $processingTime = round((microtime(true) - $startTime) * 1000);

            return response()->json([
                'status' => 'success',
                'processing_time_ms' => $processingTime,
            ]);

        } catch (Exception $e) {
            Log::error('BayarCash webhook processing failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json(['error' => 'Internal server error'], 500);
        }
    }

    /**
     * Handle callback from BayarCash (alias for webhook)
     */
    public function callback(Request $request): JsonResponse
    {
        return $this->webhook($request);
    }

    /**
     * Validate payment amount and requirements
     */
    protected function validatePaymentAmount(Request $request, Invoice $invoice): array
    {
        // Validate and sanitize remaining_credit input
        $remainingCredit = $request->get('remaining_credit', 0);
        
        // Ensure remaining_credit is a valid non-negative number
        if (!is_numeric($remainingCredit) || $remainingCredit < 0) {
            return [
                'valid' => false,
                'message' => 'Invalid credit amount specified.'
            ];
        }
        
        $remainingCredit = (float) $remainingCredit;
        $invoiceTotal = $invoice->total();
        
        // Validate credit doesn't exceed invoice total
        if ($remainingCredit > $invoiceTotal) {
            return [
                'valid' => false,
                'message' => 'Credit amount cannot exceed invoice total.'
            ];
        }
        
        // Validate tenant has sufficient credit balance (if credit is being used)
        if ($remainingCredit > 0) {
            $tenantCredit = $invoice->tenant->credit_balance ?? 0;
            if ($remainingCredit > $tenantCredit) {
                return [
                    'valid' => false,
                    'message' => 'Insufficient credit balance.'
                ];
            }
        }
        
        $finalAmount = $invoiceTotal - $remainingCredit;
        $currency = $invoice->currency->code ?? 'MYR';
        $minimumAmount = $this->gateway->getMinimumChargeAmount($currency);

        if ($finalAmount < $minimumAmount) {
            return [
                'valid' => false,
                'message' => "Minimum payment amount is {$minimumAmount} {$currency}."
            ];
        }

        return [
            'valid' => true,
            'remaining_credit' => $remainingCredit,
            'final_amount' => round($finalAmount, 2),
            'currency' => $currency
        ];
    }

    /**
     * Process payment with BayarCash
     */
    protected function processPayment(Invoice $invoice, array $paymentData): array
    {
        // Create pending transaction
        $transaction = $invoice->createPendingTransaction($this->gateway, $invoice->tenant_id);

        // Initialize payment with BayarCash
        $paymentResult = $this->gateway->initializePayment($invoice, [
            'transaction_id' => $transaction->id,
            'remaining_credit' => $paymentData['remaining_credit'],
            'final_amount' => $paymentData['final_amount'],
        ]);

        // Update transaction with BayarCash details
        $transaction->update([
            'idempotency_key' => $paymentResult['transaction_id'],
            'metadata' => [
                'order_id' => $paymentResult['order_id'],
                'bayarcash_payment_intent_id' => $paymentResult['transaction_id'],
                'bayarcash_order_id' => $paymentResult['order_id'],
                'payment_url' => $paymentResult['payment_url'],
                'remaining_credit' => $paymentData['remaining_credit'],
                'final_amount' => $paymentData['final_amount'],
            ],
        ]);

        return $paymentResult;
    }

    /**
     * Find transaction for the invoice
     */
    protected function findTransaction(Invoice $invoice, ?string $transactionId): ?Transaction
    {
        if ($transactionId) {
            // Try to find by various ID fields
            $transaction = Transaction::where('invoice_id', $invoice->id)
                ->where(function($query) use ($transactionId) {
                    $query->where('idempotency_key', $transactionId)
                          ->orWhereJsonContains('metadata->bayarcash_transaction_id', $transactionId)
                          ->orWhereJsonContains('metadata->order_id', $transactionId);
                })
                ->first();

            if ($transaction) {
                return $transaction;
            }
        }

        // Fallback to most recent pending transaction
        return Transaction::where('invoice_id', $invoice->id)
            ->where('status', 'pending')
            ->latest()
            ->first();
    }

    /**
     * Update transaction with BayarCash transaction ID
     */
    protected function updateTransactionId(Transaction $transaction, ?string $transactionId): void
    {
        if ($transactionId && !isset($transaction->metadata['bayarcash_transaction_id'])) {
            $metadata = $transaction->metadata ?? [];
            $metadata['bayarcash_transaction_id'] = $transactionId;
            $transaction->update(['metadata' => $metadata]);

            Log::info('BayarCash return: Updated transaction with transaction ID', [
                'transaction_id' => $transaction->id,
                'bayarcash_transaction_id' => $transactionId,
            ]);
        }
    }

    /**
     * Handle verification result
     */
    protected function handleVerificationResult(string $subdomain, Invoice $invoice, $verificationResult): RedirectResponse
    {
        if ($verificationResult->isDone()) {
            return $this->redirectWithSuccess($subdomain, $invoice, 'Payment completed successfully!');
        }

        if ($verificationResult->isPending()) {
            return $this->redirectWithInfo($subdomain, $invoice, 'Payment is being processed. Please wait for confirmation.');
        }

        return $this->redirectWithError($subdomain, $invoice, 'Payment failed: ' . $verificationResult->getMessage());
    }

    /**
     * Validate webhook signature and parse data
     */
    protected function validateWebhookSignature(Request $request): array
    {
        $payload = $request->getContent();
        $signature = $request->header('x-signature') ?? $request->header('signature');

        if (!$signature) {
            Log::warning('BayarCash webhook: Missing signature');
            return [
                'valid' => false,
                'message' => 'Missing signature'
            ];
        }

        if (!$this->gateway->validateWebhookSignature($payload, $signature)) {
            Log::warning('BayarCash webhook: Invalid signature');
            return [
                'valid' => false,
                'message' => 'Invalid signature'
            ];
        }

        $webhookData = json_decode($payload, true);

        if (!$webhookData) {
            Log::warning('BayarCash webhook: Invalid JSON payload');
            return [
                'valid' => false,
                'message' => 'Invalid JSON'
            ];
        }

        return [
            'valid' => true,
            'data' => $webhookData
        ];
    }

    /**
     * Redirect with error message
     */
    protected function redirectWithError(string $subdomain, Invoice $invoice, string $message): RedirectResponse
    {
        return redirect()->route('tenant.invoices.show', [$subdomain, $invoice])
            ->with('error', $message);
    }

    /**
     * Redirect with success message
     */
    protected function redirectWithSuccess(string $subdomain, Invoice $invoice, string $message): RedirectResponse
    {
        return redirect()->route('tenant.invoices.show', [$subdomain, $invoice])
            ->with('success', $message);
    }

    /**
     * Redirect with info message
     */
    protected function redirectWithInfo(string $subdomain, Invoice $invoice, string $message): RedirectResponse
    {
        return redirect()->route('tenant.invoices.show', [$subdomain, $invoice])
            ->with('info', $message);
    }
}