<?php

namespace Modules\BayarCash\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Settings\PaymentSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class BayarCashSettingsController extends Controller
{
    /**
     * Show BayarCash settings page
     */
    public function index()
    {
        if (! checkPermission('admin.payment_settings.view')) {
            session()->flash('notification', [
                'type' => 'danger',
                'message' => t('access_denied_note'),
            ]);

            return redirect()->to(route('admin.dashboard'));
        }

        $settings = app(PaymentSettings::class);

        return view('BayarCash::admin.settings', [
            'settings' => $settings,
            'bayarcash_enabled' => $settings->bayarcash_enabled ?? false,
            'bayarcash_portal_key' => $settings->bayarcash_portal_key ?? '',
            'bayarcash_pat_key' => $settings->bayarcash_pat_key ?? '',
            'bayarcash_api_secret_key' => $settings->bayarcash_api_secret_key ?? '',
            'bayarcash_sandbox' => $settings->bayarcash_sandbox ?? true,
        ]);
    }

    /**
     * Update BayarCash settings
     */
    public function update(Request $request)
    {
        if (! checkPermission('admin.payment_settings.edit')) {
            session()->flash('notification', [
                'type' => 'danger',
                'message' => t('access_denied_note'),
            ]);

            return redirect()->to(route('admin.dashboard'));
        }

        $validator = Validator::make($request->all(), [
            'bayarcash_enabled' => 'required|boolean',
            'bayarcash_portal_key' => 'required_if:bayarcash_enabled,true|string|max:255',
            'bayarcash_pat_key' => 'required_if:bayarcash_enabled,true|string|max:1000',
            'bayarcash_api_secret_key' => 'required_if:bayarcash_enabled,true|string|max:255',
            'bayarcash_sandbox' => 'required|boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $settings = app(PaymentSettings::class);

        // Update settings using dynamic properties
        $settings->bayarcash_enabled = $request->boolean('bayarcash_enabled');
        $settings->bayarcash_portal_key = $request->get('bayarcash_portal_key', '');
        $settings->bayarcash_pat_key = $request->get('bayarcash_pat_key', '');
        $settings->bayarcash_api_secret_key = $request->get('bayarcash_api_secret_key', '');
        $settings->bayarcash_sandbox = $request->boolean('bayarcash_sandbox');

        $settings->save();

        session()->flash('notification', [
            'type' => 'success',
            'message' => t('settings_saved_successfully'),
        ]);

        return redirect()->back();
    }
}