<?php

namespace Modules\BayarCash\Services;

use App\Models\Invoice\Invoice;
use Illuminate\Support\Facades\Log;
use Webimpian\BayarcashSdk\Bayarcash;

/**
 * BayarCash DuitNow QR Payment Gateway
 *
 * Specialized payment gateway for DuitNow QR payments through BayarCash.
 * Provides QR code scanning functionality for mobile banking and e-wallet payments.
 *
 * @package Modules\BayarCash\Services
 * @version 2.0.0
 */
class BayarCashDuitNowQrPaymentGateway extends BayarCashPaymentGateway
{
    /**
     * Gateway type identifier
     */
    public const TYPE = 'bayarcash_duitnow_qr';

    /**
     * Payment channel constant
     */
    protected const PAYMENT_CHANNEL = Bayarcash::DUITNOW_QR;

    /**
     * Get the gateway name
     */
    public function getName(): string
    {
        return 'BayarCash DuitNow QR';
    }

    /**
     * Get the gateway type
     */
    public function getType(): string
    {
        return self::TYPE;
    }

    /**
     * Get the gateway description
     */
    public function getDescription(): string
    {
        return 'Pay securely using DuitNow QR - Scan QR code with your mobile banking app or e-wallet.';
    }

    /**
     * Get the gateway short description
     */
    public function getShortDescription(): string
    {
        return 'DuitNow QR';
    }

    /**
     * Get the settings URL
     */
    public function getSettingsUrl(): string
    {
        return route('admin.settings.payment.bayarcash');
    }

    /**
     * Get the checkout URL for an invoice
     */
    public function getCheckoutUrl(Invoice $invoice): string
    {
        return tenant_route('tenant.payment.bayarcash.duitnow.checkout', ['invoice' => $invoice->id]);
    }

    /**
     * Build payment data array with DuitNow QR channel
     */
    protected function buildPaymentData(Invoice $invoice, string $orderId, float $finalAmount, array $metadata): array
    {
        $paymentData = parent::buildPaymentData($invoice, $orderId, $finalAmount, $metadata);

        // Override payment channel for DuitNow QR
        $paymentData['payment_channel'] = self::PAYMENT_CHANNEL;

        // Add DuitNow QR specific metadata
        $metadataArray = json_decode($paymentData['metadata'], true);
        $metadataArray['payment_method'] = 'duitnow_qr';
        $metadataArray['payment_channel'] = 'DuitNow QR';
        $paymentData['metadata'] = json_encode($metadataArray);

        return $paymentData;
    }

    /**
     * Generate unique order ID with DuitNow QR prefix
     */
    protected function generateOrderId(Invoice $invoice): string
    {
        return 'DNQ_INV_' . $invoice->id . '_' . time() . '_' . substr(md5(uniqid()), 0, 8);
    }

    /**
     * Log payment initialization with DuitNow QR context
     */
    protected function logPaymentInitialization(Invoice $invoice, float $finalAmount, float $remainingCredit): void
    {
        parent::logPaymentInitialization($invoice, $finalAmount, $remainingCredit);

        Log::info('BayarCash DuitNow QR payment channel selected', [
            'invoice_id' => $invoice->id,
            'payment_channel' => 'DuitNow QR',
            'channel_code' => self::PAYMENT_CHANNEL,
        ]);
    }
}