import { authMiddleware } from '../middleware/auth.js';
import { subscribe, CHANNELS } from '../services/redis.js';
import logger from '../utils/logger.js';
import messageHandler from './handlers/message.js';
import typingHandler from './handlers/typing.js';
import presenceHandler from './handlers/presence.js';
import conversationHandler from './handlers/conversation.js';

/**
 * Setup Socket.IO handlers and middleware
 */
export default function setupSocketHandlers(io) {
  // Authentication middleware
  io.use(authMiddleware);

  // Connection handler
  io.on('connection', (socket) => {
    logger.info('Client connected', {
      socketId: socket.id,
      userId: socket.userId,
      tenantId: socket.tenantId,
    });

    // Join user's personal room (for direct notifications)
    socket.join(`user:${socket.userId}`);

    // Join tenant room (for tenant-wide broadcasts)
    if (socket.tenantId) {
      socket.join(`tenant:${socket.tenantId}`);
    }

    // Initialize handlers
    messageHandler(io, socket);
    typingHandler(io, socket);
    presenceHandler(io, socket);
    conversationHandler(io, socket);

    // Handle disconnect
    socket.on('disconnect', (reason) => {
      logger.info('Client disconnected', {
        socketId: socket.id,
        userId: socket.userId,
        reason,
      });
    });

    // Handle errors
    socket.on('error', (error) => {
      logger.error('Socket error', {
        socketId: socket.id,
        userId: socket.userId,
        error: error.message,
      });
    });
  });

  // Subscribe to Laravel Redis channels
  setupLaravelSubscriptions(io);

  logger.info('Socket.IO handlers setup complete');
}

/**
 * Subscribe to Redis channels for Laravel events
 */
async function setupLaravelSubscriptions(io) {
  try {
    // Handle incoming messages from WhatsApp (via Laravel webhook)
    await subscribe(CHANNELS.LARAVEL_MESSAGE_INCOMING, (data) => {
      logger.debug('Received message from Laravel', { conversationId: data.conversationId });

      // Emit to conversation room
      io.to(`conversation:${data.conversationId}`).emit('message:new', {
        id: data.id,
        conversationId: data.conversationId,
        from: data.from,
        to: data.to,
        message: data.message,
        messageType: data.messageType || 'text',
        media: data.media || null,
        timestamp: data.timestamp,
        status: 'delivered',
        metadata: data.metadata || {},
      });

      // Also emit to user rooms for notification
      if (data.userIds && Array.isArray(data.userIds)) {
        data.userIds.forEach(userId => {
          io.to(`user:${userId}`).emit('notification:message', {
            conversationId: data.conversationId,
            preview: data.message?.substring(0, 100),
            from: data.from,
            timestamp: data.timestamp,
          });
        });
      }
    });

    // Handle message status updates (sent, delivered, read)
    await subscribe(CHANNELS.LARAVEL_MESSAGE_STATUS, (data) => {
      logger.debug('Message status update', { messageId: data.messageId, status: data.status });

      io.to(`conversation:${data.conversationId}`).emit('message:status', {
        messageId: data.messageId,
        tempId: data.tempId,
        status: data.status,
        timestamp: data.timestamp,
      });
    });

    // Handle user status changes (online/offline from Laravel)
    await subscribe(CHANNELS.LARAVEL_USER_STATUS, (data) => {
      io.to(`tenant:${data.tenantId}`).emit('user:status', {
        userId: data.userId,
        status: data.status,
        lastSeen: data.lastSeen,
      });
    });

    // Handle general broadcasts from Laravel
    await subscribe(CHANNELS.LARAVEL_BROADCAST, (data) => {
      if (data.userIds && Array.isArray(data.userIds)) {
        // Broadcast to specific users
        data.userIds.forEach(userId => {
          io.to(`user:${userId}`).emit(data.event, data.data);
        });
      } else if (data.tenantId) {
        // Broadcast to entire tenant
        io.to(`tenant:${data.tenantId}`).emit(data.event, data.data);
      }
    });

    logger.info('Subscribed to Laravel Redis channels');
  } catch (error) {
    logger.warn('Redis pub/sub not available - running without Laravel real-time sync', { error: error.message });
    logger.info('Server will continue - chat works but without real-time Laravel events');
  }
}
