import logger from '../../utils/logger.js';

// Track typing timeouts to auto-stop typing
const typingTimeouts = new Map();
const TYPING_TIMEOUT = 5000; // 5 seconds

/**
 * Typing indicator event handlers
 */
export default function typingHandler(io, socket) {
  /**
   * Handle typing start
   */
  socket.on('typing:start', (data) => {
    const { conversationId } = data;

    if (!conversationId) return;

    // Clear existing timeout for this user/conversation
    const timeoutKey = `${socket.userId}:${conversationId}`;
    const existingTimeout = typingTimeouts.get(timeoutKey);
    if (existingTimeout) {
      clearTimeout(existingTimeout);
    }

    // Broadcast typing status to conversation (except sender)
    socket.to(`conversation:${conversationId}`).emit('typing:user', {
      userId: socket.userId,
      userName: socket.user?.name || 'User',
      userAvatar: socket.user?.avatar || null,
      conversationId,
      timestamp: Date.now(),
    });

    // Auto-stop typing after timeout
    const timeout = setTimeout(() => {
      socket.to(`conversation:${conversationId}`).emit('typing:stop', {
        userId: socket.userId,
        conversationId,
      });
      typingTimeouts.delete(timeoutKey);
    }, TYPING_TIMEOUT);

    typingTimeouts.set(timeoutKey, timeout);

    logger.debug('User started typing', {
      userId: socket.userId,
      conversationId,
    });
  });

  /**
   * Handle typing stop
   */
  socket.on('typing:stop', (data) => {
    const { conversationId } = data;

    if (!conversationId) return;

    // Clear timeout
    const timeoutKey = `${socket.userId}:${conversationId}`;
    const existingTimeout = typingTimeouts.get(timeoutKey);
    if (existingTimeout) {
      clearTimeout(existingTimeout);
      typingTimeouts.delete(timeoutKey);
    }

    // Broadcast stop typing to conversation
    socket.to(`conversation:${conversationId}`).emit('typing:stop', {
      userId: socket.userId,
      conversationId,
    });

    logger.debug('User stopped typing', {
      userId: socket.userId,
      conversationId,
    });
  });

  /**
   * Cleanup typing status on disconnect
   */
  socket.on('disconnect', () => {
    // Find and clear all typing timeouts for this user
    for (const [key, timeout] of typingTimeouts.entries()) {
      if (key.startsWith(`${socket.userId}:`)) {
        clearTimeout(timeout);
        typingTimeouts.delete(key);

        // Extract conversationId and notify
        const conversationId = key.split(':')[1];
        socket.to(`conversation:${conversationId}`).emit('typing:stop', {
          userId: socket.userId,
          conversationId,
        });
      }
    }
  });
}
