import { publish, CHANNELS } from '../../services/redis.js';
import { checkConversationAccess, sendMessage } from '../../services/laravel.js';
import logger from '../../utils/logger.js';

/**
 * Message event handlers
 */
export default function messageHandler(io, socket) {
  /**
   * Handle sending a new message
   */
  socket.on('message:send', async (data) => {
    const { conversationId, message, messageType = 'text', media = null, tempId } = data;

    try {
      // Validate input
      if (!conversationId || !message?.trim()) {
        return socket.emit('message:error', {
          tempId,
          error: 'Invalid message data',
        });
      }

      // Check message length
      if (message.length > 4096) {
        return socket.emit('message:error', {
          tempId,
          error: 'Message too long (max 4096 characters)',
        });
      }

      // Verify user has access to this conversation
      const hasAccess = await checkConversationAccess(socket.token, conversationId);

      if (!hasAccess) {
        logger.warn('Unauthorized message attempt', {
          userId: socket.userId,
          conversationId,
        });
        return socket.emit('message:error', {
          tempId,
          error: 'Access denied to this conversation',
        });
      }

      // Send optimistic acknowledgment to sender
      socket.emit('message:sending', {
        tempId,
        status: 'sending',
        timestamp: Date.now(),
      });

      // Publish to Redis queue for Laravel to process and send to WABA
      await publish(CHANNELS.NODEJS_MESSAGE_OUTGOING, {
        userId: socket.userId,
        tenantId: socket.tenantId,
        conversationId,
        message: message.trim(),
        messageType,
        media,
        tempId,
        timestamp: Date.now(),
      });

      // Acknowledge message queued
      socket.emit('message:queued', {
        tempId,
        status: 'queued',
        timestamp: Date.now(),
      });

      logger.debug('Message queued for sending', {
        userId: socket.userId,
        conversationId,
        tempId,
      });

    } catch (error) {
      logger.error('Error sending message:', error);
      socket.emit('message:error', {
        tempId,
        error: 'Failed to send message',
      });
    }
  });

  /**
   * Handle marking messages as read
   */
  socket.on('message:read', async (data) => {
    const { conversationId, messageIds } = data;

    try {
      if (!conversationId) return;

      // Verify access
      const hasAccess = await checkConversationAccess(socket.token, conversationId);
      if (!hasAccess) return;

      // Publish read receipt to Laravel
      await publish(CHANNELS.NODEJS_MESSAGE_READ, {
        userId: socket.userId,
        tenantId: socket.tenantId,
        conversationId,
        messageIds: messageIds || [],
        timestamp: Date.now(),
      });

      // Notify other users in the conversation
      socket.to(`conversation:${conversationId}`).emit('message:read', {
        userId: socket.userId,
        conversationId,
        messageIds,
        timestamp: Date.now(),
      });

    } catch (error) {
      logger.error('Error marking messages as read:', error);
    }
  });

  /**
   * Handle message deletion request
   */
  socket.on('message:delete', async (data) => {
    const { conversationId, messageId } = data;

    try {
      // Verify access
      const hasAccess = await checkConversationAccess(socket.token, conversationId);
      if (!hasAccess) {
        return socket.emit('message:error', {
          messageId,
          error: 'Access denied',
        });
      }

      // Publish deletion request to Laravel
      await publish('nodejs:message:delete', {
        userId: socket.userId,
        tenantId: socket.tenantId,
        conversationId,
        messageId,
        timestamp: Date.now(),
      });

    } catch (error) {
      logger.error('Error deleting message:', error);
      socket.emit('message:error', {
        messageId,
        error: 'Failed to delete message',
      });
    }
  });
}
