import Redis from 'ioredis';
import config from '../config/index.js';
import logger from '../utils/logger.js';

let redis = null;
let redisSub = null;
let redisPub = null;

// Event handlers registry
const messageHandlers = new Map();

/**
 * Initialize Redis connections
 */
export async function initRedis() {
  const redisConfig = {
    host: config.redis.host,
    port: config.redis.port,
    username: config.redis.username || undefined,
    password: config.redis.password || undefined,
    retryStrategy: (times) => {
      const delay = Math.min(times * 50, 2000);
      logger.warn(`Redis connection retry #${times}, waiting ${delay}ms`);
      return delay;
    },
    maxRetriesPerRequest: 3,
  };

  // Main Redis client
  redis = new Redis(redisConfig);

  // Subscriber client (dedicated connection for pub/sub)
  redisSub = new Redis(redisConfig);

  // Publisher client
  redisPub = new Redis(redisConfig);

  // Event handlers
  redis.on('connect', () => logger.debug('Redis main client connected'));
  redis.on('error', (err) => logger.error('Redis main client error:', err));

  redisSub.on('connect', () => logger.debug('Redis subscriber connected'));
  redisSub.on('error', (err) => logger.error('Redis subscriber error:', err));

  redisPub.on('connect', () => logger.debug('Redis publisher connected'));
  redisPub.on('error', (err) => logger.error('Redis publisher error:', err));

  // Handle incoming messages from Laravel
  redisSub.on('message', (channel, message) => {
    try {
      const data = JSON.parse(message);
      const handlers = messageHandlers.get(channel);

      if (handlers) {
        handlers.forEach(handler => handler(data));
      }
    } catch (error) {
      logger.error(`Error processing Redis message on ${channel}:`, error);
    }
  });

  // Wait for connections
  await Promise.all([
    new Promise((resolve) => redis.once('ready', resolve)),
    new Promise((resolve) => redisSub.once('ready', resolve)),
    new Promise((resolve) => redisPub.once('ready', resolve)),
  ]);

  return redis;
}

/**
 * Get Redis client
 */
export function getRedis() {
  return redis;
}

/**
 * Subscribe to a Redis channel
 */
export async function subscribe(channel, handler) {
  if (!messageHandlers.has(channel)) {
    messageHandlers.set(channel, new Set());
    await redisSub.subscribe(channel);
    logger.debug(`Subscribed to Redis channel: ${channel}`);
  }

  messageHandlers.get(channel).add(handler);
}

/**
 * Unsubscribe from a Redis channel
 */
export async function unsubscribe(channel, handler) {
  const handlers = messageHandlers.get(channel);

  if (handlers) {
    handlers.delete(handler);

    if (handlers.size === 0) {
      messageHandlers.delete(channel);
      await redisSub.unsubscribe(channel);
      logger.debug(`Unsubscribed from Redis channel: ${channel}`);
    }
  }
}

/**
 * Publish message to Redis channel
 */
export async function publish(channel, data) {
  const message = typeof data === 'string' ? data : JSON.stringify(data);
  await redisPub.publish(channel, message);
  logger.debug(`Published to Redis channel: ${channel}`);
}

/**
 * Store data in Redis with optional TTL
 */
export async function set(key, value, ttlSeconds = null) {
  const data = typeof value === 'string' ? value : JSON.stringify(value);

  if (ttlSeconds) {
    await redis.setex(key, ttlSeconds, data);
  } else {
    await redis.set(key, data);
  }
}

/**
 * Get data from Redis
 */
export async function get(key) {
  const data = await redis.get(key);

  if (!data) return null;

  try {
    return JSON.parse(data);
  } catch {
    return data;
  }
}

/**
 * Delete key from Redis
 */
export async function del(key) {
  await redis.del(key);
}

/**
 * Add to set
 */
export async function sadd(key, ...members) {
  await redis.sadd(key, ...members);
}

/**
 * Remove from set
 */
export async function srem(key, ...members) {
  await redis.srem(key, ...members);
}

/**
 * Get all set members
 */
export async function smembers(key) {
  return redis.smembers(key);
}

/**
 * Check if member exists in set
 */
export async function sismember(key, member) {
  return redis.sismember(key, member);
}

/**
 * Hash set
 */
export async function hset(key, field, value) {
  const data = typeof value === 'string' ? value : JSON.stringify(value);
  await redis.hset(key, field, data);
}

/**
 * Hash get
 */
export async function hget(key, field) {
  const data = await redis.hget(key, field);

  if (!data) return null;

  try {
    return JSON.parse(data);
  } catch {
    return data;
  }
}

/**
 * Hash get all
 */
export async function hgetall(key) {
  return redis.hgetall(key);
}

/**
 * Push to list (queue)
 */
export async function lpush(key, ...values) {
  const data = values.map(v => typeof v === 'string' ? v : JSON.stringify(v));
  await redis.lpush(key, ...data);
}

/**
 * Pop from list (queue)
 */
export async function rpop(key) {
  const data = await redis.rpop(key);

  if (!data) return null;

  try {
    return JSON.parse(data);
  } catch {
    return data;
  }
}

// Export Redis channels used for Laravel communication
export const CHANNELS = {
  // Laravel -> Node.js (incoming messages from WhatsApp)
  LARAVEL_MESSAGE_INCOMING: 'laravel:message:incoming',
  LARAVEL_MESSAGE_STATUS: 'laravel:message:status',
  LARAVEL_USER_STATUS: 'laravel:user:status',
  LARAVEL_BROADCAST: 'laravel:broadcast',

  // Node.js -> Laravel (outgoing messages to WhatsApp)
  NODEJS_MESSAGE_OUTGOING: 'nodejs:message:outgoing',
  NODEJS_MESSAGE_READ: 'nodejs:message:read',
};
