import { sadd, srem, smembers, hset, hgetall } from '../../services/redis.js';
import logger from '../../utils/logger.js';

const ONLINE_USERS_KEY = 'users:online';
const USER_STATUS_PREFIX = 'user:status:';

/**
 * Presence (online/offline) event handlers
 */
export default function presenceHandler(io, socket) {
  /**
   * Handle user coming online
   */
  socket.on('user:online', async () => {
    try {
      // Add to online users set
      await sadd(ONLINE_USERS_KEY, socket.userId.toString());

      // Store user status info
      await hset(`${USER_STATUS_PREFIX}${socket.userId}`, 'data', {
        status: 'online',
        lastSeen: Date.now(),
        socketId: socket.id,
        userName: socket.user?.name,
      });

      // Broadcast to tenant
      if (socket.tenantId) {
        io.to(`tenant:${socket.tenantId}`).emit('user:status', {
          userId: socket.userId,
          status: 'online',
          userName: socket.user?.name,
        });
      }

      logger.debug('User online', { userId: socket.userId });

    } catch (error) {
      logger.error('Error setting user online:', error);
    }
  });

  /**
   * Handle user going away (idle)
   */
  socket.on('user:away', async () => {
    try {
      await hset(`${USER_STATUS_PREFIX}${socket.userId}`, 'data', {
        status: 'away',
        lastSeen: Date.now(),
        socketId: socket.id,
        userName: socket.user?.name,
      });

      if (socket.tenantId) {
        io.to(`tenant:${socket.tenantId}`).emit('user:status', {
          userId: socket.userId,
          status: 'away',
        });
      }

      logger.debug('User away', { userId: socket.userId });

    } catch (error) {
      logger.error('Error setting user away:', error);
    }
  });

  /**
   * Get list of online users
   */
  socket.on('users:getOnline', async (callback) => {
    try {
      const onlineUserIds = await smembers(ONLINE_USERS_KEY);

      // Filter to only users in the same tenant
      // (In production, you'd want to store tenant info with each online user)
      const onlineUsers = onlineUserIds.map(id => parseInt(id, 10));

      if (typeof callback === 'function') {
        callback({ success: true, users: onlineUsers });
      } else {
        socket.emit('users:online', onlineUsers);
      }

    } catch (error) {
      logger.error('Error getting online users:', error);
      if (typeof callback === 'function') {
        callback({ success: false, error: 'Failed to get online users' });
      }
    }
  });

  /**
   * Check if specific user is online
   */
  socket.on('user:checkOnline', async (data, callback) => {
    try {
      const { userId } = data;
      const statusData = await hgetall(`${USER_STATUS_PREFIX}${userId}`);

      let status = 'offline';
      let lastSeen = null;

      if (statusData?.data) {
        const parsed = typeof statusData.data === 'string'
          ? JSON.parse(statusData.data)
          : statusData.data;
        status = parsed.status || 'offline';
        lastSeen = parsed.lastSeen;
      }

      if (typeof callback === 'function') {
        callback({ userId, status, lastSeen });
      }

    } catch (error) {
      logger.error('Error checking user status:', error);
      if (typeof callback === 'function') {
        callback({ userId: data.userId, status: 'unknown' });
      }
    }
  });

  /**
   * Handle disconnect - mark user offline
   */
  socket.on('disconnect', async () => {
    try {
      // Check if user has other active connections
      const userSockets = await io.in(`user:${socket.userId}`).fetchSockets();

      // If no other connections, mark as offline
      if (userSockets.length === 0) {
        await srem(ONLINE_USERS_KEY, socket.userId.toString());

        await hset(`${USER_STATUS_PREFIX}${socket.userId}`, 'data', {
          status: 'offline',
          lastSeen: Date.now(),
          socketId: null,
          userName: socket.user?.name,
        });

        // Broadcast offline status
        if (socket.tenantId) {
          io.to(`tenant:${socket.tenantId}`).emit('user:status', {
            userId: socket.userId,
            status: 'offline',
            lastSeen: Date.now(),
          });
        }

        logger.debug('User offline', { userId: socket.userId });
      }

    } catch (error) {
      logger.error('Error handling disconnect presence:', error);
    }
  });
}
