import { checkConversationAccess } from '../../services/laravel.js';
import logger from '../../utils/logger.js';

/**
 * Conversation room management handlers
 */
export default function conversationHandler(io, socket) {
  // Track joined conversations for this socket
  const joinedConversations = new Set();

  /**
   * Join a conversation room
   */
  socket.on('conversation:join', async (data, callback) => {
    const { conversationId } = data;

    try {
      if (!conversationId) {
        return sendCallback(callback, { success: false, error: 'Conversation ID required' });
      }

      // Verify user has access to this conversation
      const hasAccess = await checkConversationAccess(socket.token, conversationId);

      if (!hasAccess) {
        logger.warn('Unauthorized conversation join attempt', {
          userId: socket.userId,
          conversationId,
        });
        return sendCallback(callback, { success: false, error: 'Access denied' });
      }

      // Join the room
      const roomName = `conversation:${conversationId}`;
      socket.join(roomName);
      joinedConversations.add(conversationId);

      logger.debug('User joined conversation', {
        userId: socket.userId,
        conversationId,
      });

      // Notify others in the conversation
      socket.to(roomName).emit('conversation:userJoined', {
        userId: socket.userId,
        userName: socket.user?.name,
        conversationId,
        timestamp: Date.now(),
      });

      sendCallback(callback, { success: true, conversationId });

    } catch (error) {
      logger.error('Error joining conversation:', error);
      sendCallback(callback, { success: false, error: 'Failed to join conversation' });
    }
  });

  /**
   * Leave a conversation room
   */
  socket.on('conversation:leave', (data, callback) => {
    const { conversationId } = data;

    if (!conversationId) {
      return sendCallback(callback, { success: false, error: 'Conversation ID required' });
    }

    const roomName = `conversation:${conversationId}`;
    socket.leave(roomName);
    joinedConversations.delete(conversationId);

    logger.debug('User left conversation', {
      userId: socket.userId,
      conversationId,
    });

    // Notify others
    socket.to(roomName).emit('conversation:userLeft', {
      userId: socket.userId,
      conversationId,
      timestamp: Date.now(),
    });

    sendCallback(callback, { success: true });
  });

  /**
   * Get list of users in a conversation
   */
  socket.on('conversation:getUsers', async (data, callback) => {
    const { conversationId } = data;

    try {
      if (!conversationId) {
        return sendCallback(callback, { success: false, error: 'Conversation ID required' });
      }

      const roomName = `conversation:${conversationId}`;
      const sockets = await io.in(roomName).fetchSockets();

      const users = sockets.map(s => ({
        id: s.userId,
        name: s.user?.name,
        avatar: s.user?.avatar,
      }));

      // Remove duplicates (same user, multiple tabs)
      const uniqueUsers = Array.from(
        new Map(users.map(u => [u.id, u])).values()
      );

      sendCallback(callback, { success: true, users: uniqueUsers });

    } catch (error) {
      logger.error('Error getting conversation users:', error);
      sendCallback(callback, { success: false, error: 'Failed to get users' });
    }
  });

  /**
   * Handle disconnect - leave all conversations
   */
  socket.on('disconnect', () => {
    for (const conversationId of joinedConversations) {
      socket.to(`conversation:${conversationId}`).emit('conversation:userLeft', {
        userId: socket.userId,
        conversationId,
        timestamp: Date.now(),
      });
    }
    joinedConversations.clear();
  });
}

/**
 * Helper to safely send callback response
 */
function sendCallback(callback, data) {
  if (typeof callback === 'function') {
    callback(data);
  }
}
