import axios from 'axios';
import config from '../config/index.js';
import logger from '../utils/logger.js';

// Create axios instance for Laravel API
const laravelApi = axios.create({
  baseURL: config.laravelApiUrl,
  timeout: 10000,
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
    'X-Requested-With': 'XMLHttpRequest',
  },
});

// Request interceptor for logging
laravelApi.interceptors.request.use(
  (requestConfig) => {
    logger.debug(`Laravel API: ${requestConfig.method?.toUpperCase()} ${requestConfig.url}`);
    return requestConfig;
  },
  (error) => {
    logger.error('Laravel API request error:', error);
    return Promise.reject(error);
  }
);

// Response interceptor for logging
laravelApi.interceptors.response.use(
  (response) => {
    return response;
  },
  (error) => {
    logger.error('Laravel API response error:', {
      url: error.config?.url,
      status: error.response?.status,
      message: error.message,
    });
    return Promise.reject(error);
  }
);

/**
 * Verify user token with Laravel
 */
export async function verifyToken(token) {
  try {
    const response = await laravelApi.post('/internal/verify-token', {}, {
      headers: {
        Authorization: `Bearer ${token}`,
      },
    });

    return response.data;
  } catch (error) {
    if (error.response?.status === 401) {
      return { valid: false, error: 'Invalid token' };
    }
    throw error;
  }
}

/**
 * Get user conversations
 */
export async function getConversations(token, params = {}) {
  const response = await laravelApi.get('/conversations', {
    headers: { Authorization: `Bearer ${token}` },
    params,
  });

  return response.data;
}

/**
 * Get conversation messages
 */
export async function getMessages(token, conversationId, params = {}) {
  const response = await laravelApi.get(`/conversations/${conversationId}/messages`, {
    headers: { Authorization: `Bearer ${token}` },
    params,
  });

  return response.data;
}

/**
 * Send message through Laravel to WABA
 */
export async function sendMessage(token, conversationId, message, tempId) {
  const response = await laravelApi.post('/internal/send-message', {
    conversation_id: conversationId,
    message,
    temp_id: tempId,
  }, {
    headers: { Authorization: `Bearer ${token}` },
  });

  return response.data;
}

/**
 * Mark conversation as read
 */
export async function markAsRead(token, conversationId) {
  const response = await laravelApi.post(`/conversations/${conversationId}/read`, {}, {
    headers: { Authorization: `Bearer ${token}` },
  });

  return response.data;
}

/**
 * Check user access to conversation
 */
export async function checkConversationAccess(token, conversationId) {
  try {
    const response = await laravelApi.get(`/internal/conversation-access/${conversationId}`, {
      headers: { Authorization: `Bearer ${token}` },
    });

    return response.data.hasAccess === true;
  } catch (error) {
    if (error.response?.status === 403 || error.response?.status === 404) {
      return false;
    }
    throw error;
  }
}

/**
 * Get user profile
 */
export async function getUserProfile(token) {
  const response = await laravelApi.get('/user', {
    headers: { Authorization: `Bearer ${token}` },
  });

  return response.data;
}

/**
 * Search contacts
 */
export async function searchContacts(token, query) {
  const response = await laravelApi.get('/contacts/search', {
    headers: { Authorization: `Bearer ${token}` },
    params: { q: query },
  });

  return response.data;
}

export default laravelApi;
