import { verifyToken } from '../services/laravel.js';
import logger from '../utils/logger.js';

/**
 * Socket.IO authentication middleware
 * Verifies JWT token with Laravel backend
 */
export async function authMiddleware(socket, next) {
  try {
    const token = socket.handshake.auth?.token ||
                  socket.handshake.headers?.authorization?.replace('Bearer ', '');

    if (!token) {
      logger.warn('Socket connection attempt without token', {
        socketId: socket.id,
        ip: socket.handshake.address,
      });
      return next(new Error('Authentication token required'));
    }

    // Verify token with Laravel
    const result = await verifyToken(token);

    if (!result.valid) {
      logger.warn('Socket connection with invalid token', {
        socketId: socket.id,
        ip: socket.handshake.address,
      });
      return next(new Error('Invalid authentication token'));
    }

    // Attach user data to socket
    socket.user = result.user;
    socket.userId = result.user.id;
    socket.tenantId = result.user.tenant_id;
    socket.token = token;

    logger.info('Socket authenticated', {
      socketId: socket.id,
      userId: socket.userId,
      tenantId: socket.tenantId,
    });

    next();

  } catch (error) {
    logger.error('Socket authentication error:', error);
    next(new Error('Authentication failed'));
  }
}

/**
 * Rate limiting middleware for socket events
 */
export function rateLimitMiddleware(maxEvents = 100, windowMs = 60000) {
  const eventCounts = new Map();

  // Cleanup old entries periodically
  setInterval(() => {
    const now = Date.now();
    for (const [key, data] of eventCounts.entries()) {
      if (now - data.timestamp > windowMs) {
        eventCounts.delete(key);
      }
    }
  }, windowMs);

  return (socket, next) => {
    const key = socket.userId || socket.id;
    const now = Date.now();

    let data = eventCounts.get(key);

    if (!data || now - data.timestamp > windowMs) {
      data = { count: 0, timestamp: now };
    }

    data.count++;

    if (data.count > maxEvents) {
      logger.warn('Rate limit exceeded', { socketId: socket.id, userId: socket.userId });
      return next(new Error('Rate limit exceeded'));
    }

    eventCounts.set(key, data);
    next();
  };
}
