import { authMiddleware } from '../middleware/auth.js';
import logger from '../utils/logger.js';
import messageHandler from './handlers/message.js';
import typingHandler from './handlers/typing.js';
import conversationHandler from './handlers/conversation.js';

/**
 * Setup Socket.IO handlers and middleware
 */
export default function setupSocketHandlers(io) {
  // Authentication middleware
  io.use(authMiddleware);

  // Connection handler
  io.on('connection', (socket) => {
    logger.info('Client connected', {
      socketId: socket.id,
      userId: socket.userId,
      tenantId: socket.tenantId,
    });

    // Join user's personal room (for direct notifications)
    socket.join(`user:${socket.userId}`);

    // Join tenant room (for tenant-wide broadcasts)
    if (socket.tenantId) {
      socket.join(`tenant:${socket.tenantId}`);
    }

    // Initialize handlers
    messageHandler(io, socket);
    typingHandler(io, socket);
    conversationHandler(io, socket);

    // Handle disconnect
    socket.on('disconnect', (reason) => {
      logger.info('Client disconnected', {
        socketId: socket.id,
        userId: socket.userId,
        reason,
      });
    });

    // Handle errors
    socket.on('error', (error) => {
      logger.error('Socket error', {
        socketId: socket.id,
        userId: socket.userId,
        error: error.message,
      });
    });
  });

  logger.info('Socket.IO handlers initialized');
}
