import { checkConversationAccess, sendMessage } from '../../services/laravel.js';
import logger from '../../utils/logger.js';

export default function messageHandler(io, socket) {
  /**
   * Handle sending a new message
   */
  socket.on('message:send', async (data) => {
    const { conversationId, message, tempId } = data;

    try {
      // Validate input
      if (!conversationId || !message?.trim()) {
        return socket.emit('message:error', { tempId, error: 'Invalid message' });
      }

      if (message.length > 4096) {
        return socket.emit('message:error', { tempId, error: 'Message too long' });
      }

      // Verify user has access to this conversation
      const hasAccess = await checkConversationAccess(socket.token, conversationId);
      if (!hasAccess) {
        logger.warn('Unauthorized message attempt', { userId: socket.userId, conversationId });
        return socket.emit('message:error', { tempId, error: 'Access denied' });
      }

      // Acknowledge sending
      socket.emit('message:sending', { tempId, status: 'sending', timestamp: Date.now() });

      // Send via Laravel API (which sends to WhatsApp)
      const result = await sendMessage(socket.token, {
        conversation_id: conversationId,
        message: message.trim(),
        temp_id: tempId,
      });

      if (result.success) {
        socket.emit('message:sent', {
          tempId,
          id: result.message.id,
          status: 'sent',
          timestamp: Date.now(),
        });

        logger.debug('Message sent', { userId: socket.userId, conversationId, messageId: result.message.id });
      } else {
        socket.emit('message:error', { tempId, error: result.error || 'Failed to send' });
      }

    } catch (error) {
      logger.error('Message send error:', error);
      socket.emit('message:error', {
        tempId,
        error: error.response?.data?.error || 'Failed to send message',
      });
    }
  });

  /**
   * Handle marking messages as read
   */
  socket.on('message:read', (data) => {
    const { conversationId, messageIds } = data;

    if (!conversationId) return;

    // Notify other users in the conversation
    socket.to(`conversation:${conversationId}`).emit('message:read', {
      userId: socket.userId,
      conversationId,
      messageIds: messageIds || [],
      timestamp: Date.now(),
    });
  });
}
