import axios from 'axios';
import config from '../config/index.js';
import logger from '../utils/logger.js';

const laravelApi = axios.create({
  baseURL: config.laravelApiUrl,
  timeout: 10000,
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
});

/**
 * Verify user token with Laravel
 */
export async function verifyToken(token) {
  try {
    const response = await laravelApi.post('/waochat/verify-token', {}, {
      headers: { Authorization: `Bearer ${token}` },
    });
    return response.data;
  } catch (error) {
    logger.error('Token verification failed:', error.message);
    return { valid: false };
  }
}

/**
 * Check if user has access to conversation
 */
export async function checkConversationAccess(token, conversationId) {
  try {
    const response = await laravelApi.post(
      `/waochat/verify-access/${conversationId}`,
      {},
      { headers: { Authorization: `Bearer ${token}` } }
    );
    return response.data.hasAccess === true;
  } catch (error) {
    logger.error('Conversation access check failed:', error.message);
    return false;
  }
}

/**
 * Send message via Laravel API
 */
export async function sendMessage(token, data) {
  try {
    const response = await laravelApi.post('/waochat/send-message', data, {
      headers: { Authorization: `Bearer ${token}` },
    });
    return response.data;
  } catch (error) {
    logger.error('Send message failed:', error.response?.data || error.message);
    throw error;
  }
}
