import { Router } from 'express';
import config from '../config/index.js';
import logger from '../utils/logger.js';

const router = Router();

/**
 * Webhook authentication middleware
 */
const webhookAuth = (req, res, next) => {
  const token = req.headers['x-webhook-secret'] || req.query.secret;

  if (!token || token !== config.webhookSecret) {
    logger.warn('Webhook auth failed', { ip: req.ip });
    return res.status(401).json({ error: 'Unauthorized' });
  }
  next();
};

/**
 * Setup webhook routes with Socket.IO instance
 */
export default function setupWebhooks(io) {
  router.use(webhookAuth);

  // New message from Laravel (incoming WhatsApp message)
  router.post('/message/incoming', (req, res) => {
    try {
      const data = req.body;
      logger.debug('Webhook: incoming message', { conversationId: data.conversationId });

      // Emit to conversation room
      io.to(`conversation:${data.conversationId}`).emit('message:new', {
        id: data.id,
        conversationId: data.conversationId,
        from: data.from,
        to: data.to,
        message: data.message,
        messageType: data.messageType || 'text',
        media: data.media || null,
        timestamp: data.timestamp,
        status: 'delivered',
      });

      // Emit notification to tenant
      if (data.tenantId) {
        io.to(`tenant:${data.tenantId}`).emit('notification:message', {
          conversationId: data.conversationId,
          preview: data.message?.substring(0, 100),
          from: data.from,
        });
      }

      res.json({ success: true });
    } catch (error) {
      logger.error('Webhook error:', error);
      res.status(500).json({ error: error.message });
    }
  });

  // Message status update (sent, delivered, read, failed)
  router.post('/message/status', (req, res) => {
    try {
      const data = req.body;
      logger.debug('Webhook: status update', { messageId: data.messageId, status: data.status });

      io.to(`conversation:${data.conversationId}`).emit('message:status', {
        messageId: data.messageId,
        tempId: data.tempId,
        status: data.status,
        error: data.error,
        timestamp: data.timestamp,
      });

      res.json({ success: true });
    } catch (error) {
      logger.error('Webhook error:', error);
      res.status(500).json({ error: error.message });
    }
  });

  // Conversation update
  router.post('/conversation/update', (req, res) => {
    try {
      const data = req.body;
      if (data.tenantId) {
        io.to(`tenant:${data.tenantId}`).emit('conversation:update', data);
      }
      res.json({ success: true });
    } catch (error) {
      logger.error('Webhook error:', error);
      res.status(500).json({ error: error.message });
    }
  });

  // Broadcast to specific users or tenant
  router.post('/broadcast', (req, res) => {
    try {
      const { event, data, userIds, tenantId } = req.body;

      if (userIds?.length) {
        userIds.forEach(userId => io.to(`user:${userId}`).emit(event, data));
      } else if (tenantId) {
        io.to(`tenant:${tenantId}`).emit(event, data);
      }

      res.json({ success: true });
    } catch (error) {
      logger.error('Webhook error:', error);
      res.status(500).json({ error: error.message });
    }
  });

  return router;
}
