<?php

namespace Modules\WaoChat\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class RealtimeService
{
    protected $baseUrl;
    protected $webhookSecret;
    protected $timeout;
    protected $enabled;

    public function __construct()
    {
        $this->enabled = config('waochat.nodejs.enabled', true);
        $this->baseUrl = config('waochat.nodejs.url', 'http://127.0.0.1:3000');
        $this->webhookSecret = config('waochat.nodejs.webhook_secret');
        $this->timeout = config('waochat.nodejs.timeout', 5);
    }

    /**
     * Send new message notification to connected clients
     */
    public function notifyNewMessage(array $data): bool
    {
        return $this->sendWebhook('/webhook/message/incoming', $data);
    }

    /**
     * Send message status update (sent, delivered, read, failed)
     */
    public function notifyMessageStatus(array $data): bool
    {
        return $this->sendWebhook('/webhook/message/status', $data);
    }

    /**
     * Send conversation update notification
     */
    public function notifyConversationUpdate(array $data): bool
    {
        return $this->sendWebhook('/webhook/conversation/update', $data);
    }

    /**
     * Broadcast event to specific users or tenant
     */
    public function broadcast(string $event, array $data, ?array $userIds = null, ?int $tenantId = null): bool
    {
        return $this->sendWebhook('/webhook/broadcast', [
            'event' => $event,
            'data' => $data,
            'userIds' => $userIds,
            'tenantId' => $tenantId,
        ]);
    }

    /**
     * Send webhook to Node.js server
     */
    protected function sendWebhook(string $endpoint, array $data): bool
    {
        if (!$this->enabled) {
            Log::debug('WaoChat: Realtime disabled, skipping webhook');
            return true;
        }

        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders([
                    'X-Webhook-Secret' => $this->webhookSecret,
                    'Content-Type' => 'application/json',
                ])
                ->post($this->baseUrl . $endpoint, $data);

            if ($response->successful()) {
                Log::debug('WaoChat: Webhook sent', ['endpoint' => $endpoint]);
                return true;
            }

            Log::warning('WaoChat: Webhook failed', [
                'endpoint' => $endpoint,
                'status' => $response->status(),
            ]);
            return false;

        } catch (\Exception $e) {
            Log::error('WaoChat: Webhook error', [
                'endpoint' => $endpoint,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Check if Node.js server is healthy
     */
    public function healthCheck(): array
    {
        try {
            $response = Http::timeout(3)->get($this->baseUrl . '/health');
            return [
                'status' => $response->successful() ? 'ok' : 'error',
                'data' => $response->json(),
            ];
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'error' => $e->getMessage(),
            ];
        }
    }
}
