<?php

namespace Modules\WaoChat\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Tenant\Chat;
use App\Models\Tenant\ChatMessage;
use App\Traits\WhatsApp;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Modules\WaoChat\Traits\NotifiesRealtime;

class RealtimeController extends Controller
{
    use WhatsApp, NotifiesRealtime;

    /**
     * Health check endpoint
     */
    public function health(): JsonResponse
    {
        $nodeHealth = $this->realtime()->healthCheck();

        return response()->json([
            'laravel' => 'ok',
            'nodejs' => $nodeHealth['status'],
            'nodejs_data' => $nodeHealth['data'] ?? null,
            'timestamp' => now()->toISOString(),
        ]);
    }

    /**
     * Verify user token and return user info
     */
    public function verifyToken(Request $request): JsonResponse
    {
        $user = $request->user();

        if (!$user) {
            return response()->json(['valid' => false], 401);
        }

        return response()->json([
            'valid' => true,
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'tenant_id' => $user->tenant_id,
                'avatar' => $user->avatar ?? null,
            ],
        ]);
    }

    /**
     * Verify user has access to conversation
     */
    public function verifyAccess(Request $request, int $conversationId): JsonResponse
    {
        $user = $request->user();

        if (!$user || !$user->tenant_id) {
            return response()->json(['hasAccess' => false], 403);
        }

        $subdomain = tenant_subdomain_by_tenant_id($user->tenant_id);

        $chat = Chat::fromTenant($subdomain)
            ->where('id', $conversationId)
            ->where('tenant_id', $user->tenant_id)
            ->first();

        if (!$chat) {
            return response()->json(['hasAccess' => false], 403);
        }

        return response()->json([
            'hasAccess' => true,
            'conversation' => [
                'id' => $chat->id,
                'phone' => $chat->receiver_id,
            ],
        ]);
    }

    /**
     * Send message via WhatsApp Cloud API
     */
    public function sendMessage(Request $request): JsonResponse
    {
        $request->validate([
            'conversation_id' => 'required|integer',
            'message' => 'required|string|max:4096',
            'temp_id' => 'required|string',
        ]);

        $user = $request->user();

        if (!$user || !$user->tenant_id) {
            return response()->json(['success' => false, 'error' => 'Unauthorized'], 403);
        }

        $subdomain = tenant_subdomain_by_tenant_id($user->tenant_id);
        $conversationId = $request->input('conversation_id');

        $chat = Chat::fromTenant($subdomain)
            ->where('id', $conversationId)
            ->where('tenant_id', $user->tenant_id)
            ->first();

        if (!$chat) {
            return response()->json(['success' => false, 'error' => 'Conversation not found'], 404);
        }

        try {
            // Initialize WhatsApp
            $this->setWaTenantId($user->tenant_id);
            $recipientPhone = ltrim($chat->receiver_id, '+');

            // Send via WhatsApp Cloud API
            $whatsappApi = $this->loadConfig();
            $result = $whatsappApi->sendTextMessage($recipientPhone, $request->input('message'), true);
            $responseData = json_decode($result->body());
            $wabaMessageId = $responseData->messages[0]->id ?? null;

            // Get sender phone
            $senderPhone = get_tenant_setting_from_db('whatsapp', 'wm_default_phone_number', $user->id);

            // Store message in database
            $message = new ChatMessage();
            $message->setTable($subdomain . '_chat_messages');
            $message->tenant_id = $user->tenant_id;
            $message->interaction_id = $conversationId;
            $message->sender_id = $senderPhone;
            $message->message = $request->input('message');
            $message->type = 'text';
            $message->time_sent = now();
            $message->status = 'sent';
            $message->message_id = $wabaMessageId;
            $message->staff_id = $user->id;
            $message->save();

            // Update chat last message
            $chat->last_message = $request->input('message');
            $chat->last_msg_time = now();
            $chat->save();

            // Notify realtime clients about status
            $this->notifyMessageStatus(
                $conversationId,
                $message->id,
                'sent',
                $request->input('temp_id')
            );

            return response()->json([
                'success' => true,
                'message' => [
                    'id' => $message->id,
                    'conversationId' => $conversationId,
                    'status' => 'sent',
                    'wabaMessageId' => $wabaMessageId,
                ],
            ]);

        } catch (\Exception $e) {
            // Notify failure
            $this->notifyMessageStatus(
                $conversationId,
                0,
                'failed',
                $request->input('temp_id'),
                $e->getMessage()
            );

            whatsapp_log('WaoChat: Send message failed', 'error', [
                'conversation_id' => $conversationId,
                'error' => $e->getMessage(),
            ], $e, $user->tenant_id);

            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
