# WaoChat Laravel Module

This module provides real-time chat functionality for the WAO application without modifying core files.

## Installation

### 1. Copy Module Files

Copy the `WaoChat` folder to `/var/www/wao.my/modules/`:

```bash
cp -r WaoChat /var/www/wao.my/modules/
```

### 2. Update composer.json

Add to `autoload.psr-4` in `/var/www/wao.my/composer.json`:

```json
{
    "autoload": {
        "psr-4": {
            "App\\": "app/",
            "Modules\\WaoChat\\": "modules/WaoChat/src/"
        }
    }
}
```

Then run:
```bash
cd /var/www/wao.my
composer dump-autoload
```

### 3. Register Service Provider

**For Laravel 11+**, edit `bootstrap/providers.php`:

```php
return [
    App\Providers\AppServiceProvider::class,
    // ... other providers
    Modules\WaoChat\WaoChatServiceProvider::class,
];
```

**For Laravel 10 and below**, edit `config/app.php`:

```php
'providers' => [
    // ... other providers
    Modules\WaoChat\WaoChatServiceProvider::class,
],
```

### 4. Add Environment Variables

Add to `/var/www/wao.my/.env`:

```env
# WaoChat Configuration
WAOCHAT_ENABLED=true
WAOCHAT_NODEJS_URL=http://127.0.0.1:3000
WAOCHAT_WEBHOOK_SECRET=your-secure-secret-here
WAOCHAT_TIMEOUT=5
WAOCHAT_PWA_URL=https://web.wao.my
```

### 5. Clear Cache

```bash
cd /var/www/wao.my
php artisan config:clear
php artisan cache:clear
php artisan route:clear
```

### 6. Verify Installation

```bash
curl https://wao.my/api/waochat/health
```

## Usage

### Sending Real-time Notifications

In your existing controllers/services, use the `NotifiesRealtime` trait:

```php
use Modules\WaoChat\Traits\NotifiesRealtime;

class YourController extends Controller
{
    use NotifiesRealtime;

    public function handleIncomingMessage($data)
    {
        // ... your existing code to save message ...

        // Notify connected clients
        $this->notifyNewMessage(
            $chatId,
            $messageId,
            $tenantId,
            $message,
            $senderPhone
        );
    }
}
```

### Available Methods

```php
// Notify about new message
$this->notifyNewMessage($chatId, $messageId, $tenantId, $message, $from, $messageType, $media);

// Notify about message status change
$this->notifyMessageStatus($conversationId, $messageId, $status, $tempId, $error);

// Notify about conversation update
$this->notifyConversationUpdate($conversationId, $tenantId, $data);

// Get realtime service directly
$this->realtime()->broadcast($event, $data, $userIds, $tenantId);
```

## API Endpoints

| Method | Endpoint | Auth | Description |
|--------|----------|------|-------------|
| GET | `/api/waochat/health` | No | Health check |
| POST | `/api/waochat/verify-token` | Sanctum | Verify user token |
| POST | `/api/waochat/verify-access/{id}` | Sanctum | Verify conversation access |
| POST | `/api/waochat/send-message` | Sanctum | Send message via WhatsApp |

## Updates

This module is designed to survive application updates since it's in a separate `modules/` directory. After any update:

1. Verify `composer.json` still has the autoload entry
2. Verify the service provider is still registered
3. Run `composer dump-autoload`
4. Clear caches
