<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Seller extends Model
{
    protected $fillable = [
        'owner_id',
        'fname',
        'lname',
        'email',
        'address',
        'region',
        'phone',
        'status',
        'type',
    ];

    protected $casts = [
        'status' => 'boolean',
    ];

    /**
     * Get the manager who owns this seller
     */
    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    /**
     * Get codes assigned to this seller
     */
    public function codes(): BelongsToMany
    {
        return $this->belongsToMany(Code::class, 'seller_qr', 'seller_id', 'code_id')
                    ->withTimestamps();
    }

    /**
     * Get full name attribute
     */
    public function getFullNameAttribute(): string
    {
        return "{$this->fname} {$this->lname}";
    }

    /**
     * Scope to get resellers only
     */
    public function scopeResellers($query)
    {
        return $query->where('type', 'reseller');
    }

    /**
     * Scope to get agents only
     */
    public function scopeAgents($query)
    {
        return $query->where('type', 'agent');
    }

    /**
     * Scope to get sellers owned by specific manager
     */
    public function scopeOwnedBy($query, $userId)
    {
        return $query->where('owner_id', $userId);
    }

    /**
     * Scope to get active sellers
     */
    public function scopeActive($query)
    {
        return $query->where('status', true);
    }
}
