<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class QrCode extends Model
{
    protected $fillable = [
        'serial',
        'qr_pin',
        'status',
        'used_at',
        'assigned_to_id',
        'assigned_to_type',
        'last_exported_at',
        'product_id',
        'is_lucky',
        'batch_id',
        'attempts',
    ];

    protected $casts = [
        'is_lucky' => 'boolean',
        'attempts' => 'integer',
        'used_at' => 'datetime',
        'last_exported_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Generate unique QR PIN (8 characters, uppercase alphanumeric)
     */
    public static function generateQrPin(): string
    {
        do {
            $pin = strtoupper(Str::random(8));
            // Ensure only alphanumeric (A-Z, 0-9)
            $pin = preg_replace('/[^A-Z0-9]/', '', $pin);

            // If length is less than 8, regenerate
            if (strlen($pin) < 8) {
                continue;
            }

            $pin = substr($pin, 0, 8);
        } while (self::where('qr_pin', $pin)->exists());

        return $pin;
    }

    /**
     * Check if code is used
     */
    public function isUsed(): bool
    {
        return $this->status === 'used';
    }

    /**
     * Mark code as used
     */
    public function markAsUsed(): void
    {
        $this->update([
            'status' => 'used',
            'used_at' => now(),
        ]);
    }

    /**
     * Toggle lucky status
     */
    public function toggleLucky(): void
    {
        $this->update(['is_lucky' => !$this->is_lucky]);
    }

    /**
     * Scope for unused codes
     */
    public function scopeUnused($query)
    {
        return $query->where('status', 'unused');
    }

    /**
     * Scope for used codes
     */
    public function scopeUsed($query)
    {
        return $query->where('status', 'used');
    }

    /**
     * Scope for lucky codes
     */
    public function scopeLucky($query)
    {
        return $query->where('is_lucky', true);
    }

    /**
     * Get formatted URL for this code
     */
    public function getUrlAttribute(): string
    {
        return url('/' . $this->serial);
    }
}
