<?php

namespace App\Livewire\Public;

use App\Models\Customer;
use App\Models\CustomerField;
use App\Models\Site;
use App\Services\CodeVerificationService;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

class CodeVerification extends Component
{
    use WithFileUploads;

    // Form fields
    public string $serial = '';
    public string $name = '';
    public string $email = '';
    public string $contact = '';
    public array $extraFields = [];

    // State
    public ?array $verificationResult = null;
    public bool $showSuccessModal = false;
    public $prizeImage;

    /**
     * Verify the code and save customer data
     */
    public function verifyCode()
    {
        $config = Site::config();

        // Build validation rules
        $rules = [
            'serial' => 'required|string|min:5',
            'name' => 'required|string|max:250',
            'email' => 'required|email|max:250',
            'contact' => 'required|string|max:50',
        ];

        // Add dynamic field validation
        $fields = CustomerField::visible()->get();
        foreach ($fields as $field) {
            if (!in_array($field->unique_key, ['name', 'email', 'contact']) && $field->visible) {
                $rules['extraFields.' . $field->unique_key] = $field->required ? 'required|string|max:500' : 'nullable|string|max:500';
            }
        }

        $this->validate($rules);

        // Verify the code
        $service = new CodeVerificationService();
        $this->verificationResult = $service->verify($this->serial, request()->ip());

        // If verification successful, save customer data
        if ($this->verificationResult['status'] === 'success') {
            Customer::create([
                'name' => $this->name,
                'email' => $this->email,
                'contact' => $this->contact,
                'serial' => $this->serial,
                'extra_fields' => $this->extraFields,
                'is_lucky_draw' => $this->verificationResult['is_lucky'],
            ]);

            // Show success modal
            $this->showSuccessModal = true;
        }
    }

    /**
     * Close the success modal
     */
    public function closeModal()
    {
        $this->showSuccessModal = false;
        $this->reset(['serial', 'name', 'email', 'contact', 'extraFields', 'verificationResult', 'prizeImage']);
    }

    /**
     * Upload prize image
     */
    public function uploadPrizeImage()
    {
        $this->validate([
            'prizeImage' => 'required|image|max:5120', // 5MB
        ]);

        // Store the image
        $path = $this->prizeImage->store('prize-images', 'public');

        session()->flash('imageUploaded', 'Image uploaded successfully!');
    }

    public function render()
    {
        $siteConfig = Site::config();
        // Get all visible fields except pincode (pincode is shown separately at the bottom)
        $customerFields = CustomerField::visible()
            ->where('unique_key', '!=', 'pincode')
            ->get();

        // Get pincode field for dynamic placeholder
        $pincodeField = CustomerField::where('unique_key', 'pincode')->first();
        $pincodePlaceholder = $pincodeField ? $pincodeField->placeholder : 'Enter pincode here *';

        return view('livewire.public.code-verification', [
            'siteConfig' => $siteConfig,
            'customerFields' => $customerFields,
            'pincodePlaceholder' => $pincodePlaceholder,
        ]);
    }
}
