<?php

namespace App\Livewire\Admin;

use App\Models\CustomerField;
use App\Models\Site;
use Livewire\Component;
use Livewire\WithFileUploads;

class Settings extends Component
{
    use WithFileUploads;

    // Site Configuration
    public $name = '';
    public $descr = '';
    public $email = '';
    public $logo;
    public $logoPath = '';
    public $width = 120;
    public $height = 120;

    // Contest Configuration
    public $contestImage;
    public $contestImagePath = '';
    public $contestWidth = 400;
    public $contestHeight = 600;

    // QR Image Configuration
    public $qrImage;
    public $qrImagePath = '';
    public $qrWidth = 500;
    public $qrHeight = 250;

    // Customer Fields
    public $fields = [];
    public $editingPlaceholder = null;
    public $placeholderValue = '';
    public $pincodeField = [];

    public function mount()
    {
        $this->loadSiteConfig();
        $this->loadCustomerFields();
    }

    public function loadSiteConfig()
    {
        $site = Site::config();

        $this->name = $site->name ?? '';
        $this->descr = $site->descr ?? '';
        $this->email = $site->email ?? '';
        $this->logoPath = $site->logo ?? '';
        $this->width = $site->width ?? 120;
        $this->height = $site->height ?? 120;

        // Load contest data
        $contest = $site->contest ?? [];
        $this->contestImagePath = $contest['image'] ?? '';
        $this->contestWidth = $contest['width'] ?? 400;
        $this->contestHeight = $contest['height'] ?? 600;

        $this->qrImagePath = $contest['qr_image'] ?? '';
        $this->qrWidth = $contest['qr_width'] ?? 500;
        $this->qrHeight = $contest['qr_height'] ?? 250;
    }

    public function loadCustomerFields()
    {
        // Load all fields except pincode (pincode is handled separately)
        $this->fields = CustomerField::where('unique_key', '!=', 'pincode')
            ->orderBy('id')
            ->get()
            ->toArray();

        // Load or create pincode field
        $pincode = CustomerField::where('unique_key', 'pincode')->first();
        if (!$pincode) {
            $pincode = CustomerField::create([
                'unique_key' => 'pincode',
                'name' => 'Pincode/Serial Number',
                'placeholder' => 'Enter pincode here *',
                'required' => true,
                'visible' => true,
            ]);
        }
        $this->pincodeField = $pincode->toArray();
    }

    public function saveSiteConfig()
    {
        $this->validate([
            'name' => 'required|string|max:255',
            'descr' => 'nullable|string|max:500',
            'email' => 'required|email|max:255',
            'width' => 'required|integer|min:50|max:500',
            'height' => 'required|integer|min:50|max:500',
            'contestWidth' => 'required|integer|min:100|max:1000',
            'contestHeight' => 'required|integer|min:100|max:1000',
            'qrWidth' => 'required|integer|min:100|max:1000',
            'qrHeight' => 'required|integer|min:100|max:1000',
            'logo' => 'nullable|image|max:2048',
            'contestImage' => 'nullable|image|max:5120',
            'qrImage' => 'nullable|image|max:2048',
        ]);

        $site = Site::firstOrNew();

        $site->name = $this->name;
        $site->descr = $this->descr;
        $site->email = $this->email;
        $site->width = $this->width;
        $site->height = $this->height;

        // Handle logo upload
        if ($this->logo) {
            $logoPath = $this->logo->store('images', 'public');
            $site->logo = '/storage/' . $logoPath;
            $this->logoPath = $site->logo;
        }

        // Prepare contest data
        $contestData = $site->contest ?? [];
        $contestData['width'] = $this->contestWidth;
        $contestData['height'] = $this->contestHeight;
        $contestData['qr_width'] = $this->qrWidth;
        $contestData['qr_height'] = $this->qrHeight;

        // Handle contest image upload
        if ($this->contestImage) {
            $contestPath = $this->contestImage->store('images', 'public');
            $contestData['image'] = '/storage/' . $contestPath;
            $this->contestImagePath = $contestData['image'];
        }

        // Handle QR image upload
        if ($this->qrImage) {
            $qrPath = $this->qrImage->store('images', 'public');
            $contestData['qr_image'] = '/storage/' . $qrPath;
            $this->qrImagePath = $contestData['qr_image'];
        }

        $site->contest = $contestData;
        $site->save();

        $this->logo = null;
        $this->contestImage = null;
        $this->qrImage = null;

        session()->flash('message', 'Site configuration updated successfully!');
    }

    public function toggleVisible($fieldId)
    {
        $field = CustomerField::find($fieldId);
        if ($field) {
            $field->visible = !$field->visible;
            $field->save();
            $this->loadCustomerFields();
            session()->flash('message', 'Field visibility updated successfully!');
        }
    }

    public function toggleRequired($fieldId)
    {
        $field = CustomerField::find($fieldId);
        if ($field) {
            $field->required = !$field->required;
            $field->save();
            $this->loadCustomerFields();
            session()->flash('message', 'Field requirement updated successfully!');
        }
    }

    public function editPlaceholder($fieldId, $currentPlaceholder)
    {
        $this->editingPlaceholder = $fieldId;
        $this->placeholderValue = $currentPlaceholder;
    }

    public function updatePlaceholder($fieldId)
    {
        $field = CustomerField::find($fieldId);
        if ($field && trim($this->placeholderValue) !== '') {
            $field->placeholder = $this->placeholderValue;
            $field->save();
            $this->loadCustomerFields();
            session()->flash('message', 'Placeholder updated successfully!');
        }
        $this->editingPlaceholder = null;
        $this->placeholderValue = '';
    }

    public function cancelEdit()
    {
        $this->editingPlaceholder = null;
        $this->placeholderValue = '';
    }

    public function render()
    {
        return view('livewire.admin.settings')
            ->layout('layouts.app');
    }
}
