<?php

namespace App\Livewire\Admin\QrCodes;

use App\Jobs\GenerateQrCodeBatch;
use App\Models\QrCode;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Livewire\Component;

class Generate extends Component
{
    public $quantity = 10000;
    public $startingSerial = '';
    public $isGenerating = false;
    public $currentBatchId = null;
    public $progress = 0;
    public $totalQuantity = 0;

    protected $rules = [
        'quantity' => 'required|integer|min:10000',
        'startingSerial' => 'nullable|integer|min:0',
    ];

    public function mount()
    {
        // Check if there's an ongoing generation
        $this->checkOngoingGeneration();
    }

    public function increment()
    {
        $this->quantity += 10000;
    }

    public function decrement()
    {
        if ($this->quantity > 10000) {
            $this->quantity -= 10000;
        }
    }

    public function generateCodes()
    {
        $this->validate();

        // Determine starting serial
        if ($this->startingSerial) {
            $startSerial = (int) $this->startingSerial;
        } else {
            // Auto-increment from last serial
            $lastCode = QrCode::orderBy('serial', 'desc')->first();
            $startSerial = $lastCode ? ((int) $lastCode->serial) + 1 : 100000;
        }

        // Create batch ID
        $batchId = 'batch_' . Str::random(10) . '_' . time();

        // Store batch info in cache
        Cache::put("qr_batch_progress_{$batchId}", 0, now()->addHours(24));
        Cache::put("qr_batch_status_{$batchId}", 'generating', now()->addHours(24));

        // Dispatch jobs in batches of 1000
        $batchSize = 1000;
        $totalBatches = ceil($this->quantity / $batchSize);

        for ($i = 0; $i < $totalBatches; $i++) {
            $batchStart = $startSerial + ($i * $batchSize);
            $batchCount = min($batchSize, $this->quantity - ($i * $batchSize));

            GenerateQrCodeBatch::dispatch($batchId, $batchStart, $batchCount, $this->quantity);
        }

        $this->currentBatchId = $batchId;
        $this->totalQuantity = $this->quantity;
        $this->isGenerating = true;

        session()->flash('message', 'QR Code generation started! Total batches: ' . $totalBatches);
    }

    public function checkOngoingGeneration()
    {
        // Check cache for any ongoing generation
        $keys = Cache::get('ongoing_batch_ids', []);
        foreach ($keys as $batchId) {
            $status = Cache::get("qr_batch_status_{$batchId}");
            if ($status === 'generating') {
                $this->currentBatchId = $batchId;
                $this->isGenerating = true;
                $this->updateProgress();
                break;
            }
        }
    }

    public function updateProgress()
    {
        if (!$this->currentBatchId) {
            return;
        }

        $this->progress = Cache::get("qr_batch_progress_{$this->currentBatchId}", 0);
        $status = Cache::get("qr_batch_status_{$this->currentBatchId}", 'generating');

        if ($status === 'completed') {
            $this->isGenerating = false;
            session()->flash('message', 'QR Code generation completed! Generated ' . $this->progress . ' codes.');
        }
    }

    public function render()
    {
        // Auto-update progress every 2 seconds if generating
        if ($this->isGenerating) {
            $this->updateProgress();
        }

        return view('livewire.admin.qr-codes.generate')
            ->layout('layouts.app');
    }
}
