<?php

namespace App\Jobs;

use App\Models\QrCode;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Cache;

class GenerateQrCodeBatch implements ShouldQueue
{
    use Queueable;

    public $batchId;
    public $startSerial;
    public $count;
    public $totalQuantity;

    /**
     * Create a new job instance.
     */
    public function __construct($batchId, $startSerial, $count, $totalQuantity)
    {
        $this->batchId = $batchId;
        $this->startSerial = $startSerial;
        $this->count = $count;
        $this->totalQuantity = $totalQuantity;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $codes = [];

        for ($i = 0; $i < $this->count; $i++) {
            $serial = str_pad($this->startSerial + $i, 6, '0', STR_PAD_LEFT);

            // Check if serial already exists
            if (QrCode::where('serial', $serial)->exists()) {
                continue; // Skip duplicate
            }

            $codes[] = [
                'serial' => $serial,
                'qr_pin' => QrCode::generateQrPin(),
                'status' => 'unused',
                'used_at' => null,
                'is_lucky' => false,
                'attempts' => 0,
                'batch_id' => $this->batchId,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Bulk insert
        if (!empty($codes)) {
            QrCode::insert($codes);
        }

        // Update progress in cache
        $currentProgress = Cache::get("qr_batch_progress_{$this->batchId}", 0);
        $newProgress = $currentProgress + count($codes);
        Cache::put("qr_batch_progress_{$this->batchId}", $newProgress, now()->addHours(24));

        // Mark as complete if done
        if ($newProgress >= $this->totalQuantity) {
            Cache::put("qr_batch_status_{$this->batchId}", 'completed', now()->addHours(24));
        }
    }
}
