# GB Secure - TALL Stack Rebuild Documentation

## Executive Summary

**Current System:** CodeIgniter 4 (PHP Framework)
**Target System:** TALL Stack (Tailwind CSS, Alpine.js, Laravel, Livewire)
**Project Name:** GB Secure - Gengbiz Hologram Security Sticker System
**Purpose:** Product authentication system with QR/hologram verification, reseller management, and customer lucky draw functionality
**Backup Location:** `C:\laragon\www\holobit_backup_[timestamp]`

---

## Table of Contents

1. [Current System Overview](#1-current-system-overview)
2. [Database Schema Analysis](#2-database-schema-analysis)
3. [Features & Functionality](#3-features--functionality)
4. [TALL Stack Architecture Design](#4-tall-stack-architecture-design)
5. [Proposed Improvements](#5-proposed-improvements)
6. [Tech Stack Details](#6-tech-stack-details)
7. [Migration Strategy](#7-migration-strategy)
8. [Development Roadmap](#8-development-roadmap)
9. [Security Enhancements](#9-security-enhancements)
10. [Testing Strategy](#10-testing-strategy)

---

## 1. Current System Overview

### 1.1 Application Purpose

GB Secure is a comprehensive hologram/QR code authentication system designed for:
- **Product Authentication** - Verify genuine products via QR/serial codes
- **Anti-Counterfeiting** - Track verification attempts and flag suspicious codes
- **Distribution Management** - Manage multi-level reseller/agent hierarchy
- **Customer Engagement** - Lucky draw campaigns for verified products
- **Supply Chain Tracking** - Monitor product flow through distribution channels

### 1.2 Current Technology Stack

```
Backend:
- CodeIgniter 4 (PHP Framework)
- MySQL 8.4.3 (Database)
- PHPMailer via Sendinblue SMTP

Frontend:
- Bootstrap 4/5 (UI Framework)
- jQuery 3.5.1 (DOM Manipulation)
- DataTables (Server-side tables)
- Font Awesome 5.11.2 (Icons)
- SweetAlert (Modals)
- Select2 (Dropdowns)
- Google Fonts (Inter, MuseoModerno)

Third-Party Services:
- Google reCAPTCHA v2
- Sendinblue SMTP (Email delivery)
```

### 1.3 User Roles & Permissions

| Role | Permissions |
|------|------------|
| **Admin** | Full system access, manage all entities, system configuration |
| **Manager** | Limited access, manage owned agents/resellers only |
| **Reseller** | Top-level distributor, receives code allocations |
| **Agent** | Sub-distributor under reseller or manager |
| **Customer** | Public user, verify products, submit lucky draw entries |

### 1.4 Project Structure

```
holobit/
├── app/
│   ├── Config/          # Routes, Database, Email configs
│   ├── Controllers/     # Business logic (Home, Admin, Codes, Products, etc.)
│   ├── Models/          # Data access layer (9 models + 1 view)
│   ├── Views/           # Templates (admin, codes, customers, manager, products, resellers)
│   ├── Helpers/         # Custom helper functions
│   └── Common.php       # Shared functions
├── public/
│   ├── assets/          # CSS, JS, images
│   ├── index.php        # Front controller
│   └── .htaccess        # Apache rewrites
├── writable/            # Cache, logs, sessions, uploads
└── vendor/              # Composer dependencies
```

---

## 2. Database Schema Analysis

### 2.1 Complete Schema Overview

```sql
-- Tables: 9 tables + 1 view
1. codes                 # QR/Serial authentication codes
2. customer              # Customer submissions
3. customer_fields       # Dynamic form fields
4. customer_assign       # VIEW: Customer-Reseller join
5. product               # Product catalog
6. seller                # Resellers & Agents
7. seller_qr             # Code assignments (junction table)
8. user                  # System users (Admin/Managers)
9. site                  # System configuration
10. password_reset       # Password recovery tokens
```

### 2.2 Detailed Table Structures

#### **Table: `codes`** (Product Authentication Codes)
```sql
CREATE TABLE `codes` (
  `id` INT PRIMARY KEY AUTO_INCREMENT,
  `guid` VARCHAR(50) NOT NULL,           -- Unique serial number
  `qr` VARCHAR(50) NOT NULL,             -- QR code identifier
  `status` TEXT NOT NULL,                -- unused/used/fake
  `islucky` INT DEFAULT 0,               -- Lucky winner flag (0/1)
  `dated` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `exported` DATE DEFAULT NULL,          -- Export tracking
  `lastverified` DATE DEFAULT NULL,      -- Last verification date
  `ipv` TEXT,                            -- IP address of verifier
  `product` INT DEFAULT NULL,            -- FK to product table
  `attempts` INT DEFAULT 0,              -- Verification attempt count
  KEY `idx_guid_qr` (`guid`, `qr`)
) ENGINE=InnoDB;

-- Status values: "unused", "used", "fake"
-- After 5 attempts without proper verification, marked as "fake"
```

#### **Table: `customer`** (Customer Database)
```sql
CREATE TABLE `customer` (
  `id` INT PRIMARY KEY AUTO_INCREMENT,
  `name` VARCHAR(250) NOT NULL,
  `contact` VARCHAR(250) NOT NULL,       -- Phone number
  `email` VARCHAR(250) NOT NULL,
  `serial` VARCHAR(50) NOT NULL,         -- QR code verified
  `updatedOn` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  `createdOn` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `extra_fields` LONGTEXT,               -- JSON: Dynamic custom fields
  `is_lucky_draw` TINYINT(1) DEFAULT 0,  -- Flag for lucky draw submissions
  CONSTRAINT CHECK (JSON_VALID(`extra_fields`))
) ENGINE=InnoDB;

-- extra_fields format: {"field_key": "value", "address": "123 Main St", ...}
```

#### **Table: `customer_fields`** (Dynamic Form Configuration)
```sql
CREATE TABLE `customer_fields` (
  `id` INT PRIMARY KEY AUTO_INCREMENT,
  `unique_key` VARCHAR(255),             -- Field identifier (e.g., "address")
  `name` VARCHAR(255),                   -- Display label
  `placeholder` VARCHAR(255),            -- Input placeholder text
  `required` TINYINT(1) DEFAULT 0,       -- Required field flag
  `visible` TINYINT(1) DEFAULT 1         -- Visibility flag
) ENGINE=InnoDB;

-- Base fields (cannot be deleted): name, email, contact
-- Custom fields can be added/removed dynamically
```

#### **Table: `product`** (Product Catalog)
```sql
CREATE TABLE `product` (
  `id` INT PRIMARY KEY,
  `name` VARCHAR(250) NOT NULL,
  `code` VARCHAR(250) NOT NULL,          -- Product code/SKU
  `kkm` VARCHAR(250) NOT NULL,           -- KKM registration number
  `sku` VARCHAR(50) NOT NULL,
  `halal_status` DATE DEFAULT NULL,      -- Halal certification date
  `manufacture_dt` DATE DEFAULT NULL,
  `expiry_dt` DATE DEFAULT NULL,
  `updatedOn` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  `createdOn` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;
```

#### **Table: `seller`** (Resellers & Agents)
```sql
CREATE TABLE `seller` (
  `userId` INT PRIMARY KEY,              -- FK to user table for agents
  `id` VARCHAR(255),                     -- Custom identifier
  `owner` INT DEFAULT NULL,              -- FK to user (manager who owns this seller)
  `fname` VARCHAR(100),
  `lname` VARCHAR(100),
  `email` VARCHAR(100) NOT NULL,
  `address` VARCHAR(255),
  `region` VARCHAR(100),
  `phone` VARCHAR(50),
  `status` TINYINT(1) DEFAULT 1,         -- Active(1) / Disabled(0)
  `type` TINYINT(1) NOT NULL DEFAULT 1,  -- Seller(1) / Agent(2)
  `updatedOn` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `createdOn` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- type values: 1 = SELLER_KEY, 2 = AGENT_KEY
-- owner: NULL for resellers, user.id for agents owned by managers
```

#### **Table: `seller_qr`** (Code Assignments)
```sql
CREATE TABLE `seller_qr` (
  `id` INT PRIMARY KEY,
  `userId` INT NOT NULL,                 -- FK to seller.userId
  `qr_id` INT NOT NULL,                  -- FK to codes.id
  `createdOn` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Junction table: Many-to-Many relationship between sellers and codes
```

#### **Table: `user`** (System Users - Admin & Managers)
```sql
CREATE TABLE `user` (
  `id` INT PRIMARY KEY,
  `firstname` VARCHAR(255),
  `lastname` VARCHAR(255),
  `email` VARCHAR(255),
  `address` TEXT,
  `region` VARCHAR(20),
  `phonenumber` VARCHAR(255),
  `password` TEXT,                       -- SHA-256 hashed
  `secure` TINYINT(1) DEFAULT 0,         -- Password hashing status
  `status` VARCHAR(255),                 -- Active/Inactive
  `role` VARCHAR(255),                   -- Admin/Manager
  `inserton` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `insertby` VARCHAR(255),
  `type` VARCHAR(11) NOT NULL,
  `bank` TEXT NOT NULL,
  `paypal` TEXT NOT NULL
) ENGINE=InnoDB;

-- secure: 0 = unsafe (plain), 1 = safe (SHA-256 hashed)
-- role: "Admin" or "Manager"
```

#### **Table: `site`** (System Configuration)
```sql
CREATE TABLE `site` (
  `id` INT PRIMARY KEY,
  `name` VARCHAR(50) NOT NULL,           -- Site name
  `descr` TEXT NOT NULL,                 -- Instructions text
  `logo` TEXT NOT NULL,                  -- Logo file path
  `width` INT NOT NULL,                  -- Logo width
  `height` INT NOT NULL,                 -- Logo height
  `email` VARCHAR(255) NOT NULL,         -- Contact email
  `customer_response` TINYINT NOT NULL,  -- 0=None, 1=Before, 2=After serial
  `contest` LONGTEXT,                    -- JSON: Contest styling & images
  CONSTRAINT CHECK (JSON_VALID(`contest`))
) ENGINE=InnoDB;

-- contest JSON structure:
{
  "images": ["contest1.jpg", "contest2.jpg"],
  "styles": {
    "primaryColor": "#007bff",
    "secondaryColor": "#6c757d",
    "footerBgColor": "#f8f9fa"
  },
  "qr_sample": "hologram_sample.jpg",
  "whatsapp": "+60123456789"
}
```

#### **Table: `password_reset`** (Password Recovery)
```sql
CREATE TABLE `password_reset` (
  `id` INT PRIMARY KEY,
  `email` VARCHAR(250) NOT NULL,
  `keyValue` VARCHAR(250) NOT NULL,      -- Reset token
  `expDate` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  `createdon` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Tokens expire after 24 hours
```

#### **View: `customer_assign`** (Customer-Reseller Join)
```sql
CREATE VIEW `customer_assign` AS
SELECT
  customer.id,
  customer.name,
  customer.contact,
  customer.email,
  customer.serial,
  CONCAT(seller.fname, seller.lname) AS assign,  -- Assigned reseller name
  customer.extra_fields,
  customer.updatedOn
FROM customer
LEFT JOIN codes ON codes.qr = customer.serial
LEFT JOIN seller_qr ON seller_qr.qr_id = codes.id
LEFT JOIN seller ON seller.userId = seller_qr.userId;
```

### 2.3 Database Relationships

```
┌──────────┐
│   user   │──────┐
│ (Admin/  │      │ owner
│ Manager) │      │
└──────────┘      │
     │            ▼
     │      ┌──────────┐         ┌──────────────┐
     │      │  seller  │────────▶│  seller_qr   │
     │      │(Reseller │ userId  │  (junction)  │
     │      │ /Agent)  │         └──────────────┘
     │      └──────────┘               │ qr_id
     │                                 ▼
     │                           ┌──────────┐         ┌──────────┐
     │                           │  codes   │────────▶│ product  │
     │                           │(QR/GUID) │ product └──────────┘
     │                           └──────────┘
     │                                 │
     │                                 │ serial
     │                                 ▼
     │                           ┌──────────┐
     └──────────────────────────▶│ customer │
         (verification)          │ (public) │
                                 └──────────┘
                                       │
                                       │ uses schema from
                                       ▼
                                 ┌────────────────┐
                                 │customer_fields │
                                 │  (form config) │
                                 └────────────────┘
```

**Key Relationships:**
1. **User → Seller**: Managers own agents (`seller.owner → user.id`)
2. **Seller → seller_qr → Codes**: Code assignments
3. **Codes → Product**: Product tagging
4. **Codes → Customer**: Verification tracking
5. **Customer → customer_fields**: Dynamic form structure

---

## 3. Features & Functionality

### 3.1 Public Features (Customer-Facing)

#### **A. Product Verification System**
- **QR/Serial Code Validation**
  - Scan QR code or manually enter GUID
  - Real-time validation with attempt tracking
  - IP address logging for security
  - Progressive attempts: 1-5 tries before marking as fake

- **Verification States**
  ```
  1. First Verification (Genuine)
     ✓ Success message
     ✓ Mark code as "used"
     ✓ Store IP address and timestamp
     ✓ Show product details

  2. Subsequent Verifications
     ⚠ Warning: "Already verified"
     ⚠ Show attempt count (2/5, 3/5, etc.)
     ⚠ Display original verification date

  3. Invalid Code / Exceeded Attempts
     ✗ Error: "Fake/Counterfeit product"
     ✗ Block further verifications
  ```

- **Lucky Draw Integration**
  - Certain codes marked as "lucky winners"
  - Special congratulations message for winners
  - Form submission for prize claims
  - Google reCAPTCHA v2 protection
  - Prevent duplicate submissions per serial

#### **B. Customer Form Modes**
Three operational modes configurable by admin:

1. **Disabled** - No customer data collection
2. **Before Serial** - Mandatory form BEFORE verification
3. **After Serial** - Optional form AFTER successful verification (lucky draw accordion)

#### **C. Dynamic Custom Fields**
- Admin-configurable form fields
- Base fields (mandatory): name, email, contact
- Custom fields: address, city, state, postcode, etc.
- JSON storage for flexible data structure

### 3.2 Admin Panel Features

#### **A. Dashboard**
- Total QR codes generated
- Quick access menu
- System status overview

#### **B. QR Code Management** (`/codes`)
- **Generate Codes**
  - Batch creation of GUID + QR pairs
  - Automatic unique ID generation
  - Track generation date

- **Code Operations**
  - View all codes (DataTables with pagination)
  - Mark codes as lucky winners
  - Export codes to Excel by date range
  - Delete codes
  - Filter by status (unused/used/fake)

- **Export Tracking**
  - Mark codes as exported with date
  - Prevent duplicate exports

#### **C. Product Management** (`/products`)
- **CRUD Operations**
  - Add/Edit/Delete products
  - Fields: name, code, KKM, SKU, halal status, manufacture date, expiry date

- **Code-Product Assignment**
  - Tag serial ranges to products (bulk operation)
  - Example: Assign GUID-1000 to GUID-1500 to "Product A"
  - Untag codes from products
  - Autocomplete product search

#### **D. Reseller/Agent Management** (`/resellers`)
- **Hierarchy Management**
  ```
  Admin
  ├── Reseller A (top-level)
  │   └── [Assigned QR codes]
  ├── Reseller B
  └── Manager
      ├── Agent 1 (owned by Manager)
      ├── Agent 2
      └── [Each agent has QR assignments]
  ```

- **Operations**
  - Add/Edit resellers and agents
  - Assign QR codes (single or bulk)
  - View assigned codes per reseller/agent
  - Bulk assignment by serial ranges
  - Track code allocation history
  - Active/Inactive status toggle

#### **E. Branch Manager System** (`/manager`)
- **Admin-Only Feature**
  - Create manager accounts with limited permissions
  - Managers can only manage their owned agents
  - Role-based access control

- **Manager Capabilities**
  - Create agents under their ownership
  - Assign codes to their agents
  - View only their owned entities
  - Cannot access other managers' data

#### **F. Customer Database** (`/customers`)
- **View All Submissions**
  - DataTables with search/filter
  - Display customer info + extra fields
  - Show assigned reseller (who distributed the code)
  - Flag lucky draw submissions

- **Export to Excel**
  - Date range filtering
  - Include all custom fields
  - Reseller assignment tracking

#### **G. System Configuration** (`/admin/setup`)
- **Branding Settings**
  - Upload logo with custom dimensions
  - Primary color (header, buttons)
  - Secondary color (footer, accents)
  - Background colors

- **Contest/Lucky Draw Settings**
  - Upload contest images (carousel)
  - Upload QR hologram sample image
  - Set WhatsApp contact number
  - Instructions text

- **Customer Form Configuration**
  - Choose form mode (disabled/before/after serial)
  - Manage custom fields
    - Add new fields (unique key, name, placeholder)
    - Set required/optional status
    - Show/hide fields
    - Delete custom fields (except base fields)

- **Email Configuration**
  - SMTP settings (Sendinblue integration)
  - From email and display name

### 3.3 Authentication & Security Features

#### **A. User Authentication**
- Login system for Admin/Manager roles
- Session management with expiry
- Password hashing (SHA-256)
- "Remember Me" functionality

#### **B. Password Recovery**
- Forgot password flow
- Token-based reset (24-hour expiry)
- Email delivery via SMTP
- Secure token generation

#### **C. Security Measures**
- CSRF protection (CodeIgniter built-in)
- Role-based access control (Admin vs Manager)
- IP logging for verification attempts
- Attempt tracking to detect counterfeits
- SQL injection prevention (prepared statements)
- XSS protection (output escaping)

### 3.4 Reporting & Analytics

#### **A. Export Capabilities**
- Export QR codes to Excel (date range filter)
- Export customer database (date range filter)
- Include all custom fields and assignments

#### **B. Data Visualization**
- DataTables with server-side processing
- Search, filter, sort, paginate
- Custom column visibility
- Responsive tables

---

## 4. TALL Stack Architecture Design

### 4.1 TALL Stack Components

**T** - **Tailwind CSS** (Utility-first CSS framework)
**A** - **Alpine.js** (Lightweight JavaScript framework)
**L** - **Laravel** (PHP Framework)
**L** - **Livewire** (Full-stack framework for Laravel)

### 4.2 Why TALL Stack?

| Feature | CodeIgniter 4 (Current) | TALL Stack (Proposed) |
|---------|------------------------|----------------------|
| **Frontend Reactivity** | jQuery + DOM manipulation | Alpine.js + Livewire (reactive) |
| **UI Framework** | Bootstrap 4/5 | Tailwind CSS (highly customizable) |
| **Code Organization** | MVC with separate views | Livewire components (self-contained) |
| **Real-time Updates** | Manual AJAX calls | Livewire auto-sync |
| **Testing** | Limited testing support | Built-in Feature/Unit tests (PHPUnit, Pest) |
| **ORM** | Query Builder | Eloquent ORM with relationships |
| **Authentication** | Custom implementation | Laravel Breeze/Jetstream |
| **API Support** | Manual routes | Laravel Sanctum/Passport |
| **Queue Management** | Not available | Laravel Queues |
| **Event System** | Custom | Laravel Events/Listeners |
| **Validation** | Controller-based | Form Requests + Livewire validation |
| **File Storage** | Manual | Laravel Storage (S3, local, etc.) |
| **Caching** | Basic | Redis, Memcached, File, Array |
| **Localization** | Manual | Built-in i18n |

### 4.3 Proposed Laravel Project Structure

```
gb-secure-tallstack/
├── app/
│   ├── Http/
│   │   ├── Controllers/          # Traditional controllers for API
│   │   ├── Livewire/             # Livewire components (main UI logic)
│   │   │   ├── Admin/
│   │   │   │   ├── Dashboard.php
│   │   │   │   ├── SystemSetup.php
│   │   │   │   └── ChangePassword.php
│   │   │   ├── Codes/
│   │   │   │   ├── CodeList.php
│   │   │   │   ├── CodeGenerator.php
│   │   │   │   ├── LuckyCodeManager.php
│   │   │   │   └── CodeExporter.php
│   │   │   ├── Products/
│   │   │   │   ├── ProductList.php
│   │   │   │   ├── ProductForm.php
│   │   │   │   ├── BulkTagging.php
│   │   │   │   └── CodeAssignment.php
│   │   │   ├── Resellers/
│   │   │   │   ├── ResellerList.php
│   │   │   │   ├── ResellerForm.php
│   │   │   │   ├── AgentList.php
│   │   │   │   ├── AgentForm.php
│   │   │   │   └── BulkAssignment.php
│   │   │   ├── Managers/
│   │   │   │   ├── ManagerList.php
│   │   │   │   └── ManagerForm.php
│   │   │   ├── Customers/
│   │   │   │   ├── CustomerList.php
│   │   │   │   └── CustomerExporter.php
│   │   │   └── Public/
│   │   │       ├── CodeVerification.php
│   │   │       ├── CustomerForm.php
│   │   │       └── LuckyDrawForm.php
│   │   ├── Middleware/
│   │   │   ├── RoleMiddleware.php
│   │   │   └── ManagerScopeMiddleware.php
│   │   └── Requests/
│   │       ├── StoreCodeRequest.php
│   │       ├── StoreProductRequest.php
│   │       └── CustomerFormRequest.php
│   ├── Models/
│   │   ├── User.php               # Admin/Manager
│   │   ├── Code.php               # QR/GUID codes
│   │   ├── Product.php
│   │   ├── Seller.php             # Resellers/Agents
│   │   ├── Customer.php
│   │   ├── CustomerField.php
│   │   ├── SellerQr.php           # Pivot model
│   │   ├── Site.php               # Configuration
│   │   └── PasswordReset.php
│   ├── Services/
│   │   ├── CodeGenerationService.php
│   │   ├── CodeVerificationService.php
│   │   ├── ExportService.php
│   │   └── EmailService.php
│   ├── Enums/                     # PHP 8.1+ Enums
│   │   ├── CodeStatus.php         # unused, used, fake
│   │   ├── UserRole.php           # Admin, Manager
│   │   ├── SellerType.php         # Reseller, Agent
│   │   └── CustomerFormMode.php   # None, Before, After
│   ├── Observers/
│   │   ├── CodeObserver.php       # Track code changes
│   │   └── CustomerObserver.php
│   ├── Events/
│   │   ├── CodeVerified.php
│   │   ├── LuckyCodeDetected.php
│   │   └── CounterfeitDetected.php
│   ├── Listeners/
│   │   ├── SendLuckyWinnerEmail.php
│   │   └── NotifyAdminCounterfeit.php
│   └── Policies/
│       ├── CodePolicy.php
│       ├── SellerPolicy.php
│       └── ProductPolicy.php
├── database/
│   ├── migrations/                # Database versioning
│   │   ├── 2024_01_01_create_users_table.php
│   │   ├── 2024_01_02_create_codes_table.php
│   │   ├── 2024_01_03_create_products_table.php
│   │   ├── 2024_01_04_create_sellers_table.php
│   │   ├── 2024_01_05_create_seller_qr_table.php
│   │   ├── 2024_01_06_create_customers_table.php
│   │   ├── 2024_01_07_create_customer_fields_table.php
│   │   ├── 2024_01_08_create_sites_table.php
│   │   └── 2024_01_09_create_password_resets_table.php
│   ├── seeders/
│   │   ├── DatabaseSeeder.php
│   │   ├── AdminUserSeeder.php
│   │   ├── DefaultCustomerFieldsSeeder.php
│   │   └── SiteConfigSeeder.php
│   └── factories/
│       ├── UserFactory.php
│       ├── CodeFactory.php
│       └── ProductFactory.php
├── resources/
│   ├── views/
│   │   ├── layouts/
│   │   │   ├── app.blade.php       # Public layout
│   │   │   ├── admin.blade.php     # Admin layout
│   │   │   └── guest.blade.php     # Login/Register layout
│   │   ├── livewire/               # Livewire component views
│   │   │   ├── admin/
│   │   │   ├── codes/
│   │   │   ├── products/
│   │   │   ├── resellers/
│   │   │   ├── managers/
│   │   │   ├── customers/
│   │   │   └── public/
│   │   ├── components/             # Blade components
│   │   │   ├── alert.blade.php
│   │   │   ├── modal.blade.php
│   │   │   ├── data-table.blade.php
│   │   │   └── sidebar.blade.php
│   │   └── emails/
│   │       ├── forgot-password.blade.php
│   │       └── lucky-winner.blade.php
│   ├── css/
│   │   └── app.css                 # Tailwind CSS entry
│   └── js/
│       ├── app.js                  # Alpine.js entry
│       └── components/
│           ├── qr-scanner.js
│           └── code-generator.js
├── routes/
│   ├── web.php                     # Web routes
│   ├── api.php                     # API routes
│   └── channels.php                # Broadcasting channels
├── tests/
│   ├── Feature/
│   │   ├── CodeVerificationTest.php
│   │   ├── CodeGenerationTest.php
│   │   ├── ResellerManagementTest.php
│   │   └── CustomerFormTest.php
│   └── Unit/
│       ├── CodeServiceTest.php
│       └── VerificationLogicTest.php
├── public/
│   ├── storage/                    # Symlink to storage/app/public
│   ├── build/                      # Compiled assets (Vite)
│   └── index.php
├── storage/
│   ├── app/
│   │   ├── public/
│   │   │   ├── logos/
│   │   │   ├── contest-images/
│   │   │   └── qr-samples/
│   │   └── exports/                # Excel exports
│   ├── logs/
│   └── framework/
├── config/
│   ├── livewire.php
│   ├── services.php                # reCAPTCHA, SMTP configs
│   └── filesystems.php
├── .env
├── composer.json
├── package.json
├── tailwind.config.js
├── vite.config.js
└── artisan
```

### 4.4 Key Laravel Packages to Use

```json
{
  "require": {
    "php": "^8.2",
    "laravel/framework": "^11.0",
    "livewire/livewire": "^3.0",
    "laravel/sanctum": "^4.0",          // API authentication
    "spatie/laravel-permission": "^6.0",  // Role/Permission management
    "maatwebsite/excel": "^3.1",        // Excel import/export
    "barryvdh/laravel-dompdf": "^2.0",  // PDF generation
    "intervention/image": "^3.0",       // Image manipulation
    "google/recaptcha": "^1.3"          // reCAPTCHA v2/v3
  },
  "require-dev": {
    "laravel/pint": "^1.0",             // Code styling
    "pestphp/pest": "^2.0",             // Testing framework
    "larastan/larastan": "^2.0",        // Static analysis
    "laravel/telescope": "^5.0"         // Debugging tool (dev only)
  }
}
```

```json
{
  "devDependencies": {
    "@tailwindcss/forms": "^0.5.0",
    "@tailwindcss/typography": "^0.5.0",
    "alpinejs": "^3.13.0",
    "autoprefixer": "^10.4.0",
    "postcss": "^8.4.0",
    "tailwindcss": "^3.4.0",
    "vite": "^5.0.0",
    "laravel-vite-plugin": "^1.0"
  }
}
```

### 4.5 Eloquent Models with Relationships

#### **Code Model** (`app/Models/Code.php`)
```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Enums\CodeStatus;

class Code extends Model
{
    protected $fillable = [
        'guid', 'qr', 'status', 'islucky', 'exported',
        'lastverified', 'ipv', 'product_id', 'attempts'
    ];

    protected $casts = [
        'status' => CodeStatus::class,
        'islucky' => 'boolean',
        'exported' => 'date',
        'lastverified' => 'date',
        'dated' => 'datetime',
    ];

    // Relationships
    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function sellers(): BelongsToMany
    {
        return $this->belongsToMany(Seller::class, 'seller_qr', 'qr_id', 'user_id')
                    ->withTimestamps();
    }

    public function customers(): HasMany
    {
        return $this->hasMany(Customer::class, 'serial', 'qr');
    }

    // Scopes
    public function scopeUnused($query)
    {
        return $query->where('status', CodeStatus::Unused);
    }

    public function scopeLucky($query)
    {
        return $query->where('islucky', true);
    }

    public function scopeVerifiedBetween($query, $start, $end)
    {
        return $query->whereBetween('lastverified', [$start, $end]);
    }
}
```

#### **Customer Model** (`app/Models/Customer.php`)
```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Customer extends Model
{
    protected $table = 'customer';

    protected $fillable = [
        'name', 'contact', 'email', 'serial',
        'extra_fields', 'is_lucky_draw'
    ];

    protected $casts = [
        'extra_fields' => 'array',
        'is_lucky_draw' => 'boolean',
    ];

    public function code(): BelongsTo
    {
        return $this->belongsTo(Code::class, 'serial', 'qr');
    }

    public function assignedSeller(): BelongsTo
    {
        return $this->hasOneThrough(
            Seller::class,
            Code::class,
            'qr',           // Foreign key on codes table
            'user_id',      // Foreign key on sellers table
            'serial',       // Local key on customers table
            'id'            // Local key on codes table
        );
    }
}
```

#### **Seller Model** (`app/Models/Seller.php`)
```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use App\Enums\SellerType;

class Seller extends Model
{
    protected $table = 'seller';
    protected $primaryKey = 'userId';

    protected $fillable = [
        'userId', 'id', 'owner', 'fname', 'lname', 'email',
        'address', 'region', 'phone', 'status', 'type'
    ];

    protected $casts = [
        'type' => SellerType::class,
        'status' => 'boolean',
    ];

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner');
    }

    public function codes(): BelongsToMany
    {
        return $this->belongsToMany(Code::class, 'seller_qr', 'userId', 'qr_id')
                    ->withTimestamps();
    }

    public function scopeResellers($query)
    {
        return $query->where('type', SellerType::Reseller);
    }

    public function scopeAgents($query)
    {
        return $query->where('type', SellerType::Agent);
    }

    public function scopeOwnedBy($query, $userId)
    {
        return $query->where('owner', $userId);
    }
}
```

### 4.6 Livewire Component Example

#### **CodeVerification Component** (`app/Http/Livewire/Public/CodeVerification.php`)
```php
<?php

namespace App\Http\Livewire\Public;

use Livewire\Component;
use App\Services\CodeVerificationService;
use App\Models\Code;
use App\Enums\CodeStatus;

class CodeVerification extends Component
{
    public string $serial = '';
    public ?array $verificationResult = null;
    public bool $showCustomerForm = false;

    protected $rules = [
        'serial' => 'required|string|min:5',
    ];

    public function verifyCode()
    {
        $this->validate();

        $service = new CodeVerificationService();
        $this->verificationResult = $service->verify($this->serial, request()->ip());

        if ($this->verificationResult['status'] === 'success' && $this->verificationResult['is_lucky']) {
            $this->dispatch('showConfetti');
        }

        $this->showCustomerForm = $this->shouldShowForm();
    }

    private function shouldShowForm(): bool
    {
        $config = \App\Models\Site::first();
        return $config->customer_response === 2 &&
               $this->verificationResult['status'] === 'success';
    }

    public function render()
    {
        return view('livewire.public.code-verification')
            ->layout('layouts.app');
    }
}
```

#### **Blade View** (`resources/views/livewire/public/code-verification.blade.php`)
```blade
<div class="max-w-4xl mx-auto p-6">
    <div class="bg-white rounded-lg shadow-lg p-8">
        <h1 class="text-3xl font-bold text-gray-900 mb-6">
            Verify Your Product
        </h1>

        <form wire:submit="verifyCode" class="space-y-4">
            <div>
                <label for="serial" class="block text-sm font-medium text-gray-700">
                    Enter Serial Number or Scan QR Code
                </label>
                <input
                    type="text"
                    wire:model="serial"
                    id="serial"
                    class="mt-1 block w-full rounded-md border-gray-300 shadow-sm
                           focus:border-indigo-500 focus:ring-indigo-500"
                    placeholder="GUID-XXXX-XXXX"
                >
                @error('serial')
                    <span class="text-red-500 text-sm">{{ $message }}</span>
                @enderror
            </div>

            <button
                type="submit"
                wire:loading.attr="disabled"
                class="w-full bg-indigo-600 text-white py-3 rounded-md
                       hover:bg-indigo-700 disabled:opacity-50"
            >
                <span wire:loading.remove>Verify Code</span>
                <span wire:loading>Verifying...</span>
            </button>
        </form>

        @if($verificationResult)
            <div class="mt-6 p-4 rounded-lg
                        {{ $verificationResult['status'] === 'success' ? 'bg-green-100' : 'bg-red-100' }}">
                <p class="font-semibold">{{ $verificationResult['message'] }}</p>

                @if($verificationResult['is_lucky'])
                    <div class="mt-4 p-4 bg-yellow-100 rounded-lg">
                        <h2 class="text-2xl font-bold text-yellow-800">
                            🎉 Congratulations! You're a Lucky Winner! 🎉
                        </h2>
                    </div>
                @endif
            </div>
        @endif

        @if($showCustomerForm)
            <livewire:public.customer-form :serial="$serial" />
        @endif
    </div>
</div>
```

### 4.7 Service Layer Pattern

#### **CodeVerificationService** (`app/Services/CodeVerificationService.php`)
```php
<?php

namespace App\Services;

use App\Models\Code;
use App\Enums\CodeStatus;
use App\Events\CodeVerified;
use App\Events\CounterfeitDetected;

class CodeVerificationService
{
    private const MAX_ATTEMPTS = 5;

    public function verify(string $serial, string $ipAddress): array
    {
        $code = Code::where('qr', $serial)
                    ->orWhere('guid', $serial)
                    ->first();

        if (!$code) {
            return [
                'status' => 'error',
                'message' => 'Invalid code. Please check and try again.',
                'is_lucky' => false,
            ];
        }

        // Increment attempts
        $code->increment('attempts');

        // Check if fake
        if ($code->status === CodeStatus::Fake || $code->attempts >= self::MAX_ATTEMPTS) {
            $code->update(['status' => CodeStatus::Fake]);
            event(new CounterfeitDetected($code));

            return [
                'status' => 'error',
                'message' => 'This code has been flagged as counterfeit.',
                'is_lucky' => false,
            ];
        }

        // First-time verification
        if ($code->status === CodeStatus::Unused) {
            $code->update([
                'status' => CodeStatus::Used,
                'lastverified' => now(),
                'ipv' => $ipAddress,
            ]);

            event(new CodeVerified($code));

            return [
                'status' => 'success',
                'message' => 'Genuine product verified!',
                'is_lucky' => $code->islucky,
                'product' => $code->product,
            ];
        }

        // Already verified
        return [
            'status' => 'warning',
            'message' => "This code was already verified on {$code->lastverified->format('M d, Y')}. " .
                        "Attempt {$code->attempts} of " . self::MAX_ATTEMPTS,
            'is_lucky' => false,
        ];
    }
}
```

---

## 5. Proposed Improvements

### 5.1 UI/UX Enhancements

#### **Current Issues:**
- Bootstrap 4/5 mixed usage (inconsistent styling)
- jQuery-heavy code (outdated patterns)
- No mobile-first approach
- Limited accessibility (ARIA labels)
- DataTables pagination not responsive

#### **Proposed Solutions:**

**A. Tailwind CSS Design System**
```javascript
// tailwind.config.js
module.exports = {
  theme: {
    extend: {
      colors: {
        primary: {
          50: '#f0f9ff',
          500: '#3b82f6',
          900: '#1e3a8a',
        },
        secondary: {
          500: '#6366f1',
        },
        success: '#10b981',
        warning: '#f59e0b',
        danger: '#ef4444',
      },
      fontFamily: {
        sans: ['Inter', 'system-ui', 'sans-serif'],
        heading: ['MuseoModerno', 'cursive'],
      },
    },
  },
  plugins: [
    require('@tailwindcss/forms'),
    require('@tailwindcss/typography'),
  ],
}
```

**B. Responsive Admin Dashboard**
- Mobile-friendly sidebar (collapsible)
- Touch-optimized navigation
- Adaptive data tables (stack on mobile)
- Dark mode support

**C. Improved Public Verification Page**
```
┌─────────────────────────────────────┐
│  [Logo]          GB Secure          │
│                                     │
│  ┌───────────────────────────┐     │
│  │  Verify Your Product      │     │
│  │                           │     │
│  │  [Scan QR Code]           │     │
│  │       or                  │     │
│  │  [Enter Serial: ______]   │     │
│  │  [Verify Button]          │     │
│  └───────────────────────────┘     │
│                                     │
│  Results Card (animated entrance)   │
│  ✓ Genuine Product                 │
│  Product: XYZ                       │
│  Verified: Date                     │
│                                     │
│  [Lucky Draw Form Accordion]        │
└─────────────────────────────────────┘
```

**D. Animation & Transitions**
- Smooth page transitions (Alpine.js `x-transition`)
- Loading skeletons (instead of spinners)
- Confetti effect for lucky winners (canvas-confetti)
- Success/error toast notifications (instead of alerts)

### 5.2 Security Enhancements

#### **A. Upgrade Password Hashing**
```php
// Current: SHA-256 (insecure for passwords)
hash('sha256', $password);

// Proposed: bcrypt/Argon2 (Laravel default)
Hash::make($password); // Uses bcrypt by default
```

#### **B. Two-Factor Authentication (2FA)**
```bash
composer require laravel/fortify
# Enable 2FA for admin accounts
# Use Google Authenticator / SMS OTP
```

#### **C. Rate Limiting**
```php
// Prevent brute force on code verification
RateLimiter::for('code-verification', function ($request) {
    return Limit::perMinute(10)->by($request->ip());
});

// Admin login rate limiting
RateLimiter::for('admin-login', function ($request) {
    return Limit::perMinute(5)->by($request->email);
});
```

#### **D. Enhanced Audit Logging**
```php
// Log all admin actions
use Spatie\Activitylog\Traits\LogsActivity;

class Code extends Model
{
    use LogsActivity;

    protected static $logAttributes = ['guid', 'qr', 'status', 'islucky'];
    protected static $logName = 'code_management';
}

// Query audit logs
activity()->log('Admin changed code status');
```

#### **E. Content Security Policy (CSP)**
```php
// config/secure-headers.php
return [
    'csp' => [
        'script-src' => [
            'self',
            'https://www.google.com/recaptcha/',
            'https://cdn.jsdelivr.net',
        ],
        'img-src' => ['self', 'data:', 'https:'],
    ],
];
```

### 5.3 Performance Improvements

#### **A. Database Indexing**
```php
// Add composite indexes for faster queries
Schema::table('codes', function (Blueprint $table) {
    $table->index(['status', 'islucky']);
    $table->index(['lastverified', 'exported']);
});

Schema::table('seller_qr', function (Blueprint $table) {
    $table->index(['userId', 'qr_id']);
});
```

#### **B. Eager Loading (N+1 Prevention)**
```php
// Bad (N+1 queries)
$customers = Customer::all();
foreach ($customers as $customer) {
    echo $customer->code->product->name; // N queries
}

// Good (2 queries total)
$customers = Customer::with('code.product')->get();
```

#### **C. Query Result Caching**
```php
// Cache configuration queries
$siteConfig = Cache::remember('site_config', 3600, function () {
    return Site::first();
});

// Cache product list
$products = Cache::remember('products_list', 600, function () {
    return Product::all();
});
```

#### **D. Queue Long-Running Tasks**
```php
// Export to Excel (background job)
dispatch(new ExportCustomersJob($startDate, $endDate, $email));

// Send lucky winner emails (queued)
dispatch(new SendLuckyWinnerEmailJob($customer));
```

#### **E. Asset Optimization**
```bash
# Vite build with optimization
npm run build

# Result: Minified CSS/JS, tree-shaking, code splitting
```

### 5.4 Feature Additions

#### **A. Advanced Analytics Dashboard**
```
┌─────────────────────────────────────┐
│  Dashboard                          │
│                                     │
│  [Total Codes]  [Verified]  [Fake] │
│    10,000        7,500      25     │
│                                     │
│  ┌─ Verification Trend ──────────┐ │
│  │  Chart.js line graph          │ │
│  │  (verifications over time)    │ │
│  └───────────────────────────────┘ │
│                                     │
│  ┌─ Top Resellers ───────────────┐ │
│  │  1. Reseller A - 2,500 codes  │ │
│  │  2. Reseller B - 1,800 codes  │ │
│  └───────────────────────────────┘ │
│                                     │
│  ┌─ Geographic Distribution ─────┐ │
│  │  Map showing verification IPs │ │
│  └───────────────────────────────┘ │
└─────────────────────────────────────┘
```

#### **B. QR Code Scanner (Web Camera)**
```javascript
// Use html5-qrcode library
import { Html5Qrcode } from "html5-qrcode";

Alpine.data('qrScanner', () => ({
    scanner: null,
    init() {
        this.scanner = new Html5Qrcode("reader");
    },
    startScan() {
        this.scanner.start(
            { facingMode: "environment" },
            { fps: 10, qrbox: 250 },
            this.onScanSuccess
        );
    },
    onScanSuccess(decodedText) {
        this.$wire.set('serial', decodedText);
        this.$wire.verifyCode();
        this.scanner.stop();
    }
}));
```

#### **C. Multi-Language Support**
```php
// resources/lang/en/messages.php
return [
    'verify_product' => 'Verify Your Product',
    'enter_serial' => 'Enter Serial Number',
    'genuine_product' => 'Genuine Product Verified!',
];

// resources/lang/ms/messages.php (Malay)
return [
    'verify_product' => 'Sahkan Produk Anda',
    'enter_serial' => 'Masukkan Nombor Siri',
    'genuine_product' => 'Produk Tulen Disahkan!',
];

// Blade usage
{{ __('messages.verify_product') }}
```

#### **D. API for Mobile Apps**
```php
// routes/api.php
Route::middleware('auth:sanctum')->group(function () {
    Route::post('/verify-code', [ApiController::class, 'verifyCode']);
    Route::get('/products', [ApiController::class, 'products']);
    Route::get('/customer-fields', [ApiController::class, 'customerFields']);
});

// Response format
{
    "status": "success",
    "data": {
        "code": "GUID-1234-5678",
        "verified_at": "2024-01-15 10:30:00",
        "is_lucky": true,
        "product": {
            "name": "Product A",
            "sku": "SKU-001"
        }
    }
}
```

#### **E. Notification System**
```php
// Real-time notifications via Laravel Echo + Pusher
// Notify admin when counterfeit detected
Notification::send($admins, new CounterfeitDetectedNotification($code));

// Email + In-app notifications
class CounterfeitDetectedNotification extends Notification
{
    public function via($notifiable)
    {
        return ['mail', 'database', 'broadcast'];
    }
}
```

#### **F. Advanced Reporting**
```php
// Custom reports
- Verification by date range
- Reseller performance ranking
- Geographic distribution
- Lucky draw winners list
- Counterfeit detection log
- Code export history

// Export formats: Excel, PDF, CSV
```

#### **G. Bulk Operations**
```php
// Bulk code generation (with progress bar)
Livewire component with progress tracking:
- Generate 10,000 codes in batches
- Show progress: "Generating... 4,567 / 10,000"
- Chunked database inserts (500 per batch)

// Bulk reseller assignment
- Select multiple resellers
- Assign code ranges to all at once
- Preview before confirm
```

### 5.5 Code Quality Improvements

#### **A. Type Hints & Strict Types**
```php
<?php

declare(strict_types=1);

namespace App\Services;

class CodeGenerationService
{
    public function generate(int $quantity, ?int $productId = null): array
    {
        // Strong typing prevents bugs
    }
}
```

#### **B. PHP 8.2+ Features**
```php
// Readonly properties
class Code extends Model
{
    public readonly string $guid;
}

// Enums instead of constants
enum CodeStatus: string
{
    case Unused = 'unused';
    case Used = 'used';
    case Fake = 'fake';
}

// Match expressions
$message = match($code->status) {
    CodeStatus::Unused => 'Ready to use',
    CodeStatus::Used => 'Already verified',
    CodeStatus::Fake => 'Counterfeit detected',
};
```

#### **C. Form Request Validation**
```php
// app/Http/Requests/StoreCodeRequest.php
class StoreCodeRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'quantity' => 'required|integer|min:1|max:10000',
            'product_id' => 'nullable|exists:products,id',
        ];
    }

    public function messages(): array
    {
        return [
            'quantity.max' => 'Cannot generate more than 10,000 codes at once.',
        ];
    }
}
```

#### **D. Repository Pattern (Optional)**
```php
// app/Repositories/CodeRepository.php
interface CodeRepositoryInterface
{
    public function findBySerial(string $serial): ?Code;
    public function generateCodes(int $quantity): array;
    public function markAsLucky(int $id): bool;
}

class CodeRepository implements CodeRepositoryInterface
{
    public function findBySerial(string $serial): ?Code
    {
        return Code::where('qr', $serial)
                   ->orWhere('guid', $serial)
                   ->first();
    }
}
```

---

## 6. Tech Stack Details

### 6.1 Laravel 11 Setup

```bash
# Install Laravel
composer create-project laravel/laravel gb-secure-tallstack

# Install Livewire
composer require livewire/livewire

# Install additional packages
composer require spatie/laravel-permission
composer require maatwebsite/excel
composer require intervention/image
composer require barryvdh/laravel-dompdf
composer require google/recaptcha

# Install frontend dependencies
npm install -D tailwindcss postcss autoprefixer
npm install -D @tailwindcss/forms @tailwindcss/typography
npm install alpinejs
npm install chart.js
npm install html5-qrcode

# Generate Tailwind config
npx tailwindcss init -p
```

### 6.2 Environment Configuration

```env
# .env
APP_NAME="GB Secure"
APP_ENV=production
APP_KEY=base64:...
APP_DEBUG=false
APP_URL=https://gbsecure.com

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=gb_secure
DB_USERNAME=root
DB_PASSWORD=

MAIL_MAILER=smtp
MAIL_HOST=smtp-relay.sendinblue.com
MAIL_PORT=587
MAIL_USERNAME=gengbizmail@gmail.com
MAIL_PASSWORD=1SRPXI6HvnC70AJE
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=gengbizmail@gmail.com
MAIL_FROM_NAME="GB Secure"

RECAPTCHA_SITE_KEY=your_site_key
RECAPTCHA_SECRET_KEY=your_secret_key

QUEUE_CONNECTION=database
CACHE_DRIVER=redis
SESSION_DRIVER=redis
```

### 6.3 Tailwind CSS Configuration

```javascript
// tailwind.config.js
export default {
  content: [
    "./resources/**/*.blade.php",
    "./resources/**/*.js",
    "./app/Http/Livewire/**/*.php",
  ],
  theme: {
    extend: {
      colors: {
        brand: {
          50: '#f0f9ff',
          100: '#e0f2fe',
          500: '#0ea5e9',
          600: '#0284c7',
          700: '#0369a1',
          900: '#0c4a6e',
        },
      },
    },
  },
  plugins: [
    require('@tailwindcss/forms'),
    require('@tailwindcss/typography'),
  ],
}
```

### 6.4 Vite Configuration

```javascript
// vite.config.js
import { defineConfig } from 'vite';
import laravel from 'laravel-vite-plugin';

export default defineConfig({
    plugins: [
        laravel({
            input: ['resources/css/app.css', 'resources/js/app.js'],
            refresh: true,
        }),
    ],
});
```

### 6.5 Alpine.js Setup

```javascript
// resources/js/app.js
import Alpine from 'alpinejs';
import 'html5-qrcode';

// Global Alpine components
Alpine.data('sidebar', () => ({
    open: false,
    toggle() {
        this.open = !this.open;
    }
}));

Alpine.data('modal', () => ({
    show: false,
    open() {
        this.show = true;
    },
    close() {
        this.show = false;
    }
}));

window.Alpine = Alpine;
Alpine.start();
```

---

## 7. Migration Strategy

### 7.1 Data Migration Plan

#### **Phase 1: Setup New Laravel Project**
```bash
1. Create new Laravel 11 project
2. Install all dependencies
3. Configure .env with database credentials
4. Set up version control (Git)
```

#### **Phase 2: Database Migration**
```bash
# Export current database
mysqldump -u root holobit > holobit_export.sql

# Import to new database
mysql -u root gb_secure < holobit_export.sql

# Run Laravel migrations (will modify structure)
php artisan migrate
```

#### **Phase 3: Create Laravel Migrations**
```php
// Create migrations matching existing schema
php artisan make:migration create_codes_table
php artisan make:migration create_customers_table
php artisan make:migration create_products_table
php artisan make:migration create_sellers_table
php artisan make:migration create_seller_qr_table
php artisan make:migration create_customer_fields_table
php artisan make:migration create_sites_table
php artisan make:migration create_users_table

// Modifications:
- Add foreign key constraints
- Add proper indexes
- Standardize timestamp columns (created_at, updated_at)
```

#### **Phase 4: Data Cleaning**
```sql
-- Update password hashing
UPDATE user SET
    password = '$2y$10$...',  -- Re-hash with bcrypt
    secure = 1
WHERE secure = 0;

-- Standardize status values
UPDATE codes SET status = 'unused' WHERE status = 'unused';
UPDATE codes SET status = 'used' WHERE status = 'used';
UPDATE codes SET status = 'fake' WHERE status = 'fake';

-- Fix timestamp columns
ALTER TABLE codes
    CHANGE dated created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    ADD updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP;
```

### 7.2 File Migration

```bash
# Copy uploaded files
cp -r holobit/writable/uploads gb-secure-tallstack/storage/app/public/

# Copy logo and images
cp -r holobit/public/assets/images/* gb-secure-tallstack/storage/app/public/images/

# Create symlink
php artisan storage:link
```

### 7.3 Testing Strategy Before Launch

#### **A. Feature Tests**
```php
// tests/Feature/CodeVerificationTest.php
public function test_can_verify_valid_code()
{
    $code = Code::factory()->create(['status' => 'unused']);

    $response = $this->post('/verify', ['serial' => $code->qr]);

    $response->assertOk();
    $this->assertEquals('used', $code->fresh()->status);
}

public function test_detects_fake_after_max_attempts()
{
    $code = Code::factory()->create(['attempts' => 5]);

    $response = $this->post('/verify', ['serial' => $code->qr]);

    $response->assertStatus(422);
    $this->assertEquals('fake', $code->fresh()->status);
}
```

#### **B. Manual Testing Checklist**
```
□ Public code verification (unused, used, fake)
□ Lucky code detection and form submission
□ Customer form submission (before/after serial modes)
□ Admin login and dashboard
□ Code generation (single and bulk)
□ Product CRUD operations
□ Reseller/Agent CRUD operations
□ Code assignment to resellers
□ Manager login and scoped access
□ Excel exports (codes, customers)
□ Password reset flow
□ System configuration (logo, colors, contest)
□ Custom field management
□ Mobile responsiveness
□ Cross-browser compatibility
```

### 7.4 Deployment Checklist

```bash
# 1. Set up production server
- PHP 8.2+
- MySQL 8.0+
- Nginx/Apache
- Redis (for cache/sessions)
- Supervisor (for queues)

# 2. Configure environment
cp .env.example .env
php artisan key:generate
php artisan config:cache
php artisan route:cache
php artisan view:cache

# 3. Set permissions
chmod -R 755 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache

# 4. Run migrations
php artisan migrate --force

# 5. Seed initial data
php artisan db:seed --class=AdminUserSeeder
php artisan db:seed --class=DefaultCustomerFieldsSeeder

# 6. Build assets
npm run build

# 7. Start queue worker
php artisan queue:work --daemon

# 8. Set up SSL certificate
certbot --nginx -d gbsecure.com
```

---

## 8. Development Roadmap

### 8.1 MVP (Minimum Viable Product) - 4 Weeks

#### **Week 1: Foundation**
- [x] Laravel 11 project setup
- [x] Database migrations
- [x] Eloquent models with relationships
- [x] Authentication (Laravel Breeze/Fortify)
- [x] Admin layout (Tailwind + Alpine.js)
- [x] Role & permission setup (Spatie)

#### **Week 2: Core Features**
- [x] Code verification (public page)
- [x] Code generation (admin)
- [x] Product management CRUD
- [x] Customer form submission
- [x] Lucky draw functionality
- [x] DataTables integration (Livewire)

#### **Week 3: Distribution Management**
- [x] Reseller/Agent CRUD
- [x] Code assignment system
- [x] Manager role implementation
- [x] Bulk operations (tagging, assignment)
- [x] Customer database with exports

#### **Week 4: Configuration & Polish**
- [x] System setup page (logo, colors, contest)
- [x] Custom field management
- [x] Excel exports
- [x] Email integration
- [x] Testing and bug fixes
- [x] Documentation
- [x] Deployment

### 8.2 Phase 2: Enhancements - 2 Weeks

#### **Week 5: Advanced Features**
- [ ] Analytics dashboard (Chart.js)
- [ ] QR scanner (web camera)
- [ ] Advanced reporting
- [ ] Notification system
- [ ] Activity logging
- [ ] API endpoints (Sanctum)

#### **Week 6: Performance & Security**
- [ ] Caching implementation
- [ ] Queue jobs for exports
- [ ] Rate limiting
- [ ] 2FA for admin accounts
- [ ] Security audit
- [ ] Performance optimization

### 8.3 Phase 3: Future Enhancements - Ongoing

- [ ] Multi-language support (i18n)
- [ ] Mobile apps (React Native / Flutter)
- [ ] Advanced analytics (heat maps, geolocation)
- [ ] SMS notifications (Twilio)
- [ ] Blockchain verification (optional)
- [ ] Wholesale portal (reseller self-service)
- [ ] Customer loyalty program
- [ ] Social media sharing (verified products)

---

## 9. Security Enhancements

### 9.1 Authentication Security

```php
// config/fortify.php
'features' => [
    Features::registration(),
    Features::resetPasswords(),
    Features::emailVerification(),
    Features::updateProfileInformation(),
    Features::updatePasswords(),
    Features::twoFactorAuthentication([
        'confirm' => true,
        'confirmPassword' => true,
    ]),
],
```

### 9.2 Input Validation & Sanitization

```php
// Strict validation rules
class CustomerFormRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => 'required|string|max:250|regex:/^[\pL\s\-]+$/u',
            'email' => 'required|email:rfc,dns|max:250',
            'contact' => 'required|regex:/^[\+]?[0-9\s\-\(\)]+$/|max:50',
            'serial' => 'required|exists:codes,qr',
            'extra_fields.*' => 'nullable|string|max:500',
        ];
    }

    protected function prepareForValidation()
    {
        $this->merge([
            'name' => strip_tags($this->name),
            'email' => filter_var($this->email, FILTER_SANITIZE_EMAIL),
        ]);
    }
}
```

### 9.3 SQL Injection Prevention

```php
// Always use Eloquent or Query Builder (prepared statements)
// Good:
Code::where('qr', $serial)->first();

// Bad:
DB::select("SELECT * FROM codes WHERE qr = '$serial'");
```

### 9.4 XSS Protection

```blade
<!-- Blade auto-escapes output -->
<p>{{ $customerName }}</p>  <!-- Safe -->
<p>{!! $html !!}</p>         <!-- Dangerous, only for trusted content -->

<!-- Use CSP headers -->
<meta http-equiv="Content-Security-Policy"
      content="default-src 'self'; script-src 'self' https://www.google.com/recaptcha/">
```

### 9.5 CSRF Protection

```blade
<!-- Laravel automatically includes CSRF token -->
<form method="POST" action="/submit">
    @csrf
    <!-- ... -->
</form>

<!-- Livewire includes CSRF automatically -->
<livewire:customer-form />
```

### 9.6 Rate Limiting

```php
// routes/web.php
Route::middleware(['throttle:10,1'])->group(function () {
    Route::post('/verify', [HomeController::class, 'verify']);
});

// Custom rate limiter
RateLimiter::for('code-verification', function (Request $request) {
    return Limit::perMinute(10)->by($request->ip())
        ->response(function () {
            return response('Too many verification attempts', 429);
        });
});
```

---

## 10. Testing Strategy

### 10.1 Unit Tests

```php
// tests/Unit/CodeServiceTest.php
use Tests\TestCase;

class CodeServiceTest extends TestCase
{
    public function test_generates_unique_guid()
    {
        $service = new CodeGenerationService();
        $guid1 = $service->generateGuid();
        $guid2 = $service->generateGuid();

        $this->assertNotEquals($guid1, $guid2);
        $this->assertMatchesRegularExpression('/^GUID-\d{4}-\d{4}$/', $guid1);
    }

    public function test_detects_counterfeit_after_max_attempts()
    {
        $code = Code::factory()->create(['attempts' => 5]);
        $service = new CodeVerificationService();

        $result = $service->verify($code->qr, '127.0.0.1');

        $this->assertEquals('error', $result['status']);
        $this->assertEquals('fake', $code->fresh()->status);
    }
}
```

### 10.2 Feature Tests

```php
// tests/Feature/ResellerManagementTest.php
use Tests\TestCase;
use App\Models\User;

class ResellerManagementTest extends TestCase
{
    public function test_admin_can_create_reseller()
    {
        $admin = User::factory()->admin()->create();

        $response = $this->actingAs($admin)
            ->post('/resellers', [
                'fname' => 'John',
                'lname' => 'Doe',
                'email' => 'john@example.com',
                'type' => 'reseller',
            ]);

        $response->assertRedirect();
        $this->assertDatabaseHas('sellers', ['email' => 'john@example.com']);
    }

    public function test_manager_cannot_create_reseller()
    {
        $manager = User::factory()->manager()->create();

        $response = $this->actingAs($manager)
            ->post('/resellers', [
                'fname' => 'Jane',
                'lname' => 'Smith',
                'email' => 'jane@example.com',
                'type' => 'reseller',
            ]);

        $response->assertForbidden();
    }
}
```

### 10.3 Browser Tests (Laravel Dusk)

```php
// tests/Browser/CodeVerificationTest.php
use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class CodeVerificationTest extends DuskTestCase
{
    public function test_user_can_verify_code()
    {
        $code = Code::factory()->create(['status' => 'unused']);

        $this->browse(function (Browser $browser) use ($code) {
            $browser->visit('/')
                    ->type('serial', $code->qr)
                    ->press('Verify Code')
                    ->waitForText('Genuine product verified!')
                    ->assertSee('Genuine product verified!');
        });
    }
}
```

---

## Appendix A: Database ERD

```
┌───────────────┐
│     user      │
│───────────────│
│ id (PK)       │
│ firstname     │
│ lastname      │
│ email         │
│ password      │
│ role          │────┐
└───────────────┘    │
                     │ owner (FK)
┌───────────────┐    │
│    seller     │◄───┘
│───────────────│
│ userId (PK)   │
│ owner (FK)    │
│ fname         │
│ lname         │
│ email         │
│ type          │────────┐
│ status        │        │
└───────────────┘        │
        │                │
        │ userId         │
        ▼                │
┌───────────────┐        │
│  seller_qr    │        │
│───────────────│        │
│ id (PK)       │        │ Many-to-Many
│ userId (FK)   │        │
│ qr_id (FK)    │────┐   │
└───────────────┘    │   │
                     ▼   ▼
                ┌───────────────┐         ┌───────────────┐
                │     codes     │────────▶│    product    │
                │───────────────│ product │───────────────│
                │ id (PK)       │  (FK)   │ id (PK)       │
                │ guid          │         │ name          │
                │ qr            │         │ code          │
                │ status        │         │ sku           │
                │ islucky       │         │ kkm           │
                │ attempts      │         │ halal_status  │
                │ lastverified  │         │ manufacture_dt│
                │ ipv           │         │ expiry_dt     │
                └───────────────┘         └───────────────┘
                        │
                        │ serial (qr)
                        ▼
                ┌───────────────┐
                │   customer    │
                │───────────────│
                │ id (PK)       │
                │ name          │
                │ email         │
                │ contact       │
                │ serial (FK)   │
                │ extra_fields  │───────┐
                │ is_lucky_draw │       │ JSON schema
                └───────────────┘       │
                                        ▼
                                ┌───────────────────┐
                                │ customer_fields   │
                                │───────────────────│
                                │ id (PK)           │
                                │ unique_key        │
                                │ name              │
                                │ placeholder       │
                                │ required          │
                                │ visible           │
                                └───────────────────┘

┌───────────────┐
│     site      │
│───────────────│
│ id (PK)       │
│ name          │
│ logo          │
│ email         │
│ customer_     │
│  response     │
│ contest (JSON)│
└───────────────┘

┌─────────────────┐
│ password_reset  │
│─────────────────│
│ id (PK)         │
│ email           │
│ keyValue        │
│ expDate         │
└─────────────────┘
```

---

## Appendix B: File Upload Structure

```
storage/
├── app/
│   └── public/                    # Symlinked to public/storage
│       ├── logos/
│       │   └── {site_id}/
│       │       └── logo.png
│       ├── contest-images/
│       │   └── {site_id}/
│       │       ├── image1.jpg
│       │       └── image2.jpg
│       ├── qr-samples/
│       │   └── hologram_sample.jpg
│       └── exports/
│           ├── codes_export_{date}.xlsx
│           └── customers_export_{date}.xlsx
└── logs/
    └── laravel.log
```

---

## Appendix C: Key Commands

```bash
# Development
php artisan serve
npm run dev

# Code generation
php artisan make:livewire Admin/Dashboard
php artisan make:model Code -mfs  # Model + Migration + Factory + Seeder
php artisan make:request StoreCodeRequest
php artisan make:service CodeGenerationService

# Database
php artisan migrate
php artisan migrate:fresh --seed
php artisan db:seed --class=AdminUserSeeder

# Testing
php artisan test
php artisan test --filter=CodeVerificationTest

# Queue management
php artisan queue:work
php artisan queue:restart

# Cache management
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# Production optimization
php artisan config:cache
php artisan route:cache
php artisan view:cache
npm run build
```

---

## Appendix D: Backup Information

**Backup Location:** `C:\laragon\www\holobit_backup_[timestamp]`

**Contents:**
- `source/` - Complete CodeIgniter 4 codebase
- `database_full_backup.sql` - Full MySQL database dump with data

**Restore Instructions:**
```bash
# Restore database
mysql -u root holobit < database_full_backup.sql

# Restore files
cp -r source/* /c/laragon/www/holobit/
```

---

## Summary

This documentation provides a comprehensive blueprint for rebuilding the GB Secure hologram security system using the TALL stack. The new system will offer:

1. **Modern Architecture** - Laravel 11 with Livewire for reactive UIs
2. **Better UX** - Tailwind CSS with mobile-first design
3. **Enhanced Security** - Bcrypt hashing, 2FA, rate limiting, CSP headers
4. **Improved Performance** - Query optimization, caching, queue jobs
5. **Maintainability** - Clean code, type hints, comprehensive tests
6. **Scalability** - Service layer, event-driven architecture, API support
7. **Feature-Rich** - Analytics, QR scanning, multi-language, notifications

The migration strategy ensures zero downtime and data integrity, with a 4-week MVP timeline followed by iterative enhancements.

**Next Steps:**
1. Review and approve this documentation
2. Set up new Laravel 11 project
3. Begin Phase 1: Foundation (Week 1)
4. Parallel development and testing
5. Staged deployment with rollback plan

---

**Document Version:** 1.0
**Created:** November 3, 2025
**Last Updated:** November 3, 2025
**Author:** Claude (AI Assistant)
**Project:** GB Secure TALL Stack Rebuild
