@extends('layouts.app')

@section('title', 'Create Purchase Order')

@section('content')
<div class="px-4 sm:px-6 lg:px-8 py-8">
    <div class="max-w-5xl mx-auto">
        <div class="mb-6">
            <h1 class="text-2xl font-semibold text-gray-900">Create Purchase Order</h1>
            <p class="mt-2 text-sm text-gray-700">Create a new purchase order for inventory stock</p>
        </div>

        <form action="{{ route('purchase-orders.store') }}" method="POST" class="bg-white shadow rounded-lg p-6" x-data="poForm()">
            @csrf

            <div class="space-y-6">
                <!-- Supplier Selection -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label for="supplier_id" class="block text-sm font-medium text-gray-700">Supplier *</label>
                        <select name="supplier_id" id="supplier_id" required
                                class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                            <option value="">Select Supplier</option>
                            @foreach($suppliers as $supplier)
                                <option value="{{ $supplier->id }}" {{ old('supplier_id') == $supplier->id ? 'selected' : '' }}>
                                    {{ $supplier->name }}
                                </option>
                            @endforeach
                        </select>
                        @error('supplier_id')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>

                    <div>
                        <label for="order_date" class="block text-sm font-medium text-gray-700">Order Date *</label>
                        <input type="date" name="order_date" id="order_date" required value="{{ old('order_date', date('Y-m-d')) }}"
                               class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                        @error('order_date')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label for="expected_date" class="block text-sm font-medium text-gray-700">Expected Delivery Date</label>
                        <input type="date" name="expected_date" id="expected_date" value="{{ old('expected_date') }}"
                               class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                        @error('expected_date')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                </div>

                <!-- Items Section -->
                <div class="border-t border-gray-200 pt-6">
                    <div class="flex items-center justify-between mb-4">
                        <h3 class="text-lg font-medium text-gray-900">Order Items</h3>
                        <button type="button" @click="addItem()" class="px-3 py-2 bg-indigo-600 text-white rounded-md hover:bg-indigo-700 text-sm">
                            + Add Item
                        </button>
                    </div>

                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Select Product</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">SKU *</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Product Name *</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Quantity *</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Unit Price (RM) *</th>
                                    <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Subtotal</th>
                                    <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase">Action</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <template x-for="(item, index) in items" :key="index">
                                    <tr>
                                        <td class="px-4 py-3">
                                            <select @change="selectProduct(index, $event.target.value)"
                                                    class="block w-full px-2 py-1 border border-gray-300 rounded-md text-sm">
                                                <option value="">-- Select Product --</option>
                                                @foreach($products as $product)
                                                    <option value="{{ $product->id }}" data-sku="{{ $product->sku }}" data-name="{{ $product->name }}">
                                                        {{ $product->sku }} - {{ $product->name }}
                                                    </option>
                                                @endforeach
                                            </select>
                                        </td>
                                        <td class="px-4 py-3">
                                            <input type="text" :name="'items[' + index + '][sku]'" x-model="item.sku" required
                                                   class="block w-full px-2 py-1 border border-gray-300 rounded-md text-sm bg-gray-50" readonly>
                                        </td>
                                        <td class="px-4 py-3">
                                            <input type="text" :name="'items[' + index + '][product_name]'" x-model="item.product_name" required
                                                   class="block w-full px-2 py-1 border border-gray-300 rounded-md text-sm bg-gray-50" readonly>
                                        </td>
                                        <td class="px-4 py-3">
                                            <input type="number" :name="'items[' + index + '][quantity]'" x-model="item.quantity" required min="1"
                                                   @input="calculateSubtotal(index)"
                                                   class="block w-24 px-2 py-1 border border-gray-300 rounded-md text-sm">
                                        </td>
                                        <td class="px-4 py-3">
                                            <input type="number" :name="'items[' + index + '][unit_price]'" x-model="item.unit_price" required min="0" step="0.01"
                                                   @input="calculateSubtotal(index)"
                                                   class="block w-32 px-2 py-1 border border-gray-300 rounded-md text-sm">
                                        </td>
                                        <td class="px-4 py-3 text-right text-sm font-medium">
                                            RM <span x-text="item.subtotal.toFixed(2)"></span>
                                        </td>
                                        <td class="px-4 py-3 text-center">
                                            <button type="button" @click="removeItem(index)" class="text-red-600 hover:text-red-900">
                                                <svg class="h-5 w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                                                </svg>
                                            </button>
                                        </td>
                                    </tr>
                                </template>
                            </tbody>
                            <tfoot>
                                <tr>
                                    <td colspan="5" class="px-4 py-3 text-right font-bold text-gray-900">Total Amount:</td>
                                    <td class="px-4 py-3 text-right text-lg font-bold text-gray-900">
                                        RM <span x-text="totalAmount.toFixed(2)"></span>
                                    </td>
                                    <td></td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>

                    <template x-if="items.length === 0">
                        <div class="text-center py-8 text-gray-500">
                            No items added yet. Click "Add Item" to start.
                        </div>
                    </template>
                </div>

                <!-- Notes -->
                <div>
                    <label for="notes" class="block text-sm font-medium text-gray-700">Notes</label>
                    <textarea name="notes" id="notes" rows="3"
                              class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">{{ old('notes') }}</textarea>
                </div>
            </div>

            <!-- Actions -->
            <div class="mt-6 flex items-center justify-end gap-3">
                <a href="{{ route('purchase-orders.index') }}" class="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50">
                    Cancel
                </a>
                <button type="submit" class="px-4 py-2 bg-indigo-600 border border-transparent rounded-md text-sm font-medium text-white hover:bg-indigo-700">
                    Create Purchase Order
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function poForm() {
    return {
        items: [],

        init() {
            // Add one empty item by default
            this.addItem();
        },

        addItem() {
            this.items.push({
                sku: '',
                product_name: '',
                quantity: 1,
                unit_price: 0,
                subtotal: 0
            });
        },

        removeItem(index) {
            this.items.splice(index, 1);
        },

        selectProduct(index, productId) {
            if (!productId) {
                // Clear if deselected
                this.items[index].sku = '';
                this.items[index].product_name = '';
                return;
            }

            // Find the selected option to get data attributes
            const selectElement = event.target;
            const selectedOption = selectElement.options[selectElement.selectedIndex];

            // Auto-fill SKU and product name
            this.items[index].sku = selectedOption.getAttribute('data-sku');
            this.items[index].product_name = selectedOption.getAttribute('data-name');
        },

        calculateSubtotal(index) {
            const item = this.items[index];
            item.subtotal = (parseFloat(item.quantity) || 0) * (parseFloat(item.unit_price) || 0);
        },

        get totalAmount() {
            return this.items.reduce((sum, item) => sum + item.subtotal, 0);
        }
    }
}
</script>
@endsection
