@extends('layouts.app')

@section('title', 'Stock Adjustment')

@section('content')
<div class="px-4 sm:px-6 lg:px-8 py-8">
    <div class="max-w-4xl mx-auto">
        <div class="mb-6">
            <h1 class="text-2xl font-semibold text-gray-900">Manual Stock Adjustment</h1>
            <p class="mt-2 text-sm text-gray-700">Adjust inventory quantities for corrections, damages, or discrepancies</p>
        </div>

        <form action="{{ route('inventory.process-adjust') }}" method="POST" class="bg-white shadow rounded-lg p-6" x-data="adjustmentForm()">
            @csrf

            <!-- Item Selection -->
            <div class="mb-6">
                <label for="sku" class="block text-sm font-medium text-gray-700">Select Product *</label>
                <select name="sku" id="sku" required @change="loadProduct($event.target.value)"
                        class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                    <option value="">-- Select Product --</option>
                    @foreach($inventory as $item)
                        <option value="{{ $item->sku }}"
                                data-name="{{ $item->product_name }}"
                                data-current="{{ $item->quantity }}">
                            {{ $item->sku }} - {{ $item->product_name }} (Current: {{ $item->quantity }})
                        </option>
                    @endforeach
                </select>
                @error('sku')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Current Stock Display -->
            <div x-show="selectedProduct.sku" class="mb-6 p-4 bg-blue-50 border border-blue-200 rounded-md">
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <p class="text-sm text-gray-600">Product</p>
                        <p class="font-medium" x-text="selectedProduct.name"></p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Current Stock</p>
                        <p class="text-2xl font-bold text-blue-600" x-text="selectedProduct.current"></p>
                    </div>
                </div>
            </div>

            <!-- Adjustment Type -->
            <div class="mb-6">
                <label class="block text-sm font-medium text-gray-700 mb-3">Adjustment Type *</label>
                <div class="grid grid-cols-2 gap-4">
                    <label class="relative flex items-center p-4 cursor-pointer border-2 rounded-lg" :class="adjustmentType === 'add' ? 'border-green-500 bg-green-50' : 'border-gray-200 hover:border-gray-300'">
                        <input type="radio" name="adjustment_type" value="add" x-model="adjustmentType" required class="sr-only">
                        <div class="flex items-center">
                            <div class="flex-shrink-0">
                                <svg class="h-8 w-8 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"/>
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm font-medium text-gray-900">Add Stock</p>
                                <p class="text-xs text-gray-500">Found items, correction</p>
                            </div>
                        </div>
                    </label>

                    <label class="relative flex items-center p-4 cursor-pointer border-2 rounded-lg" :class="adjustmentType === 'remove' ? 'border-red-500 bg-red-50' : 'border-gray-200 hover:border-gray-300'">
                        <input type="radio" name="adjustment_type" value="remove" x-model="adjustmentType" required class="sr-only">
                        <div class="flex items-center">
                            <div class="flex-shrink-0">
                                <svg class="h-8 w-8 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4"/>
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm font-medium text-gray-900">Remove Stock</p>
                                <p class="text-xs text-gray-500">Damaged, expired, lost</p>
                            </div>
                        </div>
                    </label>
                </div>
                @error('adjustment_type')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Quantity -->
            <div class="mb-6">
                <label for="quantity" class="block text-sm font-medium text-gray-700">Adjustment Quantity *</label>
                <input type="number" name="quantity" id="quantity" required min="1" x-model="quantity"
                       class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                @error('quantity')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- New Balance Preview -->
            <div x-show="quantity > 0 && adjustmentType" class="mb-6 p-4 bg-gray-50 border border-gray-200 rounded-md">
                <div class="flex items-center justify-between">
                    <span class="text-sm text-gray-600">New Stock Balance:</span>
                    <span class="text-2xl font-bold" :class="newBalance >= 0 ? 'text-green-600' : 'text-red-600'" x-text="newBalance"></span>
                </div>
            </div>

            <!-- Reason -->
            <div class="mb-6">
                <label for="reason" class="block text-sm font-medium text-gray-700">Reason for Adjustment *</label>
                <select name="reason" id="reason" required
                        class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                    <option value="">-- Select Reason --</option>
                    <option value="Physical count correction">Physical count correction</option>
                    <option value="Damaged goods">Damaged goods</option>
                    <option value="Expired products">Expired products</option>
                    <option value="Lost/Missing items">Lost/Missing items</option>
                    <option value="Found items">Found items</option>
                    <option value="System error correction">System error correction</option>
                    <option value="Other">Other (specify in notes)</option>
                </select>
                @error('reason')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Notes -->
            <div class="mb-6">
                <label for="notes" class="block text-sm font-medium text-gray-700">Additional Notes</label>
                <textarea name="notes" id="notes" rows="3" placeholder="Add any additional details about this adjustment..."
                          class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500"></textarea>
                @error('notes')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Actions -->
            <div class="flex items-center justify-end gap-3">
                <a href="{{ route('inventory.index') }}" class="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50">
                    Cancel
                </a>
                <button type="submit" class="px-4 py-2 bg-indigo-600 border border-transparent rounded-md text-sm font-medium text-white hover:bg-indigo-700">
                    Process Adjustment
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function adjustmentForm() {
    return {
        selectedProduct: {
            sku: '',
            name: '',
            current: 0
        },
        adjustmentType: '',
        quantity: 0,

        loadProduct(sku) {
            if (!sku) {
                this.selectedProduct = { sku: '', name: '', current: 0 };
                return;
            }

            const select = document.getElementById('sku');
            const option = select.options[select.selectedIndex];

            this.selectedProduct = {
                sku: sku,
                name: option.getAttribute('data-name'),
                current: parseInt(option.getAttribute('data-current'))
            };
        },

        get newBalance() {
            let balance = this.selectedProduct.current;
            if (this.adjustmentType === 'add') {
                balance += parseInt(this.quantity || 0);
            } else if (this.adjustmentType === 'remove') {
                balance -= parseInt(this.quantity || 0);
            }
            return balance;
        }
    }
}
</script>
@endsection
