<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Symfony\Component\HttpFoundation\Response;

class OmniaSSOAuth
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Get session ID from query parameter (for initial SSO) or cookie
        $sessionId = $request->query('omnia_session') ?? $request->cookie('omnia_session');

        // No session cookie - redirect to login
        if (!$sessionId) {
            return redirect(env('AUTH_SERVICE_URL') . '/login');
        }

        // Verify session with Auth service
        try {
            $response = Http::withHeaders([
                'Cookie' => "omnia_session={$sessionId}"
            ])->get(env('AUTH_SERVICE_URL') . '/api/auth/verify');

            if (!$response->successful()) {
                // Session invalid - redirect to login
                return redirect(env('AUTH_SERVICE_URL') . '/login');
            }

            $userData = $response->json('data');

            // Check if user has production.access permission
            if (!in_array('production.access', $userData['permissions'])) {
                abort(403, 'You do not have permission to access Production module');
            }

            // Store user data in request for use in controllers
            $request->merge(['omnia_user' => $userData]);

            // If session came from query parameter, set cookie and redirect to clean URL
            if ($request->query('omnia_session')) {
                return redirect($request->url())->cookie(
                    'omnia_session',
                    $sessionId,
                    1440, // 24 hours
                    '/',
                    null, // Current domain only
                    false,
                    true,
                    false,
                    'lax'
                );
            }

        } catch (\Exception $e) {
            \Log::error('SSO Auth Error: ' . $e->getMessage());
            return redirect(env('AUTH_SERVICE_URL') . '/login');
        }

        return $next($request);
    }
}
