<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PurchaseOrderController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->input('omnia_user');

        $status = $request->input('status');
        $query = DB::table('purchase_orders')
            ->join('suppliers', 'purchase_orders.supplier_id', '=', 'suppliers.id')
            ->select('purchase_orders.*', 'suppliers.name as supplier_name');

        if ($status) {
            $query->where('purchase_orders.status', $status);
        }

        $purchaseOrders = $query->orderBy('purchase_orders.order_date', 'desc')->paginate(20);

        return view('purchase-orders.index', compact('purchaseOrders', 'user', 'status'));
    }

    public function create(Request $request)
    {
        $user = $request->input('omnia_user');

        // Get active suppliers
        $suppliers = DB::table('suppliers')
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        // Get products with SKUs from marketing database
        $products = DB::connection('marketing')
            ->table('products')
            ->select('id', 'name', 'sku')
            ->whereNotNull('sku')
            ->where('sku', '!=', '')
            ->orderBy('name')
            ->get();

        return view('purchase-orders.create', compact('user', 'suppliers', 'products'));
    }

    public function store(Request $request)
    {
        $user = $request->input('omnia_user');

        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'order_date' => 'required|date',
            'expected_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.sku' => 'required|string',
            'items.*.product_name' => 'required|string',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            // Generate PO number
            $poNumber = 'PO-' . date('Ymd') . '-' . strtoupper(substr(uniqid(), -4));

            // Calculate total amount
            $totalAmount = 0;
            foreach ($validated['items'] as $item) {
                $totalAmount += $item['quantity'] * $item['unit_price'];
            }

            // Create purchase order
            $poId = DB::table('purchase_orders')->insertGetId([
                'po_number' => $poNumber,
                'supplier_id' => $validated['supplier_id'],
                'order_date' => $validated['order_date'],
                'expected_date' => $validated['expected_date'] ?? null,
                'total_amount' => $totalAmount,
                'status' => 'draft',
                'notes' => $validated['notes'] ?? null,
                'created_by' => $user['user_id'],
                'created_by_name' => $user['name'] ?? $user['email'],
                'created_at' => now(),
                'updated_at' => now()
            ]);

            // Create purchase order items
            foreach ($validated['items'] as $item) {
                $subtotal = $item['quantity'] * $item['unit_price'];
                DB::table('purchase_order_items')->insert([
                    'purchase_order_id' => $poId,
                    'sku' => $item['sku'],
                    'product_name' => $item['product_name'],
                    'qty_ordered' => $item['quantity'],
                    'qty_received' => 0,
                    'unit_cost' => $item['unit_price'],
                    'total_cost' => $subtotal,
                    'created_at' => now(),
                    'updated_at' => now()
                ]);
            }

            DB::commit();

            Log::info("Purchase order created", [
                'po_number' => $poNumber,
                'po_id' => $poId,
                'user_id' => $user['user_id']
            ]);

            return redirect()->route('purchase-orders.show', $poId)
                ->with('success', "Purchase order {$poNumber} created successfully!");

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Failed to create purchase order", [
                'error' => $e->getMessage(),
                'user_id' => $user['user_id']
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to create purchase order. Please try again.');
        }
    }

    public function show(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        $purchaseOrder = DB::table('purchase_orders')
            ->join('suppliers', 'purchase_orders.supplier_id', '=', 'suppliers.id')
            ->select('purchase_orders.*', 'suppliers.name as supplier_name', 'suppliers.contact_person', 'suppliers.email', 'suppliers.phone')
            ->where('purchase_orders.id', $id)
            ->first();

        if (!$purchaseOrder) {
            abort(404, 'Purchase order not found');
        }

        $items = DB::table('purchase_order_items')
            ->where('purchase_order_id', $id)
            ->get();

        return view('purchase-orders.show', compact('purchaseOrder', 'items', 'user'));
    }

    public function print(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        $purchaseOrder = DB::table('purchase_orders')
            ->join('suppliers', 'purchase_orders.supplier_id', '=', 'suppliers.id')
            ->select(
                'purchase_orders.*',
                'suppliers.name as supplier_name',
                'suppliers.contact_person',
                'suppliers.phone',
                'suppliers.email',
                'suppliers.address as supplier_address'
            )
            ->where('purchase_orders.id', $id)
            ->first();

        if (!$purchaseOrder) {
            abort(404, 'Purchase order not found');
        }

        $items = DB::table('purchase_order_items')
            ->where('purchase_order_id', $id)
            ->get();

        return view('purchase-orders.print', compact('purchaseOrder', 'items', 'user'));
    }

    public function confirm(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        $purchaseOrder = DB::table('purchase_orders')
            ->where('id', $id)
            ->first();

        if (!$purchaseOrder) {
            return redirect()->route('purchase-orders.index')
                ->with('error', 'Purchase order not found');
        }

        if ($purchaseOrder->status !== 'draft') {
            return redirect()->route('purchase-orders.show', $id)
                ->with('error', 'Only draft purchase orders can be confirmed');
        }

        try {
            DB::table('purchase_orders')
                ->where('id', $id)
                ->update([
                    'status' => 'confirmed',
                    'updated_at' => now()
                ]);

            Log::info("Purchase order confirmed", [
                'po_number' => $purchaseOrder->po_number,
                'po_id' => $id,
                'user_id' => $user['user_id'],
                'user_name' => $user['name'] ?? $user['email']
            ]);

            return redirect()->route('purchase-orders.show', $id)
                ->with('success', 'Purchase order confirmed successfully!');

        } catch (\Exception $e) {
            Log::error("Failed to confirm purchase order", [
                'po_id' => $id,
                'error' => $e->getMessage()
            ]);

            return redirect()->route('purchase-orders.show', $id)
                ->with('error', 'Failed to confirm purchase order. Please try again.');
        }
    }

    public function receive(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        $purchaseOrder = DB::table('purchase_orders')
            ->join('suppliers', 'purchase_orders.supplier_id', '=', 'suppliers.id')
            ->select('purchase_orders.*', 'suppliers.name as supplier_name')
            ->where('purchase_orders.id', $id)
            ->first();

        if (!$purchaseOrder) {
            abort(404, 'Purchase order not found');
        }

        if (!in_array($purchaseOrder->status, ['confirmed', 'partial'])) {
            return redirect()->route('purchase-orders.show', $id)
                ->with('error', 'Only confirmed or partial purchase orders can be received');
        }

        $items = DB::table('purchase_order_items')
            ->where('purchase_order_id', $id)
            ->get();

        return view('purchase-orders.receive', compact('purchaseOrder', 'items', 'user'));
    }

    public function processReceive(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        $validated = $request->validate([
            'items' => 'required|array|min:1',
            'items.*.id' => 'required|exists:purchase_order_items,id',
            'items.*.quantity_received' => 'required|integer|min:0',
            'receive_notes' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            $purchaseOrder = DB::table('purchase_orders')->where('id', $id)->first();

            if (!$purchaseOrder) {
                throw new \Exception('Purchase order not found');
            }

            $totalReceived = 0;
            $totalOrdered = 0;

            // Process each item
            foreach ($validated['items'] as $itemData) {
                $quantityReceiving = $itemData['quantity_received'];

                if ($quantityReceiving <= 0) {
                    continue;
                }

                // Get current item data
                $poItem = DB::table('purchase_order_items')->where('id', $itemData['id'])->first();

                // Update purchase order item received quantity
                $newQuantityReceived = $poItem->qty_received + $quantityReceiving;
                DB::table('purchase_order_items')
                    ->where('id', $itemData['id'])
                    ->update([
                        'qty_received' => $newQuantityReceived,
                        'updated_at' => now()
                    ]);

                // Update or create inventory record
                $inventory = DB::table('inventory')->where('sku', $poItem->sku)->first();

                if ($inventory) {
                    // Update existing inventory
                    $newBalance = $inventory->quantity + $quantityReceiving;
                    DB::table('inventory')
                        ->where('sku', $poItem->sku)
                        ->update([
                            'quantity' => $newBalance,
                            'updated_at' => now()
                        ]);
                } else {
                    // Create new inventory record
                    $newBalance = $quantityReceiving;
                    DB::table('inventory')->insert([
                        'sku' => $poItem->sku,
                        'product_name' => $poItem->product_name,
                        'quantity' => $newBalance,
                        'reorder_point' => 10, // Default reorder point
                        'location' => null,
                        'created_at' => now(),
                        'updated_at' => now()
                    ]);
                }

                // Create stock movement record
                DB::table('stock_movements')->insert([
                    'sku' => $poItem->sku,
                    'type' => 'in',
                    'quantity' => $quantityReceiving,
                    'balance_after' => $newBalance,
                    'reference_type' => 'purchase_order',
                    'reference_number' => $purchaseOrder->po_number,
                    'notes' => $validated['receive_notes'] ?? "Received from PO {$purchaseOrder->po_number}",
                    'performed_by' => $user['user_id'],
                    'performed_by_name' => $user['name'] ?? $user['email'],
                    'performed_at' => now(),
                    'created_at' => now(),
                    'updated_at' => now()
                ]);

                $totalReceived += $newQuantityReceived;
                $totalOrdered += $poItem->qty_ordered;
            }

            // Update PO status based on received quantities
            $allItems = DB::table('purchase_order_items')
                ->where('purchase_order_id', $id)
                ->get();

            $allFullyReceived = true;
            $anyReceived = false;

            foreach ($allItems as $item) {
                if ($item->qty_received < $item->qty_ordered) {
                    $allFullyReceived = false;
                }
                if ($item->qty_received > 0) {
                    $anyReceived = true;
                }
            }

            $newStatus = $allFullyReceived ? 'received' : ($anyReceived ? 'partial' : 'confirmed');

            DB::table('purchase_orders')
                ->where('id', $id)
                ->update([
                    'status' => $newStatus,
                    'received_by' => $user['user_id'],
                    'received_by_name' => $user['name'] ?? $user['email'],
                    'received_at' => now(),
                    'updated_at' => now()
                ]);

            DB::commit();

            Log::info("Purchase order items received", [
                'po_number' => $purchaseOrder->po_number,
                'po_id' => $id,
                'new_status' => $newStatus,
                'user_id' => $user['user_id']
            ]);

            return redirect()->route('purchase-orders.show', $id)
                ->with('success', 'Stock received successfully! Purchase order status updated to ' . ucfirst($newStatus));

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Failed to receive purchase order items", [
                'po_id' => $id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to receive items. Please try again.');
        }
    }
}
