<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class OrderController extends Controller
{
    /**
     * Display pending orders (status: printed)
     */
    public function index(Request $request)
    {
        $user = $request->input('omnia_user');

        // Get current status tab (default: pending = printed status)
        $currentStatus = $request->get('status', 'pending');

        // Map status tabs to actual database status values
        $statusMap = [
            'pending' => 'printed',
            'packed' => 'packed',
            'completed' => 'completed'
        ];

        $dbStatus = $statusMap[$currentStatus] ?? 'printed';

        // Get per-page setting (default: 25)
        $perPage = $request->get('per_page', 25);

        // Get search and filter parameters
        $search = $request->get('search');
        $dateFrom = $request->get('date_from');
        $dateTo = $request->get('date_to');

        // Build query with filters
        $query = DB::connection('marketing')
            ->table('orders')
            ->where('status', $dbStatus);

        // Apply search filter
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhere('tracking_number', 'like', "%{$search}%")
                  ->orWhere('billing', 'like', "%{$search}%") // Search in JSON billing field
                  ->orWhere('shipping', 'like', "%{$search}%"); // Search in JSON shipping field
            });
        }

        // Apply date range filter
        if ($dateFrom) {
            $query->whereDate('created_at', '>=', $dateFrom);
        }
        if ($dateTo) {
            $query->whereDate('created_at', '<=', $dateTo);
        }

        // Get orders with pagination
        $orders = $query->orderBy('created_at', 'desc')
            ->paginate($perPage)
            ->appends($request->except('page')); // Preserve filters in pagination links

        // Parse JSON fields for each order
        foreach ($orders as $order) {
            // Parse billing data
            $billing = json_decode($order->billing, true);
            $order->customer_name = isset($billing['first_name']) && isset($billing['last_name'])
                ? $billing['first_name'] . ' ' . $billing['last_name']
                : 'N/A';
            $order->customer_phone = $billing['phone'] ?? 'N/A';
            $order->customer_email = $billing['email'] ?? 'N/A';

            // Parse shipping data for address
            $shipping = json_decode($order->shipping, true);
            $addressParts = array_filter([
                $shipping['address_1'] ?? '',
                $shipping['address_2'] ?? '',
                $shipping['city'] ?? '',
                $shipping['postcode'] ?? '',
                $shipping['state'] ?? '',
            ]);
            $order->delivery_address = implode(', ', $addressParts) ?: 'N/A';

            // Parse line items
            $lineItems = json_decode($order->line_items, true);
            $order->items = collect($lineItems)->map(function($item) {
                return (object) [
                    'product_name' => $item['name'] ?? 'Unknown',
                    'product_sku' => $item['sku'] ?? 'N/A',
                    'quantity' => $item['quantity'] ?? 0,
                    'unit_price' => $item['price'] ?? 0,
                ];
            });

            // Use 'total' field instead of 'total_amount'
            $order->total_amount = $order->total;
        }

        // Get status counts for tabs
        $statusCounts = [
            'pending' => DB::connection('marketing')->table('orders')->where('status', 'printed')->count(),
            'packed' => DB::connection('marketing')->table('orders')->where('status', 'packed')->count(),
            'completed' => DB::connection('marketing')->table('orders')->where('status', 'completed')->count(),
        ];

        return view('orders.index', [
            'orders' => $orders,
            'user' => $user,
            'currentStatus' => $currentStatus,
            'statusCounts' => $statusCounts
        ]);
    }

    /**
     * Show single order details
     */
    public function show(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        $order = DB::connection('marketing')
            ->table('orders')
            ->where('id', $id)
            ->first();

        if (!$order) {
            abort(404, 'Order not found');
        }

        // Parse billing and shipping data
        $billing = json_decode($order->billing, true);
        $shipping = json_decode($order->shipping, true);

        $order->customer_name = isset($billing['first_name']) && isset($billing['last_name'])
            ? $billing['first_name'] . ' ' . $billing['last_name']
            : 'N/A';
        $order->customer_phone = $billing['phone'] ?? 'N/A';
        $order->customer_email = $billing['email'] ?? 'N/A';

        // Build delivery address
        $addressParts = array_filter([
            $shipping['address_1'] ?? '',
            $shipping['address_2'] ?? '',
            $shipping['city'] ?? '',
            $shipping['postcode'] ?? '',
            $shipping['state'] ?? '',
        ]);
        $order->delivery_address = implode(', ', $addressParts) ?: 'N/A';

        // Parse line items
        $lineItems = json_decode($order->line_items, true);
        $order->items = collect($lineItems)->map(function($item) {
            return (object) [
                'product_name' => $item['name'] ?? 'Unknown',
                'product_sku' => $item['sku'] ?? 'N/A',
                'quantity' => $item['quantity'] ?? 0,
                'unit_price' => $item['price'] ?? 0,
            ];
        });

        // Use 'total' field instead of 'total_amount'
        $order->total_amount = $order->total;

        return view('orders.show', [
            'order' => $order,
            'user' => $user
        ]);
    }

    /**
     * Mark order as packed
     */
    public function markAsPacked(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        // Get order to verify it exists and is in correct status
        $order = DB::connection('marketing')
            ->table('orders')
            ->where('id', $id)
            ->first();

        if (!$order) {
            return redirect()->route('dashboard')->with('error', 'Order not found');
        }

        if ($order->status !== 'printed') {
            return redirect()->route('dashboard')->with('error', 'Order cannot be packed. Current status: ' . $order->status);
        }

        try {
            // Deduct stock for order items
            $stockResult = $this->deductStockForOrder($order, $user);

            if (!$stockResult['success']) {
                return redirect()->route('dashboard')->with('error', $stockResult['message']);
            }

            // Update order status in Marketing database
            DB::connection('marketing')
                ->table('orders')
                ->where('id', $id)
                ->update([
                    'status' => 'packed',
                    'packed_by' => $user['user_id'],
                    'packed_by_name' => $user['name'] ?? $user['email'],
                    'packed_at' => now(),
                    'updated_at' => now()
                ]);

            // Log the action
            Log::info("Order #{$id} marked as packed", [
                'user_id' => $user['user_id'],
                'user_email' => $user['email'],
                'order_number' => $order->order_number,
                'stock_deducted' => $stockResult['items_processed']
            ]);

            $successMessage = "Order #{$order->order_number} has been marked as packed!";
            if (!empty($stockResult['warnings'])) {
                $successMessage .= " Note: " . implode('; ', $stockResult['warnings']);
            }

            return redirect()->route('dashboard', ['status' => 'pending'])->with('success', $successMessage);

        } catch (\Exception $e) {
            Log::error("Failed to mark order #{$id} as packed", [
                'error' => $e->getMessage(),
                'user_id' => $user['user_id']
            ]);

            return redirect()->route('dashboard')->with('error', 'Failed to update order status. Please try again.');
        }
    }

    /**
     * Mark order as completed
     */
    public function markAsCompleted(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        // Get order to verify it exists and is in correct status
        $order = DB::connection('marketing')
            ->table('orders')
            ->where('id', $id)
            ->first();

        if (!$order) {
            return redirect()->route('dashboard')->with('error', 'Order not found');
        }

        if ($order->status !== 'packed') {
            return redirect()->route('dashboard')->with('error', 'Order cannot be completed. Current status: ' . $order->status);
        }

        try {
            // Update order status in Marketing database
            DB::connection('marketing')
                ->table('orders')
                ->where('id', $id)
                ->update([
                    'status' => 'completed',
                    'completed_by' => $user['user_id'],
                    'completed_by_name' => $user['name'] ?? $user['email'],
                    'completed_at' => now(),
                    'updated_at' => now()
                ]);

            // Log the action
            Log::info("Order #{$id} marked as completed", [
                'user_id' => $user['user_id'],
                'user_email' => $user['email'],
                'order_number' => $order->order_number
            ]);

            return redirect()->route('dashboard', ['status' => 'packed'])->with('success', "Order #{$order->order_number} has been marked as completed!");

        } catch (\Exception $e) {
            Log::error("Failed to mark order #{$id} as completed", [
                'error' => $e->getMessage(),
                'user_id' => $user['user_id']
            ]);

            return redirect()->route('dashboard')->with('error', 'Failed to update order status. Please try again.');
        }
    }

    /**
     * Bulk complete multiple packed orders
     */
    public function bulkComplete(Request $request)
    {
        $user = $request->input('omnia_user');

        // Validate request
        $request->validate([
            'order_ids' => 'required|array|min:1',
            'order_ids.*' => 'required|integer'
        ]);

        $orderIds = $request->input('order_ids');
        $successCount = 0;
        $errorCount = 0;
        $errors = [];

        try {
            foreach ($orderIds as $orderId) {
                // Get order to verify it exists and is in correct status
                $order = DB::connection('marketing')
                    ->table('orders')
                    ->where('id', $orderId)
                    ->first();

                if (!$order) {
                    $errors[] = "Order ID {$orderId} not found";
                    $errorCount++;
                    continue;
                }

                if ($order->status !== 'packed') {
                    $errors[] = "Order #{$order->order_number} is not in packed status (current: {$order->status})";
                    $errorCount++;
                    continue;
                }

                // Update order to completed status
                DB::connection('marketing')
                    ->table('orders')
                    ->where('id', $orderId)
                    ->update([
                        'status' => 'completed',
                        'completed_by' => $user['user_id'],
                        'completed_by_name' => $user['name'] ?? $user['email'],
                        'completed_at' => now(),
                        'updated_at' => now()
                    ]);

                $successCount++;

                // Log the action
                Log::info("Order #{$orderId} bulk completed", [
                    'user_id' => $user['user_id'],
                    'user_email' => $user['email'],
                    'order_number' => $order->order_number
                ]);
            }

            $message = "Successfully completed {$successCount} order(s)";
            if ($errorCount > 0) {
                $message .= " ({$errorCount} failed)";
            }

            return redirect()
                ->route('dashboard', ['status' => 'packed'])
                ->with('success', $message)
                ->with('bulk_errors', $errors);

        } catch (\Exception $e) {
            Log::error("Failed to bulk complete orders", [
                'error' => $e->getMessage(),
                'user_id' => $user['user_id']
            ]);

            return redirect()
                ->route('dashboard', ['status' => 'packed'])
                ->with('error', 'Failed to complete orders. Please try again.');
        }
    }

    /**
     * Undo packed order back to pending (printed status)
     * Only accessible by production managers and admins
     */
    public function undoPacked(Request $request, $id)
    {
        $user = $request->input('omnia_user');

        // Check role authorization
        $userRoles = $user['roles'] ?? [];
        if (!in_array('admin', $userRoles) && !in_array('production-manager', $userRoles)) {
            return response()->json([
                'success' => false,
                'message' => 'Only Production Managers and Admins can undo packed orders'
            ], 403);
        }

        // Validate reason
        $request->validate([
            'reason' => 'required|string|min:5|max:500'
        ]);

        // Get order
        $order = DB::connection('marketing')
            ->table('orders')
            ->where('id', $id)
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        // Verify order is in packed status
        if ($order->status !== 'packed') {
            return response()->json([
                'success' => false,
                'message' => 'Order is not in packed status. Current status: ' . $order->status
            ], 400);
        }

        try {
            // Restore stock for order items
            $restoreResult = $this->restoreStockForOrder($order, $user, $request->input('reason'));

            if (!$restoreResult['success']) {
                return response()->json([
                    'success' => false,
                    'message' => $restoreResult['message']
                ], 500);
            }

            // Revert to printed (pending) status and record undo audit fields
            DB::connection('marketing')
                ->table('orders')
                ->where('id', $id)
                ->update([
                    'status' => 'printed',
                    'unpacked_by' => $user['user_id'],
                    'unpacked_by_name' => $user['name'] ?? $user['email'],
                    'unpacked_at' => now(),
                    'unpacked_reason' => $request->input('reason'),
                    'updated_at' => now()
                    // Keep packed_by, packed_by_name, packed_at for history
                ]);

            // Log the undo action
            Log::info("Order unpacked by manager", [
                'order_id' => $id,
                'order_number' => $order->order_number,
                'unpacked_by' => $user['user_id'],
                'unpacked_by_name' => $user['name'] ?? $user['email'],
                'reason' => $request->input('reason'),
                'original_packed_by' => $order->packed_by,
                'original_packed_at' => $order->packed_at,
                'stock_restored' => $restoreResult['items_processed']
            ]);

            $message = "Order #{$order->order_number} has been reverted to Pending status";
            if (!empty($restoreResult['warnings'])) {
                $message .= ". Note: " . implode('; ', $restoreResult['warnings']);
            }

            return response()->json([
                'success' => true,
                'message' => $message
            ]);

        } catch (\Exception $e) {
            Log::error("Failed to undo packed order", [
                'order_id' => $id,
                'error' => $e->getMessage(),
                'user_id' => $user['user_id']
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to undo order. Please try again.'
            ], 500);
        }
    }

    /**
     * Scan tracking number to mark order as packed
     */
    public function scanTrackingNumber(Request $request)
    {
        $user = $request->input('omnia_user');
        $trackingNumber = trim($request->input('tracking_number'));

        // Validate tracking number
        if (empty($trackingNumber)) {
            return response()->json([
                'success' => false,
                'message' => 'Please enter a tracking number'
            ], 400);
        }

        try {
            // Find order by tracking number
            $order = DB::connection('marketing')
                ->table('orders')
                ->where('tracking_number', $trackingNumber)
                ->first();

            if (!$order) {
                return response()->json([
                    'success' => false,
                    'message' => "No order found with tracking number: {$trackingNumber}"
                ], 404);
            }

            // Check if already packed or completed
            if ($order->status === 'packed') {
                return response()->json([
                    'success' => false,
                    'message' => "Order #{$order->order_number} is already packed!",
                    'order_number' => $order->order_number,
                    'status' => 'already_packed'
                ], 400);
            }

            if ($order->status === 'completed') {
                return response()->json([
                    'success' => false,
                    'message' => "Order #{$order->order_number} is already completed!",
                    'order_number' => $order->order_number,
                    'status' => 'already_completed'
                ], 400);
            }

            // Verify order is in printed status
            if ($order->status !== 'printed') {
                return response()->json([
                    'success' => false,
                    'message' => "Order #{$order->order_number} cannot be packed. Current status: {$order->status}",
                    'order_number' => $order->order_number,
                    'current_status' => $order->status
                ], 400);
            }

            // Deduct stock for order items
            $stockResult = $this->deductStockForOrder($order, $user);

            if (!$stockResult['success']) {
                return response()->json([
                    'success' => false,
                    'message' => $stockResult['message'],
                    'order_number' => $order->order_number
                ], 400);
            }

            // Update order to packed status
            DB::connection('marketing')
                ->table('orders')
                ->where('id', $order->id)
                ->update([
                    'status' => 'packed',
                    'packed_by' => $user['user_id'],
                    'packed_by_name' => $user['name'] ?? $user['email'],
                    'packed_at' => now(),
                    'updated_at' => now()
                ]);

            // Log the scan action
            Log::info("Order scanned and marked as packed", [
                'tracking_number' => $trackingNumber,
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'user_id' => $user['user_id'],
                'user_email' => $user['email'],
                'stock_deducted' => $stockResult['items_processed']
            ]);

            $message = "Order #{$order->order_number} marked as packed successfully!";
            if (!empty($stockResult['warnings'])) {
                $message .= " Warning: " . implode('; ', $stockResult['warnings']);
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'order_number' => $order->order_number,
                'order_id' => $order->id,
                'tracking_number' => $trackingNumber,
                'packed_by' => $user['name'] ?? $user['email'],
                'packed_at' => now()->format('Y-m-d H:i:s'),
                'stock_warnings' => $stockResult['warnings'] ?? []
            ]);

        } catch (\Exception $e) {
            Log::error("Failed to scan tracking number", [
                'tracking_number' => $trackingNumber,
                'error' => $e->getMessage(),
                'user_id' => $user['user_id']
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to process scan. Please try again.'
            ], 500);
        }
    }

    /**
     * Get pending orders count (for dashboard)
     */
    public function getPendingCount()
    {
        $count = DB::connection('marketing')
            ->table('orders')
            ->where('status', 'printed')
            ->count();

        return response()->json([
            'success' => true,
            'count' => $count
        ]);
    }

    /**
     * Performance Dashboard - Staff productivity metrics
     */
    public function performance(Request $request)
    {
        $user = $request->input('omnia_user');

        // Check if user is manager or admin
        $userRoles = $user['roles'] ?? [];
        $isManagerOrAdmin = in_array('admin', $userRoles) || in_array('production-manager', $userRoles);

        // Get date range (default: last 30 days)
        $dateFrom = $request->get('date_from', now()->subDays(30)->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));
        $staffFilter = $request->get('staff_id');

        // Query for packed orders performance
        $packedQuery = DB::connection('marketing')
            ->table('orders')
            ->select(
                'packed_by',
                'packed_by_name',
                DB::raw('COUNT(*) as total_packed'),
                DB::raw('DATE(packed_at) as date')
            )
            ->whereNotNull('packed_by')
            ->whereNotNull('packed_at')
            ->whereDate('packed_at', '>=', $dateFrom)
            ->whereDate('packed_at', '<=', $dateTo);

        if ($staffFilter) {
            $packedQuery->where('packed_by', $staffFilter);
        }

        $packedStats = $packedQuery
            ->groupBy('packed_by', 'packed_by_name', DB::raw('DATE(packed_at)'))
            ->get();

        // Query for completed orders performance
        $completedQuery = DB::connection('marketing')
            ->table('orders')
            ->select(
                'completed_by',
                'completed_by_name',
                DB::raw('COUNT(*) as total_completed'),
                DB::raw('DATE(completed_at) as date')
            )
            ->whereNotNull('completed_by')
            ->whereNotNull('completed_at')
            ->whereDate('completed_at', '>=', $dateFrom)
            ->whereDate('completed_at', '<=', $dateTo);

        if ($staffFilter) {
            $completedQuery->where('completed_by', $staffFilter);
        }

        $completedStats = $completedQuery
            ->groupBy('completed_by', 'completed_by_name', DB::raw('DATE(completed_at)'))
            ->get();

        // Aggregate by staff member
        $staffPerformance = [];

        // Process packed stats
        foreach ($packedStats as $stat) {
            $staffId = $stat->packed_by;
            if (!isset($staffPerformance[$staffId])) {
                $staffPerformance[$staffId] = [
                    'staff_id' => $staffId,
                    'staff_name' => $stat->packed_by_name,
                    'total_packed' => 0,
                    'total_completed' => 0,
                    'daily_breakdown' => []
                ];
            }
            $staffPerformance[$staffId]['total_packed'] += $stat->total_packed;

            if (!isset($staffPerformance[$staffId]['daily_breakdown'][$stat->date])) {
                $staffPerformance[$staffId]['daily_breakdown'][$stat->date] = [
                    'date' => $stat->date,
                    'packed' => 0,
                    'completed' => 0
                ];
            }
            $staffPerformance[$staffId]['daily_breakdown'][$stat->date]['packed'] += $stat->total_packed;
        }

        // Process completed stats
        foreach ($completedStats as $stat) {
            $staffId = $stat->completed_by;
            if (!isset($staffPerformance[$staffId])) {
                $staffPerformance[$staffId] = [
                    'staff_id' => $staffId,
                    'staff_name' => $stat->completed_by_name,
                    'total_packed' => 0,
                    'total_completed' => 0,
                    'daily_breakdown' => []
                ];
            }
            $staffPerformance[$staffId]['total_completed'] += $stat->total_completed;

            if (!isset($staffPerformance[$staffId]['daily_breakdown'][$stat->date])) {
                $staffPerformance[$staffId]['daily_breakdown'][$stat->date] = [
                    'date' => $stat->date,
                    'packed' => 0,
                    'completed' => 0
                ];
            }
            $staffPerformance[$staffId]['daily_breakdown'][$stat->date]['completed'] += $stat->total_completed;
        }

        // Convert to collection and sort by total performance
        $staffPerformance = collect($staffPerformance)->sortByDesc(function($staff) {
            return $staff['total_packed'] + $staff['total_completed'];
        })->values();

        // Get list of all staff for filter dropdown
        $allStaff = DB::connection('marketing')
            ->table('orders')
            ->select('packed_by as staff_id', 'packed_by_name as staff_name')
            ->whereNotNull('packed_by')
            ->groupBy('packed_by', 'packed_by_name')
            ->union(
                DB::connection('marketing')
                    ->table('orders')
                    ->select('completed_by as staff_id', 'completed_by_name as staff_name')
                    ->whereNotNull('completed_by')
                    ->groupBy('completed_by', 'completed_by_name')
            )
            ->get()
            ->unique('staff_id')
            ->sortBy('staff_name')
            ->values();

        return view('performance.index', [
            'user' => $user,
            'staffPerformance' => $staffPerformance,
            'allStaff' => $allStaff,
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'staffFilter' => $staffFilter,
            'isManagerOrAdmin' => $isManagerOrAdmin
        ]);
    }

    /**
     * Print individual staff performance report
     */
    public function printStaffReport(Request $request, $staffId)
    {
        $user = $request->input('omnia_user');

        // Check if user is manager or admin
        $userRoles = $user['roles'] ?? [];
        $isManagerOrAdmin = in_array('admin', $userRoles) || in_array('production-manager', $userRoles);

        if (!$isManagerOrAdmin) {
            abort(403, 'Only managers and admins can print staff reports');
        }

        // Get date range from request or default to last 30 days
        $dateFrom = $request->get('date_from', now()->subDays(30)->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        // Query for packed orders by this staff
        $packedOrders = DB::connection('marketing')
            ->table('orders')
            ->select(
                DB::raw('COUNT(*) as total'),
                DB::raw('DATE(packed_at) as date'),
                'packed_by_name'
            )
            ->where('packed_by', $staffId)
            ->whereNotNull('packed_at')
            ->whereDate('packed_at', '>=', $dateFrom)
            ->whereDate('packed_at', '<=', $dateTo)
            ->groupBy(DB::raw('DATE(packed_at)'), 'packed_by_name')
            ->get();

        // Query for completed orders by this staff
        $completedOrders = DB::connection('marketing')
            ->table('orders')
            ->select(
                DB::raw('COUNT(*) as total'),
                DB::raw('DATE(completed_at) as date'),
                'completed_by_name'
            )
            ->where('completed_by', $staffId)
            ->whereNotNull('completed_at')
            ->whereDate('completed_at', '>=', $dateFrom)
            ->whereDate('completed_at', '<=', $dateTo)
            ->groupBy(DB::raw('DATE(completed_at)'), 'completed_by_name')
            ->get();

        // Get staff name
        $staffName = $packedOrders->first()->packed_by_name ??
                     $completedOrders->first()->completed_by_name ??
                     'Unknown Staff';

        // Calculate totals
        $totalPacked = $packedOrders->sum('total');
        $totalCompleted = $completedOrders->sum('total');
        $totalOrders = $totalPacked + $totalCompleted;

        // Calculate daily average
        $dayCount = \Carbon\Carbon::parse($dateFrom)->diffInDays(\Carbon\Carbon::parse($dateTo)) + 1;
        $dailyAverage = $dayCount > 0 ? round($totalOrders / $dayCount, 1) : 0;

        // Merge daily breakdown
        $dailyBreakdown = [];
        foreach ($packedOrders as $order) {
            if (!isset($dailyBreakdown[$order->date])) {
                $dailyBreakdown[$order->date] = ['date' => $order->date, 'packed' => 0, 'completed' => 0];
            }
            $dailyBreakdown[$order->date]['packed'] = $order->total;
        }
        foreach ($completedOrders as $order) {
            if (!isset($dailyBreakdown[$order->date])) {
                $dailyBreakdown[$order->date] = ['date' => $order->date, 'packed' => 0, 'completed' => 0];
            }
            $dailyBreakdown[$order->date]['completed'] = $order->total;
        }

        // Sort by date descending
        $dailyBreakdown = collect($dailyBreakdown)->sortByDesc('date')->values();

        return view('performance.print', [
            'staffId' => $staffId,
            'staffName' => $staffName,
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'totalPacked' => $totalPacked,
            'totalCompleted' => $totalCompleted,
            'totalOrders' => $totalOrders,
            'dailyAverage' => $dailyAverage,
            'dailyBreakdown' => $dailyBreakdown,
            'dayCount' => $dayCount,
            'preparedBy' => $user['name'] ?? $user['email'],
            'preparedDate' => now()->format('Y-m-d')
        ]);
    }

    /**
     * Deduct stock for order items when order is packed
     *
     * @param object $order The order object from marketing database
     * @param array $user The user performing the action
     * @return array Result with success status, message, warnings, and items processed
     */
    private function deductStockForOrder($order, $user)
    {
        $warnings = [];
        $itemsProcessed = 0;

        try {
            // Parse line items from order
            $lineItems = json_decode($order->line_items, true);

            if (empty($lineItems)) {
                return [
                    'success' => true,
                    'message' => 'No items to process',
                    'warnings' => ['Order has no line items'],
                    'items_processed' => 0
                ];
            }

            // Process each line item
            foreach ($lineItems as $item) {
                $sku = $item['sku'] ?? null;
                $quantity = $item['quantity'] ?? 0;
                $productName = $item['name'] ?? 'Unknown Product';

                // Skip items without SKU or with zero quantity
                if (empty($sku) || $quantity <= 0) {
                    if (empty($sku)) {
                        $warnings[] = "{$productName} has no SKU - stock not deducted";
                    }
                    continue;
                }

                // Find inventory item by SKU
                $inventoryItem = DB::table('inventory')
                    ->where('sku', $sku)
                    ->first();

                if (!$inventoryItem) {
                    // SKU not found in inventory - log warning but don't fail
                    $warnings[] = "SKU {$sku} not found in inventory - stock not deducted";
                    Log::warning("SKU not found in inventory during order packing", [
                        'order_number' => $order->order_number,
                        'sku' => $sku,
                        'product_name' => $productName
                    ]);
                    continue;
                }

                // Check if sufficient stock is available
                if ($inventoryItem->quantity < $quantity) {
                    // Insufficient stock - log warning but allow packing
                    $warnings[] = "Insufficient stock for {$sku} (Available: {$inventoryItem->quantity}, Needed: {$quantity})";
                    Log::warning("Insufficient stock during order packing", [
                        'order_number' => $order->order_number,
                        'sku' => $sku,
                        'available' => $inventoryItem->quantity,
                        'needed' => $quantity
                    ]);
                }

                // Calculate new balance
                $newBalance = $inventoryItem->quantity - $quantity;

                // Update inventory quantity
                DB::table('inventory')
                    ->where('sku', $sku)
                    ->update([
                        'quantity' => $newBalance,
                        'updated_at' => now()
                    ]);

                // Create stock movement record
                DB::table('stock_movements')->insert([
                    'sku' => $sku,
                    'type' => 'out',
                    'quantity' => -$quantity, // Negative for stock out
                    'balance_after' => $newBalance,
                    'reference_type' => 'order',
                    'reference_number' => $order->order_number,
                    'notes' => "Stock deducted for order #{$order->order_number}",
                    'performed_by' => $user['user_id'],
                    'performed_by_name' => $user['name'] ?? $user['email'],
                    'performed_at' => now(),
                    'created_at' => now(),
                    'updated_at' => now()
                ]);

                $itemsProcessed++;

                Log::info("Stock deducted for order item", [
                    'order_number' => $order->order_number,
                    'sku' => $sku,
                    'quantity_deducted' => $quantity,
                    'new_balance' => $newBalance
                ]);
            }

            return [
                'success' => true,
                'message' => "Stock deducted successfully",
                'warnings' => $warnings,
                'items_processed' => $itemsProcessed
            ];

        } catch (\Exception $e) {
            Log::error("Failed to deduct stock for order", [
                'order_number' => $order->order_number,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => "Failed to deduct stock: " . $e->getMessage(),
                'warnings' => $warnings,
                'items_processed' => $itemsProcessed
            ];
        }
    }

    /**
     * Restore stock for order items when order is unpacked (reverted to pending)
     *
     * @param object $order The order object from marketing database
     * @param array $user The user performing the action
     * @param string $reason The reason for unpacking
     * @return array Result with success status, message, warnings, and items processed
     */
    private function restoreStockForOrder($order, $user, $reason)
    {
        $warnings = [];
        $itemsProcessed = 0;

        try {
            // Parse line items from order
            $lineItems = json_decode($order->line_items, true);

            if (empty($lineItems)) {
                return [
                    'success' => true,
                    'message' => 'No items to restore',
                    'warnings' => ['Order has no line items'],
                    'items_processed' => 0
                ];
            }

            // Process each line item
            foreach ($lineItems as $item) {
                $sku = $item['sku'] ?? null;
                $quantity = $item['quantity'] ?? 0;
                $productName = $item['name'] ?? 'Unknown Product';

                // Skip items without SKU or with zero quantity
                if (empty($sku) || $quantity <= 0) {
                    if (empty($sku)) {
                        $warnings[] = "{$productName} has no SKU - stock not restored";
                    }
                    continue;
                }

                // Find inventory item by SKU
                $inventoryItem = DB::table('inventory')
                    ->where('sku', $sku)
                    ->first();

                if (!$inventoryItem) {
                    // SKU not found - log warning but don't fail
                    $warnings[] = "SKU {$sku} not found in inventory - stock not restored";
                    Log::warning("SKU not found in inventory during order unpacking", [
                        'order_number' => $order->order_number,
                        'sku' => $sku,
                        'product_name' => $productName
                    ]);
                    continue;
                }

                // Calculate new balance (restore stock)
                $newBalance = $inventoryItem->quantity + $quantity;

                // Update inventory quantity
                DB::table('inventory')
                    ->where('sku', $sku)
                    ->update([
                        'quantity' => $newBalance,
                        'updated_at' => now()
                    ]);

                // Create stock movement record for the restoration
                DB::table('stock_movements')->insert([
                    'sku' => $sku,
                    'type' => 'adjustment',
                    'quantity' => $quantity, // Positive for stock in
                    'balance_after' => $newBalance,
                    'reference_type' => 'order_unpack',
                    'reference_number' => $order->order_number,
                    'notes' => "Stock restored - Order #{$order->order_number} unpacked. Reason: {$reason}",
                    'performed_by' => $user['user_id'],
                    'performed_by_name' => $user['name'] ?? $user['email'],
                    'performed_at' => now(),
                    'created_at' => now(),
                    'updated_at' => now()
                ]);

                $itemsProcessed++;

                Log::info("Stock restored for unpacked order item", [
                    'order_number' => $order->order_number,
                    'sku' => $sku,
                    'quantity_restored' => $quantity,
                    'new_balance' => $newBalance,
                    'reason' => $reason
                ]);
            }

            return [
                'success' => true,
                'message' => "Stock restored successfully",
                'warnings' => $warnings,
                'items_processed' => $itemsProcessed
            ];

        } catch (\Exception $e) {
            Log::error("Failed to restore stock for order", [
                'order_number' => $order->order_number,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => "Failed to restore stock: " . $e->getMessage(),
                'warnings' => $warnings,
                'items_processed' => $itemsProcessed
            ];
        }
    }
}
