<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class InventoryController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->input('omnia_user');

        $search = $request->input('search');
        $query = DB::table('inventory');

        if ($search) {
            $query->where('sku', 'like', "%{$search}%")
                  ->orWhere('product_name', 'like', "%{$search}%");
        }

        $inventory = $query->orderBy('product_name')->paginate(25);

        // Count low stock items
        $lowStockCount = DB::table('inventory')
            ->whereColumn('quantity', '<', 'reorder_point')
            ->count();

        return view('inventory.index', compact('inventory', 'user', 'lowStockCount', 'search'));
    }

    public function movements()
    {
        $user = request()->input('omnia_user');

        $movements = DB::table('stock_movements')
            ->orderBy('performed_at', 'desc')
            ->paginate(50);

        return view('inventory.movements', compact('movements', 'user'));
    }

    public function lowStock()
    {
        $user = request()->input('omnia_user');

        $inventory = DB::table('inventory')
            ->whereColumn('quantity', '<', 'reorder_point')
            ->orderBy('quantity')
            ->paginate(25);

        return view('inventory.low-stock', compact('inventory', 'user'));
    }

    public function adjust(Request $request)
    {
        $user = $request->input('omnia_user');

        $inventory = DB::table('inventory')
            ->orderBy('product_name')
            ->get();

        return view('inventory.adjust', compact('inventory', 'user'));
    }

    public function processAdjustment(Request $request)
    {
        $user = $request->input('omnia_user');

        $validated = $request->validate([
            'sku' => 'required|exists:inventory,sku',
            'adjustment_type' => 'required|in:add,remove',
            'quantity' => 'required|integer|min:1',
            'reason' => 'required|string',
            'notes' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            // Get current inventory
            $inventory = DB::table('inventory')->where('sku', $validated['sku'])->first();

            if (!$inventory) {
                return redirect()->back()->with('error', 'Product not found in inventory');
            }

            // Calculate new balance
            $adjustmentQty = $validated['adjustment_type'] === 'add' ? $validated['quantity'] : -$validated['quantity'];
            $newBalance = $inventory->quantity + $adjustmentQty;

            if ($newBalance < 0) {
                return redirect()->back()->with('error', 'Cannot remove more stock than available');
            }

            // Update inventory
            DB::table('inventory')
                ->where('sku', $validated['sku'])
                ->update([
                    'quantity' => $newBalance,
                    'updated_at' => now()
                ]);

            // Create stock movement record
            $notes = $validated['reason'];
            if (!empty($validated['notes'])) {
                $notes .= ' - ' . $validated['notes'];
            }

            DB::table('stock_movements')->insert([
                'sku' => $validated['sku'],
                'type' => 'adjustment',
                'quantity' => $adjustmentQty,
                'balance_after' => $newBalance,
                'reference_type' => 'manual_adjustment',
                'reference_number' => 'ADJ-' . date('YmdHis'),
                'notes' => $notes,
                'performed_by' => $user['user_id'],
                'performed_by_name' => $user['name'] ?? $user['email'],
                'performed_at' => now(),
                'created_at' => now(),
                'updated_at' => now()
            ]);

            DB::commit();

            $message = $validated['adjustment_type'] === 'add'
                ? "Added {$validated['quantity']} units to {$inventory->product_name}"
                : "Removed {$validated['quantity']} units from {$inventory->product_name}";

            return redirect()->route('inventory.index')->with('success', $message . ". New balance: {$newBalance}");

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error("Failed to process stock adjustment", [
                'error' => $e->getMessage(),
                'user_id' => $user['user_id']
            ]);

            return redirect()->back()->with('error', 'Failed to process adjustment. Please try again.');
        }
    }

    public function valueReport(Request $request)
    {
        $user = $request->input('omnia_user');

        // Get all inventory with value calculation
        $inventory = DB::table('inventory')
            ->leftJoin(
                DB::raw('(SELECT sku, AVG(unit_cost) as avg_cost FROM purchase_order_items GROUP BY sku) as costs'),
                'inventory.sku',
                '=',
                'costs.sku'
            )
            ->select(
                'inventory.*',
                DB::raw('COALESCE(costs.avg_cost, 0) as avg_cost'),
                DB::raw('inventory.quantity * COALESCE(costs.avg_cost, 0) as total_value')
            )
            ->orderBy('total_value', 'desc')
            ->get();

        $totalValue = $inventory->sum('total_value');
        $totalItems = $inventory->sum('quantity');

        return view('inventory.value-report', compact('inventory', 'user', 'totalValue', 'totalItems'));
    }
}
