<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use App\Models\User;

class PermissionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Reset cached roles and permissions
        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

        // Create permissions
        $permissions = [
            'marketing.access',
            'kitchen.access',
            'production.access',
            'finance.access',
            'hr.access',
        ];

        foreach ($permissions as $permission) {
            Permission::firstOrCreate(['name' => $permission]);
        }

        // Create roles
        // Admin role - has access to everything
        $adminRole = Role::firstOrCreate(['name' => 'admin']);
        $adminRole->givePermissionTo($permissions);

        // Seller role - only Marketing access
        $sellerRole = Role::firstOrCreate(['name' => 'seller']);
        $sellerRole->givePermissionTo('marketing.access');

        // Manager role - Marketing + Kitchen access
        $managerRole = Role::firstOrCreate(['name' => 'manager']);
        $managerRole->givePermissionTo(['marketing.access', 'kitchen.access']);

        // Kitchen staff role - only Kitchen access
        $kitchenRole = Role::firstOrCreate(['name' => 'kitchen-staff']);
        $kitchenRole->givePermissionTo('kitchen.access');

        // Production staff role - only Production access
        $productionRole = Role::firstOrCreate(['name' => 'production-staff']);
        $productionRole->givePermissionTo('production.access');

        // Production manager role - Production access with management permissions
        $productionManagerRole = Role::firstOrCreate(['name' => 'production-manager']);
        $productionManagerRole->givePermissionTo('production.access');

        // Finance staff role - only Finance access
        $financeRole = Role::firstOrCreate(['name' => 'finance-staff']);
        $financeRole->givePermissionTo('finance.access');

        // HR staff role - only HR access
        $hrRole = Role::firstOrCreate(['name' => 'hr-staff']);
        $hrRole->givePermissionTo('hr.access');

        // Assign admin role to first user (System Administrator)
        $adminUser = User::find(1);
        if ($adminUser) {
            $adminUser->assignRole('admin');
            $this->command->info("✅ Admin role assigned to: {$adminUser->email}");
        }

        $this->command->info('✅ Permissions and roles created successfully!');
        $this->command->info('');
        $this->command->info('Created Permissions:');
        foreach ($permissions as $permission) {
            $this->command->info("  - {$permission}");
        }
        $this->command->info('');
        $this->command->info('Created Roles:');
        $this->command->info('  - admin (all access)');
        $this->command->info('  - seller (marketing only)');
        $this->command->info('  - manager (marketing + kitchen)');
        $this->command->info('  - kitchen-staff (kitchen only)');
        $this->command->info('  - production-staff (production only)');
        $this->command->info('  - production-manager (production with management permissions)');
        $this->command->info('  - finance-staff (finance only)');
        $this->command->info('  - hr-staff (hr only)');
    }
}
