<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Cache;

class AuthController extends Controller
{
    /**
     * Login and create session
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (!Auth::attempt($request->only('email', 'password'))) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        $user = Auth::user();
        $user->load('roles', 'permissions');

        $sessionId = Str::random(40);

        $sessionData = [
            'user_id' => $user->id,
            'email' => $user->email,
            'name' => $user->name,
            'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
            'roles' => $user->roles->pluck('name')->toArray(),
            'created_at' => now()->toDateTimeString(),
        ];

        Cache::put("omnia_session:{$sessionId}", $sessionData, now()->addHours(24));

        \Log::info("User logged in: {$user->email}", ['session_id' => $sessionId]);

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'data' => [
                'user' => $user,
                'session_id' => $sessionId,
                'permissions' => $sessionData['permissions'],
                'roles' => $sessionData['roles'],
            ]
        ])->cookie(
            'omnia_session',
            $sessionId,
            1440,
            '/',
            '127.0.0.1',
            false,
            true,
            false,
            'lax'
        );
    }

    /**
     * Verify session (called by other services)
     */
    public function verify(Request $request)
    {
        $sessionId = $request->cookie('omnia_session')
                    ?? $request->input('session_id')
                    ?? $request->header('X-Session-ID');

        if (!$sessionId) {
            return response()->json([
                'success' => false,
                'message' => 'No session provided'
            ], 401);
        }

        $sessionData = Cache::get("omnia_session:{$sessionId}");

        if (!$sessionData) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired session'
            ], 401);
        }

        Cache::put("omnia_session:{$sessionId}", $sessionData, now()->addHours(24));

        return response()->json([
            'success' => true,
            'message' => 'Session valid',
            'data' => $sessionData
        ]);
    }

    /**
     * Check if user has specific permission
     */
    public function checkPermission(Request $request)
    {
        $request->validate([
            'permission' => 'required|string',
        ]);

        $sessionId = $request->cookie('omnia_session')
                    ?? $request->input('session_id');

        if (!$sessionId) {
            return response()->json([
                'success' => false,
                'message' => 'No session provided'
            ], 401);
        }

        $sessionData = Cache::get("omnia_session:{$sessionId}");

        if (!$sessionData) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid session'
            ], 401);
        }

        $hasPermission = in_array($request->permission, $sessionData['permissions']);

        return response()->json([
            'success' => true,
            'has_permission' => $hasPermission,
            'permission' => $request->permission,
        ]);
    }

    /**
     * Logout and destroy session
     */
    public function logout(Request $request)
    {
        $sessionId = $request->cookie('omnia_session');

        if ($sessionId) {
            Cache::forget("omnia_session:{$sessionId}");
            \Log::info("User logged out", ['session_id' => $sessionId]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully'
        ])->withoutCookie('omnia_session');
    }

    /**
     * Get current user info
     */
    public function me(Request $request)
    {
        $sessionId = $request->cookie('omnia_session');

        if (!$sessionId) {
            return response()->json([
                'success' => false,
                'message' => 'Not authenticated'
            ], 401);
        }

        $sessionData = Cache::get("omnia_session:{$sessionId}");

        if (!$sessionData) {
            return response()->json([
                'success' => false,
                'message' => 'Session expired'
            ], 401);
        }

        $user = \App\Models\User::with('roles', 'permissions')->find($sessionData['user_id']);

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'user' => $user,
                'permissions' => $user->getAllPermissions()->pluck('name'),
                'roles' => $user->roles->pluck('name'),
            ]
        ]);
    }

    /**
     * Login via web form and redirect based on role
     */
    public function loginWeb(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (!Auth::attempt($request->only('email', 'password'))) {
            return back()->with('error', 'Invalid email or password')->withInput();
        }

        $user = Auth::user();
        $user->load('roles', 'permissions');

        $sessionId = Str::random(40);

        $sessionData = [
            'user_id' => $user->id,
            'email' => $user->email,
            'name' => $user->name,
            'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
            'roles' => $user->roles->pluck('name')->toArray(),
            'created_at' => now()->toDateTimeString(),
        ];

        Cache::put("omnia_session:{$sessionId}", $sessionData, now()->addHours(24));

        \Log::info("User logged in via web: {$user->email}", ['session_id' => $sessionId]);

        // Determine redirect URL based on role/permission
        $redirectUrl = $this->getRedirectUrl($user);

        // Append session ID to URL for cross-port SSO
        $redirectUrl .= '?omnia_session=' . $sessionId;

        return redirect($redirectUrl)->cookie(
            'omnia_session',
            $sessionId,
            1440,
            '/',
            null, // null domain means cookie is set for current host:port only
            false,
            true,
            false,
            'lax'
        );
    }

    /**
     * Logout via web and redirect to login
     */
    public function logoutWeb(Request $request)
    {
        $sessionId = $request->cookie('omnia_session');

        if ($sessionId) {
            Cache::forget("omnia_session:{$sessionId}");
            \Log::info("User logged out via web", ['session_id' => $sessionId]);
        }

        return redirect('/login')->withoutCookie('omnia_session');
    }

    /**
     * Determine redirect URL based on user's primary role/permission
     */
    private function getRedirectUrl($user)
    {
        // Check roles in priority order
        if ($user->hasRole('admin')) {
            return 'http://omnia.test/dashboard';
        }

        // Check permissions for specific modules
        if ($user->can('kitchen.access')) {
            return 'http://kitchen.omnia.test/dashboard';
        }

        if ($user->can('production.access')) {
            return 'http://127.0.0.1:8002/dashboard';
        }

        if ($user->can('finance.access')) {
            return 'http://finance.omnia.test/dashboard';
        }

        if ($user->can('hr.access')) {
            return 'http://hr.omnia.test/dashboard';
        }

        if ($user->can('marketing.access')) {
            return 'http://omnia.test/dashboard';
        }

        // Default fallback to marketing
        return 'http://omnia.test/dashboard';
    }

    /**
     * SSO Redirect - Creates/retrieves session for authenticated user and redirects to target service
     * Used for cross-module navigation
     */
    public function ssoRedirect(Request $request)
    {
        $request->validate([
            'user_id' => 'required|integer',
            'target_url' => 'required|url',
        ]);

        // Find the user
        $user = \App\Models\User::with('roles', 'permissions')->find($request->user_id);

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        // Create or retrieve existing session
        $sessionId = Str::random(40);

        $sessionData = [
            'user_id' => $user->id,
            'email' => $user->email,
            'name' => $user->name,
            'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
            'roles' => $user->roles->pluck('name')->toArray(),
            'created_at' => now()->toDateTimeString(),
        ];

        Cache::put("omnia_session:{$sessionId}", $sessionData, now()->addHours(24));

        \Log::info("SSO redirect session created", [
            'user_id' => $user->id,
            'email' => $user->email,
            'target_url' => $request->target_url,
            'session_id' => $sessionId
        ]);

        // Redirect to target service with session ID
        $targetUrl = $request->target_url;
        $separator = parse_url($targetUrl, PHP_URL_QUERY) ? '&' : '?';
        $redirectUrl = $targetUrl . $separator . 'omnia_session=' . $sessionId;

        return redirect($redirectUrl);
    }
}
