# OmniaBoss Production Module - Complete Documentation

**Last Updated:** November 25, 2025
**Module Location:** `C:\laragon\www\omniaboss\services\production`
**Access URL:** http://127.0.0.1:8002/dashboard
**Version:** 1.0 - Phase 1 (Complete)

---

## 📋 Table of Contents

1. [Overview](#overview)
2. [Architecture](#architecture)
3. [Features](#features)
4. [User Interface](#user-interface)
5. [Database Schema](#database-schema)
6. [Workflow](#workflow)
7. [API Endpoints](#api-endpoints)
8. [SSO Authentication](#sso-authentication)
9. [Testing](#testing)
10. [Future Enhancements (Phase 2)](#future-enhancements-phase-2)
11. [Troubleshooting](#troubleshooting)

---

## Overview

The **Production Module** is part of the OmniaBoss microservices ecosystem. It allows production staff to view, process, and manage orders that have been printed and are ready for packing.

### Purpose
- View all pending orders (status: 'printed') from Marketing database in a single view
- Mark orders as 'packed' after packing completion
- Mark orders as 'completed' after final processing
- Track order progress through production stages
- Provide visibility to all departments on order processing status

### Key Benefits
- ✅ **Single-Page View** - All order details visible without clicking
- ✅ **SKU-Based** - Production team reads SKUs (faster than product names)
- ✅ **Tooltips** - Product names available on hover
- ✅ **Status Tracking** - Pending → Packed → Completed
- ✅ **Real-time Updates** - Status changes reflected immediately in Marketing dashboard
- ✅ **SSO Integration** - Seamless authentication across modules

---

## Architecture

### Microservices Structure

```
OmniaBoss Ecosystem
│
├── Auth Service (Port 8001)
│   ├── SSO Authentication
│   ├── Role-Based Access Control
│   └── Session Management
│
├── Marketing Core (Port 80 / omnia.test)
│   ├── Main Database (omnia)
│   ├── Orders Management
│   └── Product Catalog
│
└── Production Module (Port 8002)
    ├── Read Orders from Marketing DB
    ├── Update Order Status
    └── Production Dashboard
```

### Database Connections

The Production module uses **TWO database connections**:

1. **Production DB** (`omnia_production_db`) - Own module data (future use)
2. **Marketing DB** (`omnia`) - Read orders, update status

**Configuration:** `.env` file

```env
# Production Module Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=omnia_production_db
DB_USERNAME=root
DB_PASSWORD=

# Marketing Core Database (Read-Only for Orders)
MARKETING_DB_HOST=127.0.0.1
MARKETING_DB_PORT=3306
MARKETING_DB_DATABASE=omnia
MARKETING_DB_USERNAME=root
MARKETING_DB_PASSWORD=
```

---

## Features

### 1. Dashboard with Status Tabs

**URL:** http://127.0.0.1:8002/dashboard

The dashboard has **3 status tabs**:

| Tab | Database Status | Description | Badge Color |
|-----|----------------|-------------|-------------|
| **Pending** | `printed` | Orders ready for packing | Yellow |
| **Packed** | `packed` | Orders that have been packed | Blue |
| **Completed** | `completed` | Finished orders | Green |

### 2. Order Table Columns

Each tab shows a table with the following columns:

| Column | Description | Example |
|--------|-------------|---------|
| **Order ID** | Order number + Global ID | `#CK-NAD-148084`<br>`ORD-2025-001234` |
| **Customer Name** | Full name + phone | `John Doe`<br>`+60123456789` |
| **Address** | Shipping address (2-line clamp) | `123 Main St, City, 50000 State` |
| **Items (SKU x Qty)** | Product SKUs with tooltips | `200WLA x1` `300ABC x2`<br>(Hover for product name) |
| **Created** | Order date & time | `Nov 25, 2025`<br>`14:30` |
| **Actions** | Status-specific button | See below |

### 3. Action Buttons

**Pending Tab:**
- **Mark Packed** button (Blue) - Updates status from `printed` → `packed`

**Packed Tab:**
- **Mark Completed** button (Green) - Updates status from `packed` → `completed`

**Completed Tab:**
- **Completed** badge (Read-only) - No action available

### 4. SKU Display with Tooltips

**Why SKUs?**
- Production staff are familiar with SKU codes
- SKUs are shorter and save space
- Faster to scan than long product names

**Implementation:**
- Each SKU is displayed as a badge: `200WLA x1`
- Hover over SKU → Tooltip shows full product name: "Women Love Advance"
- Uses HTML `title` attribute for native browser tooltips

**Example:**
```html
<span title="Women Love Advance">200WLA x1</span>
<span title="Men Vitality Boost">300MVB x2</span>
```

---

## User Interface

### Status Tab Appearance

**Active Tab:**
- Colored background (Yellow/Blue/Green)
- Colored bottom border
- Badge shows count with white background

**Inactive Tab:**
- Gray text
- Gray badge
- Hover effect

### Table Layout

```
┌─────────────┬──────────────┬──────────────┬──────────────┬──────────┬──────────────┐
│  Order ID   │ Customer Name│   Address    │ Items (SKU)  │ Created  │   Actions    │
├─────────────┼──────────────┼──────────────┼──────────────┼──────────┼──────────────┤
│ #CK-148084  │ John Doe     │ 123 Main St, │ 200WLA x1    │ Nov 25   │ [Mark Packed]│
│ ORD-001234  │ +6012345678  │ City, 50000  │ 300ABC x2    │ 14:30    │              │
└─────────────┴──────────────┴──────────────┴──────────────┴──────────┴──────────────┘
```

### Empty State

When no orders in the current tab:
- Box icon
- Message: "No orders in [Tab Name]"
- Encouragement message

---

## Database Schema

### Orders Table (Marketing DB)

**Table:** `omnia.orders`

**Key Fields Used:**

| Field | Type | Description |
|-------|------|-------------|
| `id` | bigint | Primary key |
| `order_number` | varchar(255) | Order number (e.g., CK-NAD-148084) |
| `global_order_id` | varchar(255) | Global unique ID |
| `status` | varchar(255) | **Order status** (`printed`, `packed`, `completed`) |
| `billing` | longtext (JSON) | Customer billing info |
| `shipping` | longtext (JSON) | Shipping address |
| `line_items` | longtext (JSON) | Products in order |
| `total` | decimal(10,2) | Order total |
| `date_created` | timestamp | Order creation date |
| `updated_at` | timestamp | Last update timestamp |

**JSON Field Structures:**

**billing JSON:**
```json
{
  "first_name": "John",
  "last_name": "Doe",
  "phone": "+60123456789",
  "email": "john@example.com",
  "address_1": "123 Main Street",
  "city": "Kuala Lumpur",
  "postcode": "50000",
  "state": "Wilayah Persekutuan"
}
```

**shipping JSON:**
```json
{
  "address_1": "123 Main Street",
  "address_2": "Apt 4B",
  "city": "Kuala Lumpur",
  "postcode": "50000",
  "state": "Wilayah Persekutuan"
}
```

**line_items JSON:**
```json
{
  "7": {
    "name": "Women Love Advance",
    "sku": "200WLA",
    "quantity": "1",
    "price": "133.00",
    "total": 133
  },
  "12": {
    "name": "Men Vitality Boost",
    "sku": "300MVB",
    "quantity": "2",
    "price": "89.00",
    "total": 178
  }
}
```

---

## Workflow

### Order Lifecycle

```
Marketing Core                    Production Module
─────────────                     ─────────────────

1. Order Created
   └─> status: 'processing'

2. Order Approved
   └─> status: 'approval'

3. Order Printed
   └─> status: 'printed' ────────> Appears in PENDING tab
                                    │
                                    │ [Production staff clicks "Mark Packed"]
                                    ↓
4. Order Packed                    Status updated to 'packed'
   └─> status: 'packed' <────────  Moves to PACKED tab
                                    │
                                    │ [Staff clicks "Mark Completed"]
                                    ↓
5. Order Completed                 Status updated to 'completed'
   └─> status: 'completed' <────── Moves to COMPLETED tab

6. Delivered (by Logistics team)
   └─> status: 'delivered'
```

### Production Staff Actions

**Step 1: View Pending Orders**
1. Login at http://127.0.0.1:8001/login
2. Credentials: `production@omnia.test` / `password123`
3. Auto-redirect to http://127.0.0.1:8002/dashboard
4. See all orders with status='printed' in table

**Step 2: Pack Order**
1. Read SKU codes from "Items" column
2. Pick products from warehouse using SKUs
3. Pack order in box
4. Click "Mark Packed" button for that order
5. Order moves from "Pending" → "Packed" tab

**Step 3: Complete Order**
1. Switch to "Packed" tab
2. Final quality check
3. Click "Mark Completed" button
4. Order moves to "Completed" tab

---

## API Endpoints

### Protected Routes (SSO Required)

**Base URL:** http://127.0.0.1:8002

| Method | Endpoint | Description | Permission Required |
|--------|----------|-------------|---------------------|
| GET | `/dashboard` | View orders dashboard | `production.access` |
| GET | `/dashboard?status=pending` | View pending orders | `production.access` |
| GET | `/dashboard?status=packed` | View packed orders | `production.access` |
| GET | `/dashboard?status=completed` | View completed orders | `production.access` |
| POST | `/orders/{id}/pack` | Mark order as packed | `production.access` |
| POST | `/orders/{id}/complete` | Mark order as completed | `production.access` |

### Controller Methods

**File:** `app/Http/Controllers/OrderController.php`

```php
// View dashboard with status tabs
public function index(Request $request)
// Input: ?status=pending|packed|completed
// Output: View with orders array, statusCounts, currentStatus

// Mark order as packed
public function markAsPacked(Request $request, $id)
// Updates status: printed → packed
// Redirects to: /dashboard?status=pending

// Mark order as completed
public function markAsCompleted(Request $request, $id)
// Updates status: packed → completed
// Redirects to: /dashboard?status=packed
```

---

## SSO Authentication

### How It Works

**Cross-Port SSO Flow:**

```
1. User visits: http://127.0.0.1:8002/dashboard
   ├─> No cookie found
   └─> Redirect to: http://127.0.0.1:8001/login

2. User logs in at Auth service
   ├─> Credentials verified
   ├─> Session created in cache
   └─> Redirect to: http://127.0.0.1:8002/dashboard?omnia_session=XXXXX

3. Production middleware receives request
   ├─> Reads session ID from URL query parameter
   ├─> Verifies session with Auth service API
   ├─> Checks for 'production.access' permission
   ├─> Sets cookie for port 8002
   └─> Redirects to clean URL: /dashboard

4. Subsequent requests
   └─> Use cookie (no query parameter needed)
```

### Middleware

**File:** `app/Http/Middleware/OmniaSSOAuth.php`

**Key Features:**
- Reads session from query parameter OR cookie
- Verifies with Auth service: `GET /api/auth/verify`
- Checks permission: `production.access`
- Sets local cookie for future requests
- Redirects to clean URL (removes query parameter)

**Permission Check:**
```php
if (!in_array('production.access', $userData['permissions'])) {
    abort(403, 'You do not have permission to access Production module');
}
```

### Test Users

**Production Staff:**
- Email: `production@omnia.test`
- Password: `password123`
- Permission: `production.access`

**Admin:**
- Email: `admin@omnia.test`
- Password: `password123`
- Role: `admin` (has all permissions)

---

## Testing

### Test Scenario 1: View Pending Orders

```bash
# Test URL
http://127.0.0.1:8002/dashboard

# Expected Result:
- Shows orders with status='printed' from Marketing DB
- Displays: Order ID, Customer Name, Address, SKU items, Date
- "Mark Packed" button visible for each order
- Status badge shows count (e.g., "1352 orders")
```

### Test Scenario 2: Mark Order as Packed

```bash
# Action: Click "Mark Packed" on order #148085

# Expected Result:
- Order status in Marketing DB updated: printed → packed
- Order disappears from Pending tab
- Order appears in Packed tab
- Success message: "Order #CK-NAD-148085 has been marked as packed!"
- Pending count decreases (1352 → 1351)
- Packed count increases
```

### Test Scenario 3: SKU Tooltips

```bash
# Action: Hover mouse over SKU badge "200WLA x1"

# Expected Result:
- Browser tooltip appears showing: "Women Love Advance"
- No delay, native browser tooltip
```

### Test Scenario 4: Cross-Module Verification

```bash
# 1. Mark order as packed in Production module
# 2. Visit Marketing dashboard: http://omnia.test/dashboard
# 3. Click "Packed" tab

# Expected Result:
- Order appears in Marketing "Packed" tab
- Status badge shows "Packed" (blue color)
- Both modules show same status
```

---

## Future Enhancements (Phase 2)

### 1. Inventory/Stock Integration

**Purpose:** Auto-reduce stock when order is packed

**Workflow:**
```
Order Packed
  └─> Read line_items SKUs
      └─> For each SKU:
          ├─> Find product in inventory DB
          ├─> Reduce quantity by order quantity
          └─> Log transaction
```

**Database Changes Needed:**
- `omnia_production_db.inventory` table
- Link SKUs to inventory items
- Stock transaction logs

**Implementation Notes:**
- Add checkbox: "Reduce stock automatically"
- Manual override option
- Stock validation before packing
- Low stock alerts

### 2. Packing Notes

**Purpose:** Add notes when packing orders

**Changes:**
- Add `notes` field to markAsPacked()
- Store in Production DB or Marketing DB
- Display notes in Marketing dashboard

### 3. Bulk Actions

**Purpose:** Mark multiple orders as packed at once

**Features:**
- Checkbox for each order
- "Select All" option
- Bulk "Mark Packed" button
- Confirm dialog with order list

### 4. Print Packing Slips

**Purpose:** Generate PDF packing slips for production

**Features:**
- Button: "Print Packing Slip"
- PDF with order details
- Barcode for tracking
- SKU list with quantities

### 5. Search & Filters

**Purpose:** Find specific orders quickly

**Features:**
- Search by Order ID, Customer Name, Phone
- Filter by date range
- Filter by product SKU
- Sort by columns

### 6. Performance Metrics

**Purpose:** Track production team performance

**Metrics:**
- Orders packed per day
- Average packing time
- Staff performance leaderboard
- Hourly processing rate

---

## Troubleshooting

### Issue 1: SSO Redirect Loop

**Symptom:** Keeps redirecting between login and dashboard

**Cause:** Cookie domain mismatch

**Solution:**
```php
// Check middleware sets cookie correctly
return redirect($request->url())->cookie(
    'omnia_session',
    $sessionId,
    1440,
    '/',
    null, // Important: null for 127.0.0.1
    false,
    true,
    false,
    'lax'
);
```

### Issue 2: No Orders Showing

**Symptom:** Dashboard shows "No orders in Pending"

**Possible Causes:**
1. No orders with status='printed' in database
2. Database connection issue

**Solution:**
```bash
# Check orders count
mysql -u root -e "SELECT COUNT(*) FROM omnia.orders WHERE status='printed'"

# Check database connection
php artisan tinker
>>> DB::connection('marketing')->table('orders')->count()
```

### Issue 3: Tooltips Not Working

**Symptom:** Hovering over SKU doesn't show product name

**Cause:** Browser tooltip blocked or title attribute missing

**Solution:**
```php
// Verify tooltip attribute in view
<span title="{{ $item->product_name }}">
    {{ $item->product_sku }} x{{ $item->quantity }}
</span>
```

### Issue 4: Status Not Updating in Marketing

**Symptom:** Order marked as packed but still shows "printed" in Marketing

**Cause:** Database transaction not committed or cache issue

**Solution:**
```bash
# Check database directly
mysql -u root -e "SELECT id, order_number, status FROM omnia.orders WHERE id=148085"

# Clear cache
php artisan cache:clear

# Check logs
tail -f storage/logs/laravel.log
```

---

## File Structure

```
services/production/
│
├── app/
│   ├── Http/
│   │   ├── Controllers/
│   │   │   └── OrderController.php          # Main controller
│   │   └── Middleware/
│   │       └── OmniaSSOAuth.php             # SSO middleware
│   └── Models/
│
├── bootstrap/
│   └── app.php                               # Middleware registration
│
├── config/
│   └── database.php                          # Database connections
│
├── resources/
│   └── views/
│       ├── layouts/
│       │   └── app.blade.php                 # Main layout
│       └── orders/
│           └── index.blade.php               # Dashboard view
│
├── routes/
│   └── web.php                               # Routes definition
│
├── .env                                      # Environment config
└── README.md                                 # Basic readme
```

---

## Logs

**Location:** `storage/logs/laravel.log`

**Log Examples:**

**Successful pack:**
```
[2025-11-25 08:38:53] local.INFO: Order #148085 marked as packed
{
  "user_id": 5,
  "user_email": "production@omnia.test",
  "order_number": "CK-NAD-148084"
}
```

**Permission denied:**
```
[2025-11-25 08:40:12] local.ERROR: SSO Auth Error:
User does not have production.access permission
```

---

## Summary

### What Production Module Does:
✅ Displays pending orders (status='printed') in table format
✅ Shows all order details: ID, Customer, Address, SKUs, Date
✅ SKU-based display with product name tooltips
✅ Mark as Packed button (updates status to 'packed')
✅ Mark as Completed button (updates status to 'completed')
✅ Status tabs: Pending | Packed | Completed
✅ SSO authentication with Auth service
✅ Real-time status sync with Marketing dashboard

### What It Doesn't Do (Yet - Phase 2):
❌ Inventory/stock reduction
❌ Packing notes
❌ Bulk actions
❌ Print packing slips
❌ Search & filters

---

## Support & Contact

For issues or questions about the Production module:
1. Check this documentation first
2. Review logs in `storage/logs/laravel.log`
3. Check Marketing database: `omnia.orders` table
4. Verify SSO session in cache

---

**Document Version:** 1.0
**Module Version:** 1.0 - Phase 1
**Status:** Production Ready ✅
