<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\InvoiceController;
use App\Http\Controllers\OrderController;

/*
|--------------------------------------------------------------------------
| Invoice Routes
|--------------------------------------------------------------------------
|
| These routes handle invoice-related functionality including viewing,
| printing, and managing invoices for orders.
|
*/

Route::middleware(['auth', 'verified'])->group(function () {
    
    /*
    |--------------------------------------------------------------------------
    | Invoice Display Routes
    |--------------------------------------------------------------------------
    */
    
    // Main invoice view
    Route::get('/orders/{order}/invoice', [InvoiceController::class, 'show'])
        ->name('invoices.show')
        ->middleware('can:view,order');
    
    // Print-optimized invoice view
    Route::get('/orders/{order}/invoice/print', [InvoiceController::class, 'print'])
        ->name('invoices.print')
        ->middleware('can:view,order');
    
    // Download invoice as PDF
    Route::get('/orders/{order}/invoice/download', [InvoiceController::class, 'download'])
        ->name('invoices.download')
        ->middleware('can:view,order');
    
    /*
    |--------------------------------------------------------------------------
    | Invoice Management Routes
    |--------------------------------------------------------------------------
    */
    
    // Update invoice/order status
    Route::patch('/orders/{order}/status', [InvoiceController::class, 'updateStatus'])
        ->name('invoices.update-status')
        ->middleware('can:update,order');
    
    // Update payment status
    Route::patch('/orders/{order}/payment-status', [InvoiceController::class, 'updatePaymentStatus'])
        ->name('invoices.update-payment-status')
        ->middleware('can:update,order');
    
    // Update tracking information
    Route::patch('/orders/{order}/tracking', [InvoiceController::class, 'updateTracking'])
        ->name('invoices.update-tracking')
        ->middleware('can:update,order');
    
    /*
    |--------------------------------------------------------------------------
    | Bulk Invoice Operations
    |--------------------------------------------------------------------------
    */
    
    // Bulk print invoices
    Route::post('/invoices/bulk-print', [InvoiceController::class, 'bulkPrint'])
        ->name('invoices.bulk-print')
        ->middleware('can:bulk-manage,App\Models\Order');
    
    // Bulk download invoices
    Route::post('/invoices/bulk-download', [InvoiceController::class, 'bulkDownload'])
        ->name('invoices.bulk-download')
        ->middleware('can:bulk-manage,App\Models\Order');
    
    // Bulk email invoices
    Route::post('/invoices/bulk-email', [InvoiceController::class, 'bulkEmail'])
        ->name('invoices.bulk-email')
        ->middleware('can:bulk-manage,App\Models\Order');
    
    /*
    |--------------------------------------------------------------------------
    | Invoice Templates & Customization
    |--------------------------------------------------------------------------
    */
    
    // Preview invoice with different template
    Route::get('/orders/{order}/invoice/preview/{template?}', [InvoiceController::class, 'preview'])
        ->name('invoices.preview')
        ->middleware('can:view,order');
    
    // Invoice settings/customization
    Route::get('/invoices/settings', [InvoiceController::class, 'settings'])
        ->name('invoices.settings')
        ->middleware('can:manage-settings');
    
    Route::post('/invoices/settings', [InvoiceController::class, 'updateSettings'])
        ->name('invoices.update-settings')
        ->middleware('can:manage-settings');
    
    /*
    |--------------------------------------------------------------------------
    | Enhanced Order Management Routes
    |--------------------------------------------------------------------------
    */
    
    // Quick status updates for order list
    Route::patch('/orders/{order}/quick-status', [OrderController::class, 'quickStatusUpdate'])
        ->name('orders.quick-status')
        ->middleware('can:update,order');
    
    // Mark order as delivered
    Route::patch('/orders/{order}/delivered', [OrderController::class, 'markAsDelivered'])
        ->name('orders.mark-delivered')
        ->middleware('can:update,order');
    
    // Mark order as pending COD
    Route::patch('/orders/{order}/pending-cod', [OrderController::class, 'markAsPendingCOD'])
        ->name('orders.pending-cod')
        ->middleware('can:update,order');
    
    // Reject/cancel order
    Route::patch('/orders/{order}/reject', [OrderController::class, 'rejectOrder'])
        ->name('orders.reject')
        ->middleware('can:update,order');
});

/*
|--------------------------------------------------------------------------
| Public Invoice Routes (for customer access)
|--------------------------------------------------------------------------
*/

// Public invoice view (with token/hash for security)
Route::get('/public/invoice/{order}/{token}', [InvoiceController::class, 'publicView'])
    ->name('invoices.public')
    ->middleware('signed');

// Public invoice print view
Route::get('/public/invoice/{order}/{token}/print', [InvoiceController::class, 'publicPrint'])
    ->name('invoices.public-print')
    ->middleware('signed');

/*
|--------------------------------------------------------------------------
| API Routes for AJAX Operations
|--------------------------------------------------------------------------
*/

Route::middleware(['auth:sanctum', 'verified'])->prefix('api')->group(function () {
    
    // Get order status options
    Route::get('/order-statuses', [InvoiceController::class, 'getStatusOptions'])
        ->name('api.order-statuses');
    
    // Get payment method options  
    Route::get('/payment-methods', [InvoiceController::class, 'getPaymentMethods'])
        ->name('api.payment-methods');
    
    // Quick order search for invoice generation
    Route::get('/orders/search', [OrderController::class, 'search'])
        ->name('api.orders.search');
    
    // Get order details for quick view
    Route::get('/orders/{order}/details', [OrderController::class, 'getDetails'])
        ->name('api.orders.details')
        ->middleware('can:view,order');
        
    // Update customer information
    Route::patch('/orders/{order}/customer', [OrderController::class, 'updateCustomer'])
        ->name('api.orders.update-customer')
        ->middleware('can:update,order');
});

/*
|--------------------------------------------------------------------------
| Webhook Routes
|--------------------------------------------------------------------------
*/

// Webhook for external service integrations
Route::post('/webhooks/invoice-status', [InvoiceController::class, 'handleWebhook'])
    ->name('webhooks.invoice-status')
    ->middleware('verify-webhook-signature');

// Payment gateway webhooks
Route::post('/webhooks/payment/{gateway}', [InvoiceController::class, 'handlePaymentWebhook'])
    ->name('webhooks.payment')
    ->middleware('verify-payment-webhook');
