@extends('layouts.app')

@section('title', 'Webhook Details')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
    <div class="mb-6">
        <div class="flex justify-between items-start">
            <div>
                <h1 class="text-2xl font-bold">{{ $webhook->name }}</h1>
                <p class="text-gray-600 dark:text-gray-400">{{ ucfirst($webhook->type) }} webhook source</p>
            </div>
            <div class="flex space-x-3">
                <a href="{{ route('stores.webhooks.edit', $webhook) }}"
                   class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700">
                    Edit
                </a>
                <form action="{{ route('stores.webhooks.toggle', $webhook) }}" method="POST" class="inline">
                    @csrf
                    <button type="submit"
                            class="px-4 py-2 rounded-lg {{ $webhook->is_active ? 'bg-orange-600 hover:bg-orange-700' : 'bg-green-600 hover:bg-green-700' }} text-white">
                        {{ $webhook->is_active ? 'Disable' : 'Enable' }}
                    </button>
                </form>
            </div>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-white dark:bg-gray-800 p-6 rounded-lg shadow">
            <div class="flex items-center">
                <div class="flex-1">
                    <p class="text-sm text-gray-600 dark:text-gray-400">Status</p>
                    <p class="text-2xl font-bold">
                        @if($webhook->is_active)
                            <span class="text-green-600 dark:text-green-400">Active</span>
                        @else
                            <span class="text-red-600 dark:text-red-400">Inactive</span>
                        @endif
                    </p>
                </div>
            </div>
        </div>

        <div class="bg-white dark:bg-gray-800 p-6 rounded-lg shadow">
            <div class="flex items-center">
                <div class="flex-1">
                    <p class="text-sm text-gray-600 dark:text-gray-400">Total Requests</p>
                    <p class="text-2xl font-bold">{{ $webhook->webhook_logs_count ?? 0 }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white dark:bg-gray-800 p-6 rounded-lg shadow">
            <div class="flex items-center">
                <div class="flex-1">
                    <p class="text-sm text-gray-600 dark:text-gray-400">Successful</p>
                    <p class="text-2xl font-bold text-green-600 dark:text-green-400">
                        {{ $webhook->success_count ?? 0 }}
                    </p>
                </div>
            </div>
        </div>

        <div class="bg-white dark:bg-gray-800 p-6 rounded-lg shadow">
            <div class="flex items-center">
                <div class="flex-1">
                    <p class="text-sm text-gray-600 dark:text-gray-400">Failed</p>
                    <p class="text-2xl font-bold text-red-600 dark:text-red-400">
                        {{ $webhook->failed_count ?? 0 }}
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Webhook URL Card -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow mb-6">
        <div class="p-6">
            <h2 class="text-lg font-semibold mb-4">Webhook Configuration</h2>

            <div class="space-y-4">
                <div>
                    <label class="block text-sm font-medium mb-2">Webhook URL</label>
                    <div class="flex items-center space-x-2">
                        <input type="text" readonly
                               value="{{ $webhook->webhook_url }}"
                               class="flex-1 rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-sm font-mono"
                               id="webhookUrl">
                        <button type="button"
                                onclick="copyWebhookUrl()"
                                class="px-3 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm whitespace-nowrap">
                            Copy URL
                        </button>
                    </div>
                    <p class="mt-2 text-xs text-gray-500 dark:text-gray-400">
                        Configure this URL in your payment platform's webhook settings
                    </p>
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">Test Endpoint</label>
                    <div class="flex items-center space-x-2">
                        <input type="text" readonly
                               value="{{ route('webhook.generic.test', ['token' => $webhook->webhook_token]) }}"
                               class="flex-1 rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-sm font-mono"
                               id="testUrl">
                        <a href="{{ route('webhook.generic.test', ['token' => $webhook->webhook_token]) }}"
                           target="_blank"
                           class="px-3 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 text-sm whitespace-nowrap">
                            Test Connection
                        </a>
                    </div>
                    <p class="mt-2 text-xs text-gray-500 dark:text-gray-400">
                        Visit this URL to test if the webhook is accessible
                    </p>
                </div>

                @if($webhook->secret_key)
                <div>
                    <label class="block text-sm font-medium mb-2">Secret Key</label>
                    <input type="password" readonly
                           value="{{ $webhook->secret_key }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 text-sm">
                </div>
                @endif

                <div class="pt-4 border-t border-gray-200 dark:border-gray-700">
                    <form action="{{ route('stores.webhooks.regenerate', $webhook) }}" method="POST"
                          onsubmit="return confirm('Are you sure? This will invalidate the old webhook URL and you will need to update it in your payment platform.');">
                        @csrf
                        <button type="submit"
                                class="px-4 py-2 bg-yellow-600 text-white rounded-md hover:bg-yellow-700 text-sm">
                            Regenerate Token
                        </button>
                        <p class="mt-2 text-xs text-gray-500 dark:text-gray-400">
                            Generate a new webhook URL. This will invalidate the current URL.
                        </p>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Recent Webhook Logs -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow">
        <div class="p-6 border-b border-gray-200 dark:border-gray-700">
            <h2 class="text-lg font-semibold">Recent Webhook Requests</h2>
            <p class="text-sm text-gray-600 dark:text-gray-400">Last 50 webhook requests received</p>
        </div>

        @if($recentLogs->count() > 0)
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider">Time</th>
                        <th class="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider">Event Type</th>
                        <th class="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider">Status</th>
                        <th class="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider">Order</th>
                        <th class="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider">IP Address</th>
                        <th class="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider">Details</th>
                    </tr>
                </thead>
                <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                    @foreach($recentLogs as $log)
                    <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                        <td class="px-6 py-4 whitespace-nowrap text-sm">
                            {{ $log->created_at->format('M d, H:i:s') }}
                            <div class="text-xs text-gray-500 dark:text-gray-400">
                                {{ $log->created_at->diffForHumans() }}
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">
                            <span class="px-2 py-1 rounded bg-gray-100 dark:bg-gray-700 text-xs">
                                {{ $log->event_type ?? 'unknown' }}
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            @if($log->status === 'success')
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                                    Success
                                </span>
                            @elseif($log->status === 'failed')
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                                    Failed
                                </span>
                            @elseif($log->status === 'ignored')
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200">
                                    Ignored
                                </span>
                            @else
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200">
                                    {{ ucfirst($log->status) }}
                                </span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">
                            @if($log->order)
                                <a href="{{ route('orders.show', $log->order) }}"
                                   class="text-blue-600 dark:text-blue-400 hover:underline">
                                    {{ $log->order->order_number }}
                                </a>
                            @else
                                <span class="text-gray-400">-</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                            {{ $log->ip_address ?? '-' }}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">
                            <button type="button"
                                    onclick="showLogDetails({{ $log->id }}, {{ json_encode($log->payload) }}, '{{ $log->error_message }}')"
                                    class="text-indigo-600 dark:text-indigo-400 hover:underline">
                                View
                            </button>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
        @else
        <div class="p-6 text-center">
            <p class="text-gray-500 dark:text-gray-400">No webhook requests received yet</p>
            <p class="text-sm text-gray-400 dark:text-gray-500 mt-2">
                Configure the webhook URL in your payment platform to start receiving orders
            </p>
        </div>
        @endif
    </div>
</div>

<!-- Log Details Modal -->
<div id="logModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-3/4 lg:w-1/2 shadow-lg rounded-md bg-white dark:bg-gray-800">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg font-semibold">Webhook Request Details</h3>
            <button onclick="closeLogModal()" class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200">
                <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                </svg>
            </button>
        </div>
        <div class="space-y-4">
            <div id="errorMessage" class="hidden bg-red-50 dark:bg-red-900 p-4 rounded-md">
                <h4 class="text-sm font-semibold text-red-800 dark:text-red-200 mb-2">Error Message:</h4>
                <p class="text-sm text-red-700 dark:text-red-300" id="errorText"></p>
            </div>
            <div>
                <h4 class="text-sm font-semibold mb-2">Payload:</h4>
                <pre class="bg-gray-100 dark:bg-gray-900 p-4 rounded-md overflow-x-auto text-xs" id="payloadContent"></pre>
            </div>
        </div>
    </div>
</div>

<script>
function copyWebhookUrl() {
    const input = document.getElementById('webhookUrl');
    input.select();
    input.setSelectionRange(0, 99999);
    document.execCommand('copy');

    alert('Webhook URL copied to clipboard!');
}

function showLogDetails(logId, payload, errorMessage) {
    document.getElementById('payloadContent').textContent = JSON.stringify(payload, null, 2);

    if (errorMessage) {
        document.getElementById('errorMessage').classList.remove('hidden');
        document.getElementById('errorText').textContent = errorMessage;
    } else {
        document.getElementById('errorMessage').classList.add('hidden');
    }

    document.getElementById('logModal').classList.remove('hidden');
}

function closeLogModal() {
    document.getElementById('logModal').classList.add('hidden');
}

// Close modal when clicking outside
document.getElementById('logModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeLogModal();
    }
});
</script>
@endsection
