@extends('layouts.app')

@section('title', 'System Settings')

@section('content')
<div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
    <div class="mb-6">
        <h1 class="text-2xl font-bold text-gray-900 dark:text-white">System Settings</h1>
        <p class="text-gray-600 dark:text-gray-400 mt-1">Configure global system preferences</p>
    </div>

    <div class="bg-white dark:bg-gray-800 shadow-md rounded-lg overflow-hidden">
        <form action="{{ route('settings.update') }}" method="POST" enctype="multipart/form-data">
            @csrf
            @method('PUT')

            <!-- Company Information Source Section -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            Company Information Source
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Choose which company information to display on invoices and shipping labels (AWB).
                        </p>
                    </div>
                </div>

                <div class="space-y-4 mt-6">
                    <!-- Seller Option -->
                    <label class="relative flex items-start p-4 border-2 rounded-lg cursor-pointer transition-all {{ $companyInfoSource === 'seller' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500' }}">
                        <input type="radio"
                               name="company_info_source"
                               value="seller"
                               {{ $companyInfoSource === 'seller' ? 'checked' : '' }}
                               class="mt-1 h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500">
                        <div class="ml-3 flex-1">
                            <div class="flex items-center gap-2">
                                <span class="font-semibold text-gray-900 dark:text-white">Use Seller Company Information</span>
                                <span class="px-2 py-0.5 text-xs font-medium bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-300 rounded-full">
                                    Multi-Tenant
                                </span>
                            </div>
                            <p class="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                Each seller's company information will be used on their own invoices and AWB labels.
                                Perfect for <strong>multi-tenant</strong> setups where different sellers manage their own stores.
                            </p>
                            <div class="mt-3 p-3 bg-gray-50 dark:bg-gray-700/50 rounded-md">
                                <p class="text-xs text-gray-600 dark:text-gray-400">
                                    <strong>Example:</strong> If "Seller A" owns Store X, their company name and details will appear on all invoices from Store X.
                                </p>
                            </div>
                        </div>
                    </label>

                    <!-- Admin Option -->
                    <label class="relative flex items-start p-4 border-2 rounded-lg cursor-pointer transition-all {{ $companyInfoSource === 'admin' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500' }}">
                        <input type="radio"
                               name="company_info_source"
                               value="admin"
                               {{ $companyInfoSource === 'admin' ? 'checked' : '' }}
                               class="mt-1 h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500">
                        <div class="ml-3 flex-1">
                            <div class="flex items-center gap-2">
                                <span class="font-semibold text-gray-900 dark:text-white">Use Admin Company Information</span>
                                <span class="px-2 py-0.5 text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300 rounded-full">
                                    Single Organization
                                </span>
                            </div>
                            <p class="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                The admin/system company information will be used on all invoices and AWB labels regardless of which seller or store.
                                Perfect for <strong>single organization</strong> managing multiple stores under one brand.
                            </p>
                            <div class="mt-3 p-3 bg-gray-50 dark:bg-gray-700/50 rounded-md">
                                <p class="text-xs text-gray-600 dark:text-gray-400">
                                    <strong>Example:</strong> Your main company "ABC Corporation" will appear on all invoices from all stores.
                                </p>
                            </div>
                        </div>
                    </label>
                </div>

                <!-- Info Box -->
                <div class="mt-6 p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
                    <div class="flex gap-3">
                        <svg class="w-5 h-5 text-blue-600 dark:text-blue-400 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
                        </svg>
                        <div class="flex-1">
                            <h4 class="text-sm font-semibold text-blue-900 dark:text-blue-200 mb-1">Important Note</h4>
                            <p class="text-sm text-blue-800 dark:text-blue-300">
                                Make sure to update company information in your profile settings. This setting only controls
                                <strong>which</strong> company information to use, not the information itself.
                            </p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Server DateTime/Timezone Section -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            Server DateTime / Timezone
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Set the timezone for your application. This is especially useful if you're using an offshore server in a different timezone.
                        </p>
                    </div>
                </div>

                <div class="mt-6 max-w-2xl">
                    <label for="app_timezone" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Application Timezone <span class="text-red-500">*</span>
                    </label>
                    <select id="app_timezone"
                            name="app_timezone"
                            required
                            class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-blue-500 focus:border-blue-500">
                        @php
                            $currentTimezone = old('app_timezone', $appTimezone);
                            $timezones = [
                                'Asia/Kuala_Lumpur' => 'Asia/Kuala Lumpur (Malaysia) - UTC+8',
                                'Asia/Singapore' => 'Asia/Singapore - UTC+8',
                                'Asia/Jakarta' => 'Asia/Jakarta (Indonesia) - UTC+7',
                                'Asia/Bangkok' => 'Asia/Bangkok (Thailand) - UTC+7',
                                'Asia/Hong_Kong' => 'Asia/Hong Kong - UTC+8',
                                'Asia/Manila' => 'Asia/Manila (Philippines) - UTC+8',
                                'Asia/Tokyo' => 'Asia/Tokyo (Japan) - UTC+9',
                                'Asia/Dubai' => 'Asia/Dubai (UAE) - UTC+4',
                                'Europe/London' => 'Europe/London (UK) - UTC+0/+1',
                                'Europe/Paris' => 'Europe/Paris (France) - UTC+1/+2',
                                'America/New_York' => 'America/New York (US East) - UTC-5/-4',
                                'America/Los_Angeles' => 'America/Los Angeles (US West) - UTC-8/-7',
                                'America/Chicago' => 'America/Chicago (US Central) - UTC-6/-5',
                                'Australia/Sydney' => 'Australia/Sydney - UTC+10/+11',
                                'UTC' => 'UTC (Universal Time) - UTC+0',
                            ];
                        @endphp
                        @foreach($timezones as $value => $label)
                            <option value="{{ $value }}" {{ $currentTimezone === $value ? 'selected' : '' }}>
                                {{ $label }}
                            </option>
                        @endforeach
                    </select>
                    <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                        Select the timezone that matches your business location. All dates and times will be displayed in this timezone.
                    </p>
                </div>

                <!-- Current Server Time Display -->
                <div class="mt-6 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                    <div class="flex items-start gap-3">
                        <svg class="w-5 h-5 text-blue-600 dark:text-blue-400 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clip-rule="evenodd"/>
                        </svg>
                        <div class="flex-1">
                            <p class="text-sm font-semibold text-blue-900 dark:text-blue-200 mb-1">Current Server Time</p>
                            <p class="text-sm text-blue-800 dark:text-blue-300">
                                <strong>{{ now()->format('l, F j, Y - H:i:s (T)') }}</strong>
                            </p>
                            <p class="text-xs text-blue-700 dark:text-blue-400 mt-1">
                                Timezone: {{ config('app.timezone') }}
                            </p>
                        </div>
                    </div>
                </div>

                <!-- Warning for offshore servers -->
                <div class="mt-4 p-4 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg">
                    <div class="flex gap-3">
                        <svg class="w-5 h-5 text-yellow-600 dark:text-yellow-400 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                        </svg>
                        <div class="flex-1">
                            <p class="text-sm font-semibold text-yellow-900 dark:text-yellow-200 mb-1">Offshore Server Note</p>
                            <p class="text-sm text-yellow-800 dark:text-yellow-300">
                                If you're using an offshore server (e.g., USA, Europe) but your business is in Malaysia,
                                set the timezone to "Asia/Kuala Lumpur" so all order timestamps match your local business hours.
                            </p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Global ID Prefix Section -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            Global Order ID Prefix
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Set a custom prefix for all global order IDs (OMS ID) in your system.
                        </p>
                    </div>
                </div>

                <div class="mt-6 max-w-md">
                    <label for="global_id_prefix" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Prefix <span class="text-red-500">*</span>
                    </label>
                    <input type="text"
                           id="global_id_prefix"
                           name="global_id_prefix"
                           value="{{ old('global_id_prefix', $globalIdPrefix) }}"
                           required
                           maxlength="10"
                           pattern="[A-Za-z0-9_-]+"
                           class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-blue-500 focus:border-blue-500"
                           placeholder="e.g., OMS, MMD, SHOP">
                    @error('global_id_prefix')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                    <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                        Use 2-10 characters. Letters, numbers, dash and underscore only. Will be converted to uppercase.
                    </p>
                </div>

                <!-- Preview -->
                <div class="mt-6 p-4 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                    <p class="text-xs text-gray-600 dark:text-gray-400 mb-2"><strong>Preview:</strong></p>
                    <div class="flex items-center gap-2">
                        <code class="px-3 py-1.5 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded font-mono text-sm">
                            <span id="prefix-preview">{{ strtoupper($globalIdPrefix) }}</span>25100001
                        </code>
                        <span class="text-xs text-gray-500">(Example Global ID)</span>
                    </div>
                </div>
            </div>

            <!-- Branding Section -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            Branding Settings
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Customize your system's appearance and branding.
                        </p>
                    </div>
                </div>

                <div class="space-y-6">
                    <!-- Site Title -->
                    <div>
                        <label for="site_title" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Site Title <span class="text-red-500">*</span>
                        </label>
                        <input type="text"
                               id="site_title"
                               name="site_title"
                               value="{{ old('site_title', $siteTitle) }}"
                               required
                               maxlength="100"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Omni Management System">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                            This will appear in the browser tab and as the main site title.
                        </p>
                    </div>

                    <!-- Site Name -->
                    <div>
                        <label for="site_name" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Site Name <span class="text-red-500">*</span>
                        </label>
                        <input type="text"
                               id="site_name"
                               name="site_name"
                               value="{{ old('site_name', $siteName) }}"
                               required
                               maxlength="50"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Omni">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                            Short name displayed in the header/logo area and welcome page.
                        </p>
                    </div>

                    <!-- Site Logo -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Logo (optional)
                        </label>

                        <!-- Current Logo Preview -->
                        @if($siteLogo)
                            <div class="mb-3 p-3 bg-gray-50 dark:bg-gray-700/50 rounded-md">
                                <p class="text-xs text-gray-600 dark:text-gray-400 mb-2"><strong>Current Logo:</strong></p>
                                <img src="{{ $siteLogo }}" alt="Logo" class="h-12 object-contain mb-2">
                                <label class="inline-flex items-center text-xs cursor-pointer">
                                    <input type="checkbox" name="remove_logo" value="1" class="rounded border-gray-300 text-red-600 focus:ring-red-500 mr-2">
                                    <span class="text-red-600 dark:text-red-400">Remove current logo</span>
                                </label>
                            </div>
                        @endif

                        <!-- Upload New Logo -->
                        <div class="mb-3">
                            <label for="logo_file" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Upload New Logo
                            </label>
                            <input type="file"
                                   id="logo_file"
                                   name="logo_file"
                                   accept="image/png,image/jpeg,image/jpg,image/gif,image/svg+xml,image/webp"
                                   class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-blue-500 focus:border-blue-500">
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                                Supported formats: PNG, JPG, GIF, SVG, WebP. Max size: 2MB. Recommended: 200-400px width.
                            </p>
                        </div>

                        <!-- Or Enter URL -->
                        <div>
                            <label for="site_logo" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Or Enter Logo URL
                            </label>
                            <input type="text"
                                   id="site_logo"
                                   name="site_logo"
                                   value="{{ old('site_logo', $siteLogo) }}"
                                   maxlength="500"
                                   class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="https://example.com/logo.png">
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                                Enter full URL if logo is hosted elsewhere. Upload takes priority over URL.
                            </p>
                        </div>
                    </div>

                    <!-- Login Description -->
                    <div>
                        <label for="login_description" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Login Page Description <span class="text-red-500">*</span>
                        </label>
                        <input type="text"
                               id="login_description"
                               name="login_description"
                               value="{{ old('login_description', $loginDescription) }}"
                               required
                               maxlength="255"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-blue-500 focus:border-blue-500"
                               placeholder="Welcome to Omni Management System">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                            This message appears on the login page.
                        </p>
                    </div>

                    <!-- Footer Text -->
                    <div>
                        <label for="footer_text" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Footer Text <span class="text-red-500">*</span>
                        </label>
                        <input type="text"
                               id="footer_text"
                               name="footer_text"
                               value="{{ old('footer_text', $footerText) }}"
                               required
                               maxlength="255"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-blue-500 focus:border-blue-500"
                               placeholder="© 2025 Omni. All rights reserved.">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                            This appears at the bottom of every page.
                        </p>
                    </div>
                </div>
            </div>

            <!-- Registration Setting -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            Registration Settings
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Control who can register new accounts.
                        </p>
                    </div>
                </div>

                <div class="space-y-4">
                    <!-- Enable Registration -->
                    <label class="relative flex items-start p-4 border-2 rounded-lg cursor-pointer transition-all {{ $registrationEnabled === 'true' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500' }}">
                        <input type="radio"
                               name="registration_enabled"
                               value="true"
                               {{ $registrationEnabled === 'true' ? 'checked' : '' }}
                               class="mt-1 h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500">
                        <div class="ml-3 flex-1">
                            <div class="flex items-center gap-2">
                                <span class="font-semibold text-gray-900 dark:text-white">Enable Registration</span>
                                <span class="px-2 py-0.5 text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300 rounded-full">
                                    Public
                                </span>
                            </div>
                            <p class="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                Allow users to register new accounts through the registration page.
                            </p>
                        </div>
                    </label>

                    <!-- Disable Registration -->
                    <label class="relative flex items-start p-4 border-2 rounded-lg cursor-pointer transition-all {{ $registrationEnabled === 'false' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500' }}">
                        <input type="radio"
                               name="registration_enabled"
                               value="false"
                               {{ $registrationEnabled === 'false' ? 'checked' : '' }}
                               class="mt-1 h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500">
                        <div class="ml-3 flex-1">
                            <div class="flex items-center gap-2">
                                <span class="font-semibold text-gray-900 dark:text-white">Disable Registration</span>
                                <span class="px-2 py-0.5 text-xs font-medium bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-300 rounded-full">
                                    Admin Only
                                </span>
                            </div>
                            <p class="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                Only administrators can create new user accounts directly in the database.
                            </p>
                        </div>
                    </label>
                </div>
            </div>

            <!-- COD Shipping Rates Section -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            COD Shipping Rates (Checkout Form)
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Set default COD shipping charges for orders created via Checkout Form. These rates will auto-populate based on customer's state.
                        </p>
                    </div>
                </div>

                <div class="space-y-4 mt-6">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <!-- East Malaysia + Labuan Rate -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                East Malaysia Rate (RM)
                                <span class="text-xs font-normal text-gray-500">(Sabah, Sarawak, Labuan)</span>
                            </label>
                            <input type="number"
                                   name="cod_rate_east_malaysia"
                                   value="{{ old('cod_rate_east_malaysia', \App\Models\SystemSetting::get('cod_rate_east_malaysia', '20.00')) }}"
                                   step="0.01"
                                   min="0"
                                   required
                                   class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Applied to: Sabah, Sarawak, Wp Labuan</p>
                        </div>

                        <!-- Peninsular Malaysia Rate -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Peninsular Malaysia Rate (RM)
                                <span class="text-xs font-normal text-gray-500">(All other states)</span>
                            </label>
                            <input type="number"
                                   name="cod_rate_peninsular"
                                   value="{{ old('cod_rate_peninsular', \App\Models\SystemSetting::get('cod_rate_peninsular', '10.00')) }}"
                                   step="0.01"
                                   min="0"
                                   required
                                   class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Applied to: All other Malaysian states</p>
                        </div>
                    </div>

                    <div class="mt-4 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                        <p class="text-sm text-blue-800 dark:text-blue-300">
                            <strong>Note:</strong> These rates only apply to orders created via the Checkout Form. Orders from WooCommerce stores and webhooks use their own shipping rates.
                        </p>
                    </div>
                </div>
            </div>

            <!-- Shipping Cost Estimation Section -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            Shipping Cost Estimation
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Configure shipping cost per order for analytics and cost estimation. COD orders will use the COD rates above (based on state), while online payment orders (Online Banking, DuitNow QR) will use the rate below.
                        </p>
                    </div>
                </div>

                <div class="space-y-4 mt-6">
                    <div class="max-w-md">
                        <!-- Online Payment Shipping Cost -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Online Payment Shipping Cost (RM)
                                <span class="text-xs font-normal text-gray-500">(Online Banking, DuitNow QR)</span>
                            </label>
                            <input type="number"
                                   name="online_payment_shipping_cost"
                                   value="{{ old('online_payment_shipping_cost', $onlinePaymentShippingCost) }}"
                                   step="0.01"
                                   min="0"
                                   required
                                   class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                Applied to orders with tracking numbers using Online Banking or DuitNow QR payment methods
                            </p>
                        </div>
                    </div>

                    <div class="mt-4 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                        <div class="flex gap-2">
                            <svg class="w-5 h-5 text-blue-600 dark:text-blue-400 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
                            </svg>
                            <div>
                                <p class="text-sm text-blue-800 dark:text-blue-300 font-semibold mb-1">How This Works:</p>
                                <ul class="text-sm text-blue-800 dark:text-blue-300 space-y-1 list-disc list-inside">
                                    <li><strong>COD Orders:</strong> Use state-based rates (East Malaysia: RM{{ number_format($codRateEastMalaysia, 2) }}, Peninsular: RM{{ number_format($codRatePeninsular, 2) }})</li>
                                    <li><strong>Online Payment Orders:</strong> Use the flat rate above (RM{{ number_format($onlinePaymentShippingCost, 2) }} per order)</li>
                                    <li><strong>Only counted:</strong> Orders with tracking numbers (excludes failed/cancelled orders)</li>
                                    <li><strong>Displayed in:</strong> Dashboard Analytics section for cost estimation</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- BayarCash Payment Gateway Section -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            BayarCash Payment Gateway
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Configure BayarCash payment gateway for FPX (Online Banking) and DuitNow QR payments on sales pages.
                        </p>
                    </div>
                </div>

                <div class="space-y-4 mt-6">
                    <!-- Environment -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                            Environment <span class="text-red-500">*</span>
                        </label>
                        <div class="flex gap-4">
                            <label class="flex items-center gap-2 p-3 border-2 rounded-lg cursor-pointer transition-all {{ $bayarcashEnvironment === 'sandbox' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400' }}">
                                <input type="radio"
                                       name="bayarcash_environment"
                                       value="sandbox"
                                       {{ $bayarcashEnvironment === 'sandbox' ? 'checked' : '' }}
                                       class="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500">
                                <div>
                                    <div class="font-semibold text-gray-900 dark:text-white">Sandbox</div>
                                    <div class="text-xs text-gray-600 dark:text-gray-400">For testing only</div>
                                </div>
                            </label>
                            <label class="flex items-center gap-2 p-3 border-2 rounded-lg cursor-pointer transition-all {{ $bayarcashEnvironment === 'production' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400' }}">
                                <input type="radio"
                                       name="bayarcash_environment"
                                       value="production"
                                       {{ $bayarcashEnvironment === 'production' ? 'checked' : '' }}
                                       class="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500">
                                <div>
                                    <div class="font-semibold text-gray-900 dark:text-white">Production</div>
                                    <div class="text-xs text-gray-600 dark:text-gray-400">Live payments</div>
                                </div>
                            </label>
                        </div>
                    </div>

                    <!-- Portal Key -->
                    <div>
                        <label for="bayarcash_portal_key" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Portal Key
                        </label>
                        <input type="text"
                               id="bayarcash_portal_key"
                               name="bayarcash_portal_key"
                               value="{{ old('bayarcash_portal_key', $bayarcashPortalKey) }}"
                               maxlength="255"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 font-mono text-sm"
                               placeholder="Your BayarCash Portal Key">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Get this from your BayarCash account settings (max 255 characters)</p>
                    </div>

                    <!-- PAT -->
                    <div>
                        <label for="bayarcash_pat" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Personal Access Token (PAT)
                        </label>
                        <textarea
                               id="bayarcash_pat"
                               name="bayarcash_pat"
                               rows="3"
                               maxlength="1000"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 font-mono text-sm"
                               placeholder="Your BayarCash Personal Access Token">{{ old('bayarcash_pat', $bayarcashPat) }}</textarea>
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Get this from your BayarCash account settings (max 1000 characters)</p>
                    </div>

                    <!-- API Secret Key -->
                    <div>
                        <label for="bayarcash_secret_key" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            API Secret Key
                        </label>
                        <input type="text"
                               id="bayarcash_secret_key"
                               name="bayarcash_secret_key"
                               value="{{ old('bayarcash_secret_key', $bayarcashSecretKey) }}"
                               maxlength="255"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 font-mono text-sm"
                               placeholder="Your BayarCash API Secret Key">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Used for verifying payment callbacks (max 255 characters)</p>
                    </div>

                    <!-- Payment Gateway Fee Settings -->
                    <div class="mt-6 p-4 bg-gray-50 dark:bg-gray-900/50 rounded-lg border border-gray-200 dark:border-gray-700">
                        <h4 class="text-base font-semibold mb-3 text-gray-900 dark:text-white">Payment Gateway Fees (Default)</h4>
                        <p class="text-xs text-gray-600 dark:text-gray-400 mb-4">
                            Configure default fees for online payment methods. These fees will be used when sellers/managers haven't configured their own payment gateway settings.
                        </p>

                        <!-- FPX (Online Banking) Fees -->
                        <div class="mb-4 p-3 bg-blue-50 dark:bg-blue-900/10 rounded-lg border border-blue-200 dark:border-blue-800">
                            <h5 class="text-sm font-semibold mb-3 text-blue-900 dark:text-blue-300">FPX (Online Banking)</h5>

                            <div class="space-y-3">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Fee Type</label>
                                    <select name="fpx_fee_type" id="system-fpx-fee-type-selector"
                                            class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                                        @php
                                            $fpxFeeType = \App\Models\SystemSetting::get('fpx_fee_type', 'none');
                                        @endphp
                                        <option value="none" {{ old('fpx_fee_type', $fpxFeeType) === 'none' ? 'selected' : '' }}>No Fee</option>
                                        <option value="fixed" {{ old('fpx_fee_type', $fpxFeeType) === 'fixed' ? 'selected' : '' }}>Fixed Amount</option>
                                        <option value="percentage" {{ old('fpx_fee_type', $fpxFeeType) === 'percentage' ? 'selected' : '' }}>Percentage</option>
                                        <option value="both" {{ old('fpx_fee_type', $fpxFeeType) === 'both' ? 'selected' : '' }}>Both (Fixed + Percentage)</option>
                                    </select>
                                </div>

                                <div id="system-fpx-fee-amount-fields" class="space-y-3">
                                    <div id="system-fpx-fixed-fee-field" class="hidden">
                                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Fixed Fee Amount (MYR)</label>
                                        <input type="number" name="fpx_fixed_fee" step="0.01" min="0"
                                               value="{{ old('fpx_fixed_fee', \App\Models\SystemSetting::get('fpx_fixed_fee', 0)) }}"
                                               class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 text-sm"
                                               placeholder="0.00">
                                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">e.g., RM 1.50 per transaction</p>
                                    </div>

                                    <div id="system-fpx-percentage-fee-field" class="hidden">
                                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Percentage Fee (%)</label>
                                        <input type="number" name="fpx_percentage_fee" step="0.01" min="0" max="100"
                                               value="{{ old('fpx_percentage_fee', \App\Models\SystemSetting::get('fpx_percentage_fee', 0)) }}"
                                               class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 text-sm"
                                               placeholder="0.00">
                                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">e.g., 2.5 for 2.5%</p>
                                    </div>
                                </div>

                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Who Pays?</label>
                                    <div class="flex gap-3">
                                        @php
                                            $fpxFeeBearer = \App\Models\SystemSetting::get('fpx_fee_bearer', 'customer');
                                        @endphp
                                        <label class="flex-1 flex items-center gap-2 p-2 border-2 rounded-lg cursor-pointer transition-all {{ old('fpx_fee_bearer', $fpxFeeBearer) === 'customer' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400' }}">
                                            <input type="radio" name="fpx_fee_bearer" value="customer"
                                                   {{ old('fpx_fee_bearer', $fpxFeeBearer) === 'customer' ? 'checked' : '' }}
                                                   class="h-4 w-4 text-blue-600">
                                            <div class="text-xs">
                                                <div class="font-semibold text-gray-900 dark:text-white">Customer</div>
                                            </div>
                                        </label>
                                        <label class="flex-1 flex items-center gap-2 p-2 border-2 rounded-lg cursor-pointer transition-all {{ old('fpx_fee_bearer', $fpxFeeBearer) === 'merchant' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400' }}">
                                            <input type="radio" name="fpx_fee_bearer" value="merchant"
                                                   {{ old('fpx_fee_bearer', $fpxFeeBearer) === 'merchant' ? 'checked' : '' }}
                                                   class="h-4 w-4 text-blue-600">
                                            <div class="text-xs">
                                                <div class="font-semibold text-gray-900 dark:text-white">Merchant</div>
                                            </div>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- DuitNow QR Fees -->
                        <div class="p-3 bg-purple-50 dark:bg-purple-900/10 rounded-lg border border-purple-200 dark:border-purple-800">
                            <h5 class="text-sm font-semibold mb-3 text-purple-900 dark:text-purple-300">DuitNow QR</h5>

                            <div class="space-y-3">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Fee Type</label>
                                    <select name="duitnow_fee_type" id="system-duitnow-fee-type-selector"
                                            class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                                        @php
                                            $duitnowFeeType = \App\Models\SystemSetting::get('duitnow_fee_type', 'none');
                                        @endphp
                                        <option value="none" {{ old('duitnow_fee_type', $duitnowFeeType) === 'none' ? 'selected' : '' }}>No Fee</option>
                                        <option value="fixed" {{ old('duitnow_fee_type', $duitnowFeeType) === 'fixed' ? 'selected' : '' }}>Fixed Amount</option>
                                        <option value="percentage" {{ old('duitnow_fee_type', $duitnowFeeType) === 'percentage' ? 'selected' : '' }}>Percentage</option>
                                        <option value="both" {{ old('duitnow_fee_type', $duitnowFeeType) === 'both' ? 'selected' : '' }}>Both (Fixed + Percentage)</option>
                                    </select>
                                </div>

                                <div id="system-duitnow-fee-amount-fields" class="space-y-3">
                                    <div id="system-duitnow-fixed-fee-field" class="hidden">
                                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Fixed Fee Amount (MYR)</label>
                                        <input type="number" name="duitnow_fixed_fee" step="0.01" min="0"
                                               value="{{ old('duitnow_fixed_fee', \App\Models\SystemSetting::get('duitnow_fixed_fee', 0)) }}"
                                               class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 text-sm"
                                               placeholder="0.00">
                                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">e.g., RM 1.00 per transaction</p>
                                    </div>

                                    <div id="system-duitnow-percentage-fee-field" class="hidden">
                                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Percentage Fee (%)</label>
                                        <input type="number" name="duitnow_percentage_fee" step="0.01" min="0" max="100"
                                               value="{{ old('duitnow_percentage_fee', \App\Models\SystemSetting::get('duitnow_percentage_fee', 0)) }}"
                                               class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 text-sm"
                                               placeholder="0.00">
                                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">e.g., 2.0 for 2%</p>
                                    </div>
                                </div>

                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Who Pays?</label>
                                    <div class="flex gap-3">
                                        @php
                                            $duitnowFeeBearer = \App\Models\SystemSetting::get('duitnow_fee_bearer', 'customer');
                                        @endphp
                                        <label class="flex-1 flex items-center gap-2 p-2 border-2 rounded-lg cursor-pointer transition-all {{ old('duitnow_fee_bearer', $duitnowFeeBearer) === 'customer' ? 'border-purple-500 bg-purple-50 dark:bg-purple-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400' }}">
                                            <input type="radio" name="duitnow_fee_bearer" value="customer"
                                                   {{ old('duitnow_fee_bearer', $duitnowFeeBearer) === 'customer' ? 'checked' : '' }}
                                                   class="h-4 w-4 text-purple-600">
                                            <div class="text-xs">
                                                <div class="font-semibold text-gray-900 dark:text-white">Customer</div>
                                            </div>
                                        </label>
                                        <label class="flex-1 flex items-center gap-2 p-2 border-2 rounded-lg cursor-pointer transition-all {{ old('duitnow_fee_bearer', $duitnowFeeBearer) === 'merchant' ? 'border-purple-500 bg-purple-50 dark:bg-purple-900/20' : 'border-gray-300 dark:border-gray-600 hover:border-gray-400' }}">
                                            <input type="radio" name="duitnow_fee_bearer" value="merchant"
                                                   {{ old('duitnow_fee_bearer', $duitnowFeeBearer) === 'merchant' ? 'checked' : '' }}
                                                   class="h-4 w-4 text-purple-600">
                                            <div class="text-xs">
                                                <div class="font-semibold text-gray-900 dark:text-white">Merchant</div>
                                            </div>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="mt-4 p-3 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-800">
                        <p class="text-sm text-yellow-800 dark:text-yellow-300">
                            <strong>Important:</strong> Keep your API credentials secure. Never share them publicly. Use sandbox mode for testing before going live.
                        </p>
                    </div>
                </div>
            </div>

            <!-- PWA Push Notification Settings -->
            <div class="p-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2 flex items-center gap-2">
                            <svg class="w-5 h-5 text-indigo-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"/>
                            </svg>
                            PWA Push Notifications
                        </h3>
                        <p class="text-sm text-gray-600 dark:text-gray-400">
                            Configure browser push notifications for new orders. Customize the message template, emojis, and sound settings.
                        </p>
                    </div>
                </div>

                <div class="space-y-6 mt-6">
                    <!-- Enable/Disable Notifications -->
                    <div>
                        <label class="flex items-center gap-3">
                            <input type="checkbox"
                                   name="pwa_notifications_enabled"
                                   value="true"
                                   {{ $pwaNotificationsEnabled === 'true' ? 'checked' : '' }}
                                   class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                            <span class="text-sm font-medium text-gray-700 dark:text-gray-300">Enable Push Notifications</span>
                        </label>
                        <p class="ml-7 text-xs text-gray-500 dark:text-gray-400 mt-1">When enabled, users will receive browser notifications for new orders</p>
                    </div>

                    <!-- Notification Title Template -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Notification Title Template
                        </label>
                        <input type="text"
                               name="pwa_notification_title"
                               value="{{ $pwaNotificationTitle }}"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                               placeholder="{emoji} Kaching! Pesanan #{order_number}">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                            Available variables: <code class="bg-gray-100 dark:bg-gray-700 px-1 py-0.5 rounded">{emoji}</code>,
                            <code class="bg-gray-100 dark:bg-gray-700 px-1 py-0.5 rounded">{order_number}</code>
                        </p>
                    </div>

                    <!-- Notification Body Template -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Notification Body Template
                        </label>
                        <input type="text"
                               name="pwa_notification_body"
                               value="{{ $pwaNotificationBody }}"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                               placeholder="{total} • {customer} • {platform}">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">
                            Available variables: <code class="bg-gray-100 dark:bg-gray-700 px-1 py-0.5 rounded">{total}</code>,
                            <code class="bg-gray-100 dark:bg-gray-700 px-1 py-0.5 rounded">{customer}</code>,
                            <code class="bg-gray-100 dark:bg-gray-700 px-1 py-0.5 rounded">{platform}</code>
                        </p>
                    </div>

                    <!-- Emoji Settings -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Regular Order Emoji
                            </label>
                            <input type="text"
                                   name="pwa_regular_emoji"
                                   value="{{ $pwaRegularEmoji }}"
                                   class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                   placeholder="💰"
                                   maxlength="10">
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Emoji for orders below threshold</p>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                High Value Order Emoji
                            </label>
                            <input type="text"
                                   name="pwa_high_value_emoji"
                                   value="{{ $pwaHighValueEmoji }}"
                                   class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                   placeholder="🎉"
                                   maxlength="10">
                            <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Emoji for high-value orders</p>
                        </div>
                    </div>

                    <!-- High Value Threshold -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            High Value Threshold (RM)
                        </label>
                        <input type="number"
                               name="pwa_high_value_threshold"
                               value="{{ $pwaHighValueThreshold }}"
                               min="0"
                               max="100000"
                               step="50"
                               class="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                               placeholder="500">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Orders with total >= this amount will use the high value emoji (🎉 instead of 💰)</p>
                    </div>

                    <!-- Notification Volume -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Notification Sound Volume: <span id="volumeValue">{{ $pwaNotificationVolume }}%</span>
                        </label>
                        <input type="range"
                               name="pwa_notification_volume"
                               value="{{ $pwaNotificationVolume }}"
                               min="0"
                               max="100"
                               step="5"
                               id="volumeSlider"
                               class="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer dark:bg-gray-700"
                               onchange="document.getElementById('volumeValue').textContent = this.value + '%'">
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">Adjust the volume of the "Kaching!" notification sound (0-100%)</p>
                    </div>

                    <!-- Preview Notification Button -->
                    <div class="pt-4 border-t border-gray-200 dark:border-gray-700">
                        <button type="button"
                                onclick="previewNotification()"
                                class="inline-flex items-center gap-2 px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 shadow-sm hover:shadow-md transition-all text-sm font-medium">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z"/>
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                            Preview Notification
                        </button>
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">Test how your notification will look and sound</p>
                    </div>

                    <!-- Info Box -->
                    <div class="bg-indigo-50 dark:bg-indigo-900/20 border border-indigo-200 dark:border-indigo-800 rounded-lg p-4">
                        <div class="flex gap-3">
                            <svg class="w-5 h-5 text-indigo-600 dark:text-indigo-400 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
                            </svg>
                            <div class="flex-1">
                                <p class="text-sm text-indigo-900 dark:text-indigo-200 font-medium mb-1">Notification Sound Files</p>
                                <p class="text-xs text-indigo-700 dark:text-indigo-300">
                                    Sound files should be placed in <code class="bg-indigo-100 dark:bg-indigo-800 px-1 py-0.5 rounded">/public/sounds/</code> directory.
                                    Currently using: <strong>kaching.mp3</strong>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Submit Button -->
            <div class="p-6 bg-gray-50 dark:bg-gray-700/50 flex justify-end">
                <button type="submit"
                        class="inline-flex items-center gap-2 px-5 py-2.5 bg-blue-600 text-white rounded-lg hover:bg-blue-700 shadow-md hover:shadow-lg transition-all font-medium">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                    </svg>
                    Save Settings
                </button>
            </div>
        </form>
    </div>

    <!-- Maintenance Mode Section (Admin Only) -->
    <div class="mt-6 bg-white dark:bg-gray-800 shadow-md rounded-lg overflow-hidden">
        <div class="p-6 border-b border-gray-200 dark:border-gray-700">
            <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                Maintenance Mode
            </h3>
            <p class="text-sm text-gray-600 dark:text-gray-400">
                Put the site in maintenance mode during disaster recovery or system updates. Admins can still access the site while maintenance mode is active.
            </p>
        </div>

        <div class="p-6">
            @if($maintenanceMode)
                <!-- Site is DOWN -->
                <div class="flex items-center justify-between p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg mb-6">
                    <div class="flex items-center">
                        <div class="flex-shrink-0">
                            <svg class="w-8 h-8 text-red-500" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                            </svg>
                        </div>
                        <div class="ml-4">
                            <h3 class="text-lg font-semibold text-red-900 dark:text-red-100">
                                Site is DOWN
                            </h3>
                            <p class="text-sm text-red-700 dark:text-red-300">
                                Maintenance mode is currently active. Regular users cannot access the site.
                            </p>
                        </div>
                    </div>
                </div>

                <div class="space-y-4">
                    <div class="p-4 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg">
                        <div class="flex gap-3">
                            <svg class="w-5 h-5 text-yellow-600 dark:text-yellow-400 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                            </svg>
                            <div class="flex-1">
                                <h4 class="text-sm font-semibold text-yellow-900 dark:text-yellow-200 mb-1">Important</h4>
                                <p class="text-sm text-yellow-800 dark:text-yellow-300">
                                    Only bring the site back up after completing all disaster recovery steps (restore database, import orders, fix sequences).
                                </p>
                            </div>
                        </div>
                    </div>

                    <form action="{{ route('settings.maintenance.disable') }}" method="POST" onsubmit="return confirm('Are you sure you want to bring the site back online? Make sure all recovery steps are completed.');">
                        @csrf
                        <button type="submit"
                                class="inline-flex items-center px-4 py-2 bg-green-600 hover:bg-green-700 text-white font-medium rounded-lg transition">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                            </svg>
                            Bring Site Back Online
                        </button>
                    </form>
                </div>

            @else
                <!-- Site is UP -->
                <div class="flex items-center justify-between p-4 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg mb-6">
                    <div class="flex items-center">
                        <div class="flex-shrink-0">
                            <svg class="w-8 h-8 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                            </svg>
                        </div>
                        <div class="ml-4">
                            <h3 class="text-lg font-semibold text-green-900 dark:text-green-100">
                                Site is UP
                            </h3>
                            <p class="text-sm text-green-700 dark:text-green-300">
                                The site is currently accessible to all users.
                            </p>
                        </div>
                    </div>
                </div>

                <div class="space-y-4">
                    <div class="p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
                        <div class="flex gap-3">
                            <svg class="w-5 h-5 text-blue-600 dark:text-blue-400 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
                            </svg>
                            <div class="flex-1">
                                <h4 class="text-sm font-semibold text-blue-900 dark:text-blue-200 mb-1">When to use Maintenance Mode</h4>
                                <ul class="text-sm text-blue-800 dark:text-blue-300 space-y-1 list-disc list-inside">
                                    <li>Before restoring database from backup</li>
                                    <li>During disaster recovery (importing orders from Google Sheets)</li>
                                    <li>When performing critical system updates</li>
                                    <li>To prevent order conflicts during data fixes</li>
                                </ul>
                            </div>
                        </div>
                    </div>

                    <form action="{{ route('settings.maintenance.enable') }}" method="POST" onsubmit="return confirm('This will make the site inaccessible to regular users. Only admins can access the site. Continue?');">
                        @csrf
                        <button type="submit"
                                class="inline-flex items-center px-4 py-2 bg-red-600 hover:bg-red-700 text-white font-medium rounded-lg transition">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                            </svg>
                            Enable Maintenance Mode
                        </button>
                    </form>
                </div>
            @endif
        </div>
    </div>

    <!-- Additional Help Section -->
    <div class="mt-6 p-4 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg">
        <h4 class="text-sm font-semibold text-yellow-900 dark:text-yellow-200 mb-2">Quick Guide</h4>
        <ul class="text-sm text-yellow-800 dark:text-yellow-300 space-y-1 list-disc list-inside">
            <li><strong>Multi-Tenant Mode:</strong> Each seller has their own branding on documents</li>
            <li><strong>Single Organization Mode:</strong> All documents show your main company brand</li>
            <li>Changes take effect immediately on new invoices and AWB labels</li>
            <li>Existing documents won't be affected by this change</li>
        </ul>
    </div>
</div>

@push('scripts')
<script>
document.getElementById('global_id_prefix').addEventListener('input', function(e) {
    const preview = document.getElementById('prefix-preview');
    preview.textContent = e.target.value.toUpperCase() || 'OMS';
});

// Toggle FPX fee amount fields based on fee type selection (System Settings)
const systemFpxFeeTypeSelector = document.getElementById('system-fpx-fee-type-selector');
const systemFpxFixedFeeField = document.getElementById('system-fpx-fixed-fee-field');
const systemFpxPercentageFeeField = document.getElementById('system-fpx-percentage-fee-field');

function updateSystemFpxFeeFields() {
    const feeType = systemFpxFeeTypeSelector.value;

    // Hide all fields first
    systemFpxFixedFeeField.classList.add('hidden');
    systemFpxPercentageFeeField.classList.add('hidden');

    // Show fields based on selected type
    if (feeType === 'fixed' || feeType === 'both') {
        systemFpxFixedFeeField.classList.remove('hidden');
    }
    if (feeType === 'percentage' || feeType === 'both') {
        systemFpxPercentageFeeField.classList.remove('hidden');
    }
}

if (systemFpxFeeTypeSelector) {
    // Initialize on page load
    updateSystemFpxFeeFields();

    // Update when selection changes
    systemFpxFeeTypeSelector.addEventListener('change', updateSystemFpxFeeFields);
}

// Toggle DuitNow fee amount fields based on fee type selection (System Settings)
const systemDuitnowFeeTypeSelector = document.getElementById('system-duitnow-fee-type-selector');
const systemDuitnowFixedFeeField = document.getElementById('system-duitnow-fixed-fee-field');
const systemDuitnowPercentageFeeField = document.getElementById('system-duitnow-percentage-fee-field');

function updateSystemDuitnowFeeFields() {
    const feeType = systemDuitnowFeeTypeSelector.value;

    // Hide all fields first
    systemDuitnowFixedFeeField.classList.add('hidden');
    systemDuitnowPercentageFeeField.classList.add('hidden');

    // Show fields based on selected type
    if (feeType === 'fixed' || feeType === 'both') {
        systemDuitnowFixedFeeField.classList.remove('hidden');
    }
    if (feeType === 'percentage' || feeType === 'both') {
        systemDuitnowPercentageFeeField.classList.remove('hidden');
    }
}

if (systemDuitnowFeeTypeSelector) {
    // Initialize on page load
    updateSystemDuitnowFeeFields();

    // Update when selection changes
    systemDuitnowFeeTypeSelector.addEventListener('change', updateSystemDuitnowFeeFields);
}

// PWA Notification Preview Function
function previewNotification() {
    // Get form values
    const title = document.querySelector('input[name="pwa_notification_title"]').value || '{emoji} Kaching! Pesanan #{order_number}';
    const body = document.querySelector('input[name="pwa_notification_body"]').value || '{total} • {customer} • {platform}';
    const regularEmoji = document.querySelector('input[name="pwa_regular_emoji"]').value || '💰';
    const highValueEmoji = document.querySelector('input[name="pwa_high_value_emoji"]').value || '🎉';
    const threshold = parseInt(document.querySelector('input[name="pwa_high_value_threshold"]').value || 500);
    const volume = parseInt(document.querySelector('input[name="pwa_notification_volume"]').value || 80) / 100;

    // Sample data for preview (high value order to show party emoji)
    const sampleTotal = 'RM888.00';
    const sampleCustomer = 'Ahmad Ibrahim';
    const samplePlatform = 'Checkout Form';
    const sampleOrderNumber = 'PREVIEW-' + Date.now();
    const isHighValue = 888 >= threshold;

    // Replace variables
    let finalTitle = title
        .replace('{emoji}', isHighValue ? highValueEmoji : regularEmoji)
        .replace('{order_number}', sampleOrderNumber);

    let finalBody = body
        .replace('{total}', sampleTotal)
        .replace('{customer}', sampleCustomer)
        .replace('{platform}', samplePlatform);

    // Check notification permission
    if (!('Notification' in window)) {
        alert('This browser does not support notifications');
        return;
    }

    if (Notification.permission === 'granted') {
        showPreview();
    } else if (Notification.permission !== 'denied') {
        Notification.requestPermission().then(permission => {
            if (permission === 'granted') {
                showPreview();
            }
        });
    } else {
        alert('Notifications are blocked. Please enable them in your browser settings.');
    }

    function showPreview() {
        // Show notification
        new Notification(finalTitle, {
            body: finalBody,
            icon: '/icons/icon-192.png',
            badge: '/icons/icon-72.png'
        });

        // Play sound at selected volume
        const audio = new Audio('/sounds/kaching.mp3');
        audio.volume = volume;
        audio.play()
            .then(() => console.log('[Preview] Sound played at ' + (volume * 100) + '% volume'))
            .catch(e => console.log('[Preview] Could not play sound:', e));
    }
}
</script>
@endpush
@endsection
