@extends('layouts.app')

@section('title', 'Profile Settings')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <h1 class="text-2xl font-bold mb-6">Profile Settings</h1>

    <!-- Success/Error Messages -->
    @if(session('success'))
        <div class="mb-6 p-4 bg-green-100 dark:bg-green-900/30 border border-green-400 dark:border-green-700 text-green-700 dark:text-green-300 rounded-lg">
            {{ session('success') }}
        </div>
    @endif

    @if(session('error'))
        <div class="mb-6 p-4 bg-red-100 dark:bg-red-900/30 border border-red-400 dark:border-red-700 text-red-700 dark:text-red-300 rounded-lg">
            {{ session('error') }}
        </div>
    @endif

    <!-- Personal Info & Password - Side by Side Grid -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">

        <!-- Personal Information Card -->
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
            <h2 class="text-lg font-semibold mb-6">Personal Information</h2>

            <form action="{{ route('profile.update') }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')

                <!-- Hidden fields to preserve company and payment info -->
                <input type="hidden" name="company_name" value="{{ $user->company_name ?? '' }}">
                <input type="hidden" name="company_registration" value="{{ $user->company_registration ?? '' }}">
                <input type="hidden" name="company_address" value="{{ $user->company_address ?? '' }}">
                <input type="hidden" name="company_city" value="{{ $user->company_city ?? '' }}">
                <input type="hidden" name="company_state" value="{{ $user->company_state ?? '' }}">
                <input type="hidden" name="company_postcode" value="{{ $user->company_postcode ?? '' }}">
                <input type="hidden" name="company_country" value="{{ $user->company_country ?? '' }}">
                <input type="hidden" name="company_phone" value="{{ $user->company_phone ?? '' }}">
                <input type="hidden" name="company_email" value="{{ $user->company_email ?? '' }}">
                <input type="hidden" name="company_website" value="{{ $user->company_website ?? '' }}">
                <input type="hidden" name="use_own_payment_gateway" value="{{ $user->use_own_payment_gateway ? 1 : 0 }}">
                <input type="hidden" name="bayarcash_portal_key" value="{{ $user->bayarcash_portal_key ?? '' }}">
                <input type="hidden" name="bayarcash_pat" value="{{ $user->bayarcash_pat ?? '' }}">
                <input type="hidden" name="bayarcash_secret_key" value="{{ $user->bayarcash_secret_key ?? '' }}">
                <input type="hidden" name="bayarcash_environment" value="{{ $user->bayarcash_environment ?? '' }}">

                <div class="space-y-4">
                    <!-- Profile Photo -->
                    <div>
                        <label class="block text-sm font-medium mb-2">Profile Photo</label>
                        @if($user->profile_photo)
                            <div class="mb-3">
                                <img src="{{ asset($user->profile_photo) }}" alt="Profile Photo"
                                     class="h-32 w-32 rounded-full object-cover border-4 border-gray-200 dark:border-gray-600">
                            </div>
                        @else
                            <div class="mb-3">
                                <div class="h-32 w-32 rounded-full bg-gradient-to-br from-blue-400 to-indigo-600 flex items-center justify-center text-white text-4xl font-bold border-4 border-gray-200 dark:border-gray-600">
                                    {{ strtoupper(substr($user->name, 0, 1)) }}
                                </div>
                            </div>
                        @endif
                        <input type="file" name="profile_photo" accept="image/*"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700">
                        <p class="text-xs text-gray-500 mt-1">Max 2MB. Square image (1:1 ratio) recommended</p>
                        @error('profile_photo')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- Name -->
                    <div>
                        <label class="block text-sm font-medium mb-2">Name</label>
                        <input type="text" name="name" value="{{ old('name', $user->name) }}"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                               required>
                        @error('name')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- Short Name -->
                    <div>
                        <label class="block text-sm font-medium mb-2">Short Name</label>
                        <input type="text" name="short_name" value="{{ old('short_name', $user->short_name) }}"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                               maxlength="20"
                               placeholder="e.g., ADMIN, SHOP1">
                        <p class="text-xs text-gray-500 mt-1">Used in order numbers (e.g., SP-ADMIN-1)</p>
                        @error('short_name')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- URL Slug (Sales Page URL) -->
                    @if($user->isSeller() || $user->isAdmin())
                    <div class="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                        <label class="block text-sm font-medium mb-3 text-gray-900 dark:text-gray-100">Sales Page URL</label>
                        <div class="flex rounded-lg overflow-hidden shadow-sm">
                            <span class="inline-flex items-center px-4 py-2.5 bg-gray-100 dark:bg-gray-800 border border-r-0 border-gray-300 dark:border-gray-600 text-gray-700 dark:text-gray-300 text-sm font-medium whitespace-nowrap">
                                {{ url('/p/') }}/
                            </span>
                            <input type="text" name="url_slug" value="{{ old('url_slug', $user->url_slug) }}"
                                   class="flex-1 px-4 py-2.5 border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all"
                                   maxlength="50"
                                   placeholder="your-shop-name"
                                   pattern="[a-z0-9-]+"
                                   title="Only lowercase letters, numbers, and hyphens allowed">
                        </div>
                        <div class="mt-3 space-y-1.5">
                            <p class="text-xs text-gray-700 dark:text-gray-300">
                                Your sales pages will be accessible at:
                            </p>
                            <p class="text-sm font-mono font-semibold text-blue-600 dark:text-blue-400 bg-white dark:bg-gray-800 px-3 py-2 rounded border border-blue-200 dark:border-blue-700">
                                {{ url('/p/') }}/{{ $user->url_slug ?: 'your-slug' }}/page-name
                            </p>
                            <p class="text-xs text-gray-600 dark:text-gray-400 pt-1">
                                💡 Only lowercase letters, numbers, and hyphens. Must be unique.
                            </p>
                        </div>
                        @error('url_slug')
                            <p class="text-red-500 text-sm mt-2">{{ $message }}</p>
                        @enderror
                    </div>
                    @endif

                    <!-- Phone -->
                    <div>
                        <label class="block text-sm font-medium mb-2">Phone Number</label>
                        <input type="text" name="phone" value="{{ old('phone', $user->phone) }}"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                               placeholder="+60123456789">
                        <p class="text-xs text-gray-500 mt-1">Displayed on seller landing page</p>
                        @error('phone')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- Auto-fill Customer Information Toggle -->
                    @if($user->isSeller() || $user->isAdmin())
                    <div class="border border-gray-200 dark:border-gray-600 rounded-lg p-4 bg-gray-50 dark:bg-gray-900">
                        <div class="flex items-start gap-3">
                            <div class="flex-shrink-0 mt-0.5">
                                <input type="checkbox" name="auto_fill_customer_info" id="auto_fill_customer_info" value="1"
                                       {{ old('auto_fill_customer_info', $user->auto_fill_customer_info) ? 'checked' : '' }}
                                       class="w-4 h-4 rounded text-blue-600 focus:ring-2 focus:ring-blue-500 border-gray-300 dark:border-gray-600">
                            </div>
                            <div class="flex-1">
                                <label for="auto_fill_customer_info" class="block text-sm font-medium cursor-pointer mb-1">
                                    🎯 Auto-fill Customer Information
                                </label>
                                <p class="text-xs text-gray-600 dark:text-gray-400 mb-2">
                                    Automatically populate customer details when phone number matches your records
                                </p>
                                <div class="text-xs text-gray-500 dark:text-gray-500 space-y-1">
                                    <div class="flex items-start gap-1.5">
                                        <span class="text-green-600 dark:text-green-400 mt-0.5">💡</span>
                                        <span>Saves time by remembering repeat customers</span>
                                    </div>
                                    <div class="flex items-start gap-1.5">
                                        <span class="text-green-600 dark:text-green-400 mt-0.5">⚡</span>
                                        <span>Faster order creation with fewer typing errors</span>
                                    </div>
                                    <div class="flex items-start gap-1.5">
                                        <span class="text-blue-600 dark:text-blue-400 mt-0.5">🔒</span>
                                        <span>Privacy protected - only your customers shown</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    @endif

                    <!-- Email -->
                    <div>
                        <label class="block text-sm font-medium mb-2">Email</label>
                        <input type="email" name="email" value="{{ old('email', $user->email) }}"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                               required>
                        @error('email')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <div class="pt-4">
                        <button type="submit" class="w-full px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors">
                            Save Personal Info
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Change Password Card -->
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
            <h2 class="text-lg font-semibold mb-6">Change Password</h2>

            <form action="{{ route('profile.password') }}" method="POST">
                @csrf
                @method('PUT')

                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium mb-2">Current Password</label>
                        <input type="password" name="current_password"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                               placeholder="Enter current password"
                               required>
                        @error('current_password')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <div>
                        <label class="block text-sm font-medium mb-2">New Password</label>
                        <input type="password" name="password"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                               placeholder="Enter new password"
                               required>
                        @error('password')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <div>
                        <label class="block text-sm font-medium mb-2">Confirm New Password</label>
                        <input type="password" name="password_confirmation"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                               placeholder="Confirm new password"
                               required>
                    </div>

                    <div class="pt-4">
                        <button type="submit" class="w-full px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 transition-colors">
                            Update Password
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    @if($user->isAdmin() || $user->isManager())
    <!-- Company Information (Admin & Manager Only) -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow p-6 mb-6">
        <h2 class="text-lg font-semibold mb-4">Company Information (for Invoices)</h2>
        <p class="text-sm text-gray-600 dark:text-gray-400 mb-4">
            This information will appear on all invoices generated from orders.
        </p>

        <form action="{{ route('profile.update') }}" method="POST" enctype="multipart/form-data">
            @csrf
            @method('PUT')

            <!-- Hidden fields to preserve personal info -->
            <input type="hidden" name="name" value="{{ $user->name }}">
            <input type="hidden" name="email" value="{{ $user->email }}">
            <input type="hidden" name="phone" value="{{ $user->phone }}">

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium mb-2">Company Name</label>
                    <input type="text" name="company_name" value="{{ old('company_name', $user->company_name) }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    @error('company_name')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">Registration No.</label>
                    <input type="text" name="company_registration" value="{{ old('company_registration', $user->company_registration) }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                           placeholder="SSM Registration Number">
                    @error('company_registration')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div class="md:col-span-2">
                    <label class="block text-sm font-medium mb-2">Address</label>
                    <textarea name="company_address" rows="2"
                              class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">{{ old('company_address', $user->company_address) }}</textarea>
                    @error('company_address')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">City</label>
                    <input type="text" name="company_city" value="{{ old('company_city', $user->company_city) }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    @error('company_city')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">State</label>
                    <input type="text" name="company_state" value="{{ old('company_state', $user->company_state) }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    @error('company_state')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">Postcode</label>
                    <input type="text" name="company_postcode" value="{{ old('company_postcode', $user->company_postcode) }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    @error('company_postcode')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">Country</label>
                    <input type="text" name="company_country" value="{{ old('company_country', $user->company_country ?? 'Malaysia') }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    @error('company_country')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">Phone</label>
                    <input type="text" name="company_phone" value="{{ old('company_phone', $user->company_phone) }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                           placeholder="+60123456789">
                    @error('company_phone')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">Email</label>
                    <input type="email" name="company_email" value="{{ old('company_email', $user->company_email) }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white">
                    @error('company_email')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div>
                    <label class="block text-sm font-medium mb-2">Website</label>
                    <input type="url" name="company_website" value="{{ old('company_website', $user->company_website) }}"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                           placeholder="https://example.com">
                    @error('company_website')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>

                <div class="md:col-span-2">
                    <label class="block text-sm font-medium mb-2">Company Logo</label>
                    @if($user->company_logo)
                        <div class="mb-2">
                            <img src="{{ asset($user->company_logo) }}" alt="Company Logo" class="h-20 object-contain">
                        </div>
                    @endif
                    <input type="file" name="company_logo" accept="image/*"
                           class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700">
                    <p class="text-xs text-gray-500 mt-1">Max 2MB. Recommended: 200x200px PNG with transparent background</p>
                    @error('company_logo')
                        <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                    @enderror
                </div>
            </div>

            <div class="mt-6 flex justify-end">
                <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                    Save Company Info
                </button>
            </div>
        </form>
    </div>

    <!-- Payment Gateway Settings (Admin & Manager Only) -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow p-6 mb-6">
        <h2 class="text-lg font-semibold mb-4">Payment Gateway Settings (BayarCash)</h2>
        <p class="text-sm text-gray-600 dark:text-gray-400 mb-4">
            Configure your own BayarCash payment gateway for sales pages. If not configured, the system admin's gateway will be used.
        </p>

        <form action="{{ route('profile.update') }}" method="POST">
            @csrf
            @method('PUT')

            <!-- Hidden fields to preserve other info -->
            <input type="hidden" name="name" value="{{ $user->name }}">
            <input type="hidden" name="email" value="{{ $user->email }}">
            <input type="hidden" name="phone" value="{{ $user->phone }}">
            <input type="hidden" name="company_name" value="{{ $user->company_name }}">
            <input type="hidden" name="company_registration" value="{{ $user->company_registration }}">
            <input type="hidden" name="company_address" value="{{ $user->company_address }}">
            <input type="hidden" name="company_city" value="{{ $user->company_city }}">
            <input type="hidden" name="company_state" value="{{ $user->company_state }}">
            <input type="hidden" name="company_postcode" value="{{ $user->company_postcode }}">
            <input type="hidden" name="company_country" value="{{ $user->company_country }}">
            <input type="hidden" name="company_phone" value="{{ $user->company_phone }}">
            <input type="hidden" name="company_email" value="{{ $user->company_email }}">
            <input type="hidden" name="company_website" value="{{ $user->company_website }}">

            <!-- Toggle Own Gateway -->
            <div class="mb-6 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border-2 border-blue-200 dark:border-blue-800">
                <label class="flex items-center gap-3 cursor-pointer">
                    <input type="checkbox" name="use_own_payment_gateway" value="1"
                           {{ old('use_own_payment_gateway', $user->use_own_payment_gateway) ? 'checked' : '' }}
                           id="use-own-gateway"
                           class="rounded h-5 w-5 text-blue-600 border-gray-300 focus:ring-blue-500">
                    <div>
                        <div class="font-semibold text-gray-900 dark:text-white">Use My Own Payment Gateway</div>
                        <div class="text-sm text-gray-600 dark:text-gray-400">Enable this to use your own BayarCash credentials instead of the admin's gateway</div>
                    </div>
                </label>
            </div>

            <div id="gateway-credentials" class="{{ old('use_own_payment_gateway', $user->use_own_payment_gateway) ? '' : 'hidden' }}">
                <div class="space-y-4">
                    <!-- Environment -->
                    <div>
                        <label class="block text-sm font-medium mb-3">Environment <span class="text-red-500">*</span></label>
                        <div class="flex gap-4">
                            <label class="flex items-center gap-2 p-3 border-2 rounded-lg cursor-pointer {{ old('bayarcash_environment', $user->bayarcash_environment) === 'sandbox' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600' }}">
                                <input type="radio" name="bayarcash_environment" value="sandbox"
                                       {{ old('bayarcash_environment', $user->bayarcash_environment ?? 'sandbox') === 'sandbox' ? 'checked' : '' }}
                                       class="h-4 w-4 text-blue-600">
                                <div>
                                    <div class="font-semibold text-gray-900 dark:text-white">Sandbox</div>
                                    <div class="text-xs text-gray-600 dark:text-gray-400">For testing</div>
                                </div>
                            </label>
                            <label class="flex items-center gap-2 p-3 border-2 rounded-lg cursor-pointer {{ old('bayarcash_environment', $user->bayarcash_environment) === 'production' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600' }}">
                                <input type="radio" name="bayarcash_environment" value="production"
                                       {{ old('bayarcash_environment', $user->bayarcash_environment) === 'production' ? 'checked' : '' }}
                                       class="h-4 w-4 text-blue-600">
                                <div>
                                    <div class="font-semibold text-gray-900 dark:text-white">Production</div>
                                    <div class="text-xs text-gray-600 dark:text-gray-400">Live payments</div>
                                </div>
                            </label>
                        </div>
                    </div>

                    <!-- Portal Key -->
                    <div>
                        <label class="block text-sm font-medium mb-2">Portal Key</label>
                        <input type="text" name="bayarcash_portal_key" value="{{ old('bayarcash_portal_key', $user->bayarcash_portal_key) }}"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white font-mono text-sm"
                               placeholder="Your BayarCash Portal Key">
                        <p class="text-xs text-gray-500 mt-1">Get this from your BayarCash account</p>
                        @error('bayarcash_portal_key')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- PAT -->
                    <div>
                        <label class="block text-sm font-medium mb-2">Personal Access Token (PAT)</label>
                        <textarea name="bayarcash_pat" rows="3"
                                  class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white font-mono text-sm"
                                  placeholder="Your BayarCash Personal Access Token">{{ old('bayarcash_pat', $user->bayarcash_pat) }}</textarea>
                        <p class="text-xs text-gray-500 mt-1">Get this from your BayarCash account</p>
                        @error('bayarcash_pat')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- API Secret Key -->
                    <div>
                        <label class="block text-sm font-medium mb-2">API Secret Key</label>
                        <input type="text" name="bayarcash_secret_key" value="{{ old('bayarcash_secret_key', $user->bayarcash_secret_key) }}"
                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white font-mono text-sm"
                               placeholder="Your BayarCash API Secret Key">
                        <p class="text-xs text-gray-500 mt-1">Used for verifying payment callbacks</p>
                        @error('bayarcash_secret_key')
                            <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                        @enderror
                    </div>

                    <!-- Payment Gateway Fee Settings -->
                    <div class="mt-6 p-4 bg-gray-50 dark:bg-gray-900/50 rounded-lg border border-gray-200 dark:border-gray-700">
                        <h3 class="text-base font-semibold mb-3 text-gray-900 dark:text-white">Payment Gateway Fees</h3>
                        <p class="text-xs text-gray-600 dark:text-gray-400 mb-4">Configure fees for online payment methods. These fees will be applied to all sales page transactions.</p>

                        <!-- FPX (Online Banking) Fees -->
                        <div class="mb-6 p-3 bg-blue-50 dark:bg-blue-900/10 rounded-lg border border-blue-200 dark:border-blue-800">
                            <h4 class="text-sm font-semibold mb-2 text-blue-900 dark:text-blue-300">FPX (Online Banking)</h4>

                            <div class="space-y-3">
                                <div>
                                    <label class="block text-sm font-medium mb-2">Fee Type</label>
                                    <select name="fpx_fee_type" id="fpx-fee-type-selector"
                                            class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white text-sm">
                                        <option value="none" {{ old('fpx_fee_type', $user->fpx_fee_type ?? 'none') === 'none' ? 'selected' : '' }}>No Fee</option>
                                        <option value="fixed" {{ old('fpx_fee_type', $user->fpx_fee_type) === 'fixed' ? 'selected' : '' }}>Fixed Amount</option>
                                        <option value="percentage" {{ old('fpx_fee_type', $user->fpx_fee_type) === 'percentage' ? 'selected' : '' }}>Percentage</option>
                                        <option value="both" {{ old('fpx_fee_type', $user->fpx_fee_type) === 'both' ? 'selected' : '' }}>Both (Fixed + Percentage)</option>
                                    </select>
                                </div>

                                <div id="fpx-fee-amount-fields" class="space-y-3">
                                    <div id="fpx-fixed-fee-field" class="hidden">
                                        <label class="block text-sm font-medium mb-2">Fixed Fee Amount (MYR)</label>
                                        <input type="number" name="fpx_fixed_fee" step="0.01" min="0"
                                               value="{{ old('fpx_fixed_fee', $user->fpx_fixed_fee ?? 0) }}"
                                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white text-sm"
                                               placeholder="0.00">
                                        <p class="text-xs text-gray-500 mt-1">e.g., RM 1.50 per transaction</p>
                                    </div>

                                    <div id="fpx-percentage-fee-field" class="hidden">
                                        <label class="block text-sm font-medium mb-2">Percentage Fee (%)</label>
                                        <input type="number" name="fpx_percentage_fee" step="0.01" min="0" max="100"
                                               value="{{ old('fpx_percentage_fee', $user->fpx_percentage_fee ?? 0) }}"
                                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white text-sm"
                                               placeholder="0.00">
                                        <p class="text-xs text-gray-500 mt-1">e.g., 2.5 for 2.5%</p>
                                    </div>
                                </div>

                                <div>
                                    <label class="block text-sm font-medium mb-2">Who Pays?</label>
                                    <div class="flex gap-2">
                                        <label class="flex-1 flex items-center gap-2 p-2 border-2 rounded-lg cursor-pointer {{ old('fpx_fee_bearer', $user->fpx_fee_bearer ?? 'customer') === 'customer' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600' }}">
                                            <input type="radio" name="fpx_fee_bearer" value="customer"
                                                   {{ old('fpx_fee_bearer', $user->fpx_fee_bearer ?? 'customer') === 'customer' ? 'checked' : '' }}
                                                   class="h-4 w-4 text-blue-600">
                                            <div class="text-xs">
                                                <div class="font-semibold text-gray-900 dark:text-white">Customer</div>
                                            </div>
                                        </label>
                                        <label class="flex-1 flex items-center gap-2 p-2 border-2 rounded-lg cursor-pointer {{ old('fpx_fee_bearer', $user->fpx_fee_bearer ?? 'customer') === 'merchant' ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'border-gray-300 dark:border-gray-600' }}">
                                            <input type="radio" name="fpx_fee_bearer" value="merchant"
                                                   {{ old('fpx_fee_bearer', $user->fpx_fee_bearer) === 'merchant' ? 'checked' : '' }}
                                                   class="h-4 w-4 text-blue-600">
                                            <div class="text-xs">
                                                <div class="font-semibold text-gray-900 dark:text-white">Merchant</div>
                                            </div>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- DuitNow QR Fees -->
                        <div class="p-3 bg-purple-50 dark:bg-purple-900/10 rounded-lg border border-purple-200 dark:border-purple-800">
                            <h4 class="text-sm font-semibold mb-2 text-purple-900 dark:text-purple-300">DuitNow QR</h4>

                            <div class="space-y-3">
                                <div>
                                    <label class="block text-sm font-medium mb-2">Fee Type</label>
                                    <select name="duitnow_fee_type" id="duitnow-fee-type-selector"
                                            class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white text-sm">
                                        <option value="none" {{ old('duitnow_fee_type', $user->duitnow_fee_type ?? 'none') === 'none' ? 'selected' : '' }}>No Fee</option>
                                        <option value="fixed" {{ old('duitnow_fee_type', $user->duitnow_fee_type) === 'fixed' ? 'selected' : '' }}>Fixed Amount</option>
                                        <option value="percentage" {{ old('duitnow_fee_type', $user->duitnow_fee_type) === 'percentage' ? 'selected' : '' }}>Percentage</option>
                                        <option value="both" {{ old('duitnow_fee_type', $user->duitnow_fee_type) === 'both' ? 'selected' : '' }}>Both (Fixed + Percentage)</option>
                                    </select>
                                </div>

                                <div id="duitnow-fee-amount-fields" class="space-y-3">
                                    <div id="duitnow-fixed-fee-field" class="hidden">
                                        <label class="block text-sm font-medium mb-2">Fixed Fee Amount (MYR)</label>
                                        <input type="number" name="duitnow_fixed_fee" step="0.01" min="0"
                                               value="{{ old('duitnow_fixed_fee', $user->duitnow_fixed_fee ?? 0) }}"
                                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white text-sm"
                                               placeholder="0.00">
                                        <p class="text-xs text-gray-500 mt-1">e.g., RM 1.00 per transaction</p>
                                    </div>

                                    <div id="duitnow-percentage-fee-field" class="hidden">
                                        <label class="block text-sm font-medium mb-2">Percentage Fee (%)</label>
                                        <input type="number" name="duitnow_percentage_fee" step="0.01" min="0" max="100"
                                               value="{{ old('duitnow_percentage_fee', $user->duitnow_percentage_fee ?? 0) }}"
                                               class="w-full px-3 py-2 rounded-md border border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white text-sm"
                                               placeholder="0.00">
                                        <p class="text-xs text-gray-500 mt-1">e.g., 2.0 for 2%</p>
                                    </div>
                                </div>

                                <div>
                                    <label class="block text-sm font-medium mb-2">Who Pays?</label>
                                    <div class="flex gap-2">
                                        <label class="flex-1 flex items-center gap-2 p-2 border-2 rounded-lg cursor-pointer {{ old('duitnow_fee_bearer', $user->duitnow_fee_bearer ?? 'customer') === 'customer' ? 'border-purple-500 bg-purple-50 dark:bg-purple-900/20' : 'border-gray-300 dark:border-gray-600' }}">
                                            <input type="radio" name="duitnow_fee_bearer" value="customer"
                                                   {{ old('duitnow_fee_bearer', $user->duitnow_fee_bearer ?? 'customer') === 'customer' ? 'checked' : '' }}
                                                   class="h-4 w-4 text-purple-600">
                                            <div class="text-xs">
                                                <div class="font-semibold text-gray-900 dark:text-white">Customer</div>
                                            </div>
                                        </label>
                                        <label class="flex-1 flex items-center gap-2 p-2 border-2 rounded-lg cursor-pointer {{ old('duitnow_fee_bearer', $user->duitnow_fee_bearer ?? 'customer') === 'merchant' ? 'border-purple-500 bg-purple-50 dark:bg-purple-900/20' : 'border-gray-300 dark:border-gray-600' }}">
                                            <input type="radio" name="duitnow_fee_bearer" value="merchant"
                                                   {{ old('duitnow_fee_bearer', $user->duitnow_fee_bearer) === 'merchant' ? 'checked' : '' }}
                                                   class="h-4 w-4 text-purple-600">
                                            <div class="text-xs">
                                                <div class="font-semibold text-gray-900 dark:text-white">Merchant</div>
                                            </div>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="p-3 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-800">
                        <p class="text-sm text-yellow-800 dark:text-yellow-300">
                            <strong>Important:</strong> Keep your credentials secure. Never share them publicly.
                        </p>
                    </div>
                </div>
            </div>

            <div class="mt-6 flex justify-end">
                <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                    Save Payment Settings
                </button>
            </div>
        </form>
    </div>
    @endif
</div>

@push('scripts')
<script>
    // Toggle payment gateway credentials visibility
    const useOwnGateway = document.getElementById('use-own-gateway');
    const gatewayCredentials = document.getElementById('gateway-credentials');

    if (useOwnGateway) {
        useOwnGateway.addEventListener('change', function() {
            if (this.checked) {
                gatewayCredentials.classList.remove('hidden');
            } else {
                gatewayCredentials.classList.add('hidden');
            }
        });
    }

    // Toggle FPX fee amount fields based on fee type selection
    const fpxFeeTypeSelector = document.getElementById('fpx-fee-type-selector');
    const fpxFixedFeeField = document.getElementById('fpx-fixed-fee-field');
    const fpxPercentageFeeField = document.getElementById('fpx-percentage-fee-field');

    function updateFpxFeeFields() {
        const feeType = fpxFeeTypeSelector.value;

        // Hide all fields first
        fpxFixedFeeField.classList.add('hidden');
        fpxPercentageFeeField.classList.add('hidden');

        // Show fields based on selected type
        if (feeType === 'fixed' || feeType === 'both') {
            fpxFixedFeeField.classList.remove('hidden');
        }
        if (feeType === 'percentage' || feeType === 'both') {
            fpxPercentageFeeField.classList.remove('hidden');
        }
    }

    if (fpxFeeTypeSelector) {
        // Initialize on page load
        updateFpxFeeFields();

        // Update when selection changes
        fpxFeeTypeSelector.addEventListener('change', updateFpxFeeFields);
    }

    // Toggle DuitNow fee amount fields based on fee type selection
    const duitnowFeeTypeSelector = document.getElementById('duitnow-fee-type-selector');
    const duitnowFixedFeeField = document.getElementById('duitnow-fixed-fee-field');
    const duitnowPercentageFeeField = document.getElementById('duitnow-percentage-fee-field');

    function updateDuitnowFeeFields() {
        const feeType = duitnowFeeTypeSelector.value;

        // Hide all fields first
        duitnowFixedFeeField.classList.add('hidden');
        duitnowPercentageFeeField.classList.add('hidden');

        // Show fields based on selected type
        if (feeType === 'fixed' || feeType === 'both') {
            duitnowFixedFeeField.classList.remove('hidden');
        }
        if (feeType === 'percentage' || feeType === 'both') {
            duitnowPercentageFeeField.classList.remove('hidden');
        }
    }

    if (duitnowFeeTypeSelector) {
        // Initialize on page load
        updateDuitnowFeeFields();

        // Update when selection changes
        duitnowFeeTypeSelector.addEventListener('change', updateDuitnowFeeFields);
    }
</script>
@endpush
@endsection
