@extends('layouts.app')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-2xl font-bold text-gray-900 dark:text-white">🎯 Set Seller Targets</h1>
            <p class="text-gray-600 dark:text-gray-400">Set monthly sales targets for all sellers</p>
        </div>
        <div>
            <a href="{{ route('forecasting.seller-kpi.index') }}" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-md">
                Back to Seller KPIs
            </a>
        </div>
    </div>

    @include('forecasting.partials.navigation')

    @if(session('success'))
    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-6">
        {{ session('success') }}
    </div>
    @endif

    <!-- Period Selector -->
    <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-4 mb-6">
        <form method="GET" action="{{ route('forecasting.seller-kpi.set-targets') }}" class="flex gap-4">
            <div>
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Year</label>
                <select name="year" class="form-select rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white" onchange="this.form.submit()">
                    @for($y = date('Y'); $y <= date('Y') + 1; $y++)
                        <option value="{{ $y }}" {{ $year == $y ? 'selected' : '' }}>{{ $y }}</option>
                    @endfor
                </select>
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Month</label>
                <select name="month" class="form-select rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white" onchange="this.form.submit()">
                    @foreach(['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'] as $m => $monthName)
                        <option value="{{ $m + 1 }}" {{ $month == ($m + 1) ? 'selected' : '' }}>{{ $monthName }}</option>
                    @endforeach
                </select>
            </div>
        </form>
    </div>

    <!-- Bulk Target Setting -->
    <div class="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4 mb-6">
        <h3 class="text-lg font-semibold text-blue-900 dark:text-blue-300 mb-3">Set Same Target for All Sellers</h3>
        <form id="bulkTargetForm" class="flex gap-4 items-end">
            <div class="flex-1">
                <label class="block text-sm font-medium text-blue-800 dark:text-blue-400 mb-1">Target Amount (RM)</label>
                <input type="number" step="0.01" id="bulkTargetAmount"
                       class="form-input w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                       placeholder="e.g., 25000">
            </div>
            <div>
                <button type="button" onclick="applyBulkTarget()" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-md">
                    Apply to All
                </button>
            </div>
        </form>
    </div>

    <!-- Individual Targets Form -->
    <form method="POST" action="{{ route('forecasting.seller-kpi.save-targets') }}">
        @csrf
        <input type="hidden" name="year" value="{{ $year }}">
        <input type="hidden" name="month" value="{{ $month }}">

        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm overflow-hidden mb-6">
            <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-700 flex items-center justify-between">
                <h2 class="text-xl font-bold text-gray-800 dark:text-white">
                    Individual Targets - {{ ['', 'January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'][$month] }} {{ $year }}
                </h2>
                <button type="submit" class="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded-md font-semibold">
                    Save All Targets
                </button>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead class="bg-gray-50 dark:bg-gray-700">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Seller</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Current Target (RM)</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">New Target (RM)</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Actual Sales (RM)</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Current Grade</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-200 dark:divide-gray-700">
                        @forelse($sellers as $seller)
                            @php
                                $forecast = $existingForecasts->where('user_id', $seller->id)->first();
                                $currentTarget = $forecast ? $forecast->target_sales : 0;
                                $actualSales = $forecast ? $forecast->actual_sales : 0;
                                $currentGrade = $forecast ? $forecast->performance_grade : '-';

                                $gradeColors = [
                                    'A' => 'bg-green-100 text-green-800',
                                    'B' => 'bg-blue-100 text-blue-800',
                                    'C' => 'bg-yellow-100 text-yellow-800',
                                    'D' => 'bg-orange-100 text-orange-800',
                                    'F' => 'bg-red-100 text-red-800',
                                ];
                            @endphp
                            <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm font-medium text-gray-900 dark:text-white">{{ $seller->name }}</div>
                                    <div class="text-xs text-gray-500 dark:text-gray-400">{{ $seller->email }}</div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                    {{ number_format($currentTarget, 2) }}
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <input type="hidden" name="targets[{{ $loop->index }}][user_id]" value="{{ $seller->id }}">
                                    <input type="number"
                                           step="0.01"
                                           name="targets[{{ $loop->index }}][target_sales]"
                                           value="{{ old('targets.' . $loop->index . '.target_sales', $currentTarget) }}"
                                           class="target-input form-input w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white"
                                           placeholder="Enter target">
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-900 dark:text-white">
                                    {{ number_format($actualSales, 2) }}
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    @if($currentGrade && $currentGrade !== '-')
                                        <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full {{ $gradeColors[$currentGrade] ?? 'bg-gray-100 text-gray-800' }}">
                                            {{ $currentGrade }}
                                        </span>
                                    @else
                                        <span class="text-gray-400">No target set</span>
                                    @endif
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="5" class="px-6 py-8 text-center text-gray-500 dark:text-gray-400">
                                    No sellers found. Please create sellers first.
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>

        @if($sellers->count() > 0)
        <div class="flex justify-end gap-4">
            <button type="submit" class="bg-green-600 hover:bg-green-700 text-white px-8 py-3 rounded-md font-semibold text-lg">
                Save All Targets
            </button>
        </div>
        @endif
    </form>

    <!-- Help Text -->
    <div class="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4 mt-6">
        <h4 class="font-semibold text-yellow-900 dark:text-yellow-300 mb-2">💡 Tips:</h4>
        <ul class="text-sm text-yellow-800 dark:text-yellow-400 space-y-1">
            <li>• Use "Set Same Target for All" to quickly apply one target to all sellers</li>
            <li>• Or set individual targets for each seller</li>
            <li>• Targets are used to calculate achievement % and performance grades</li>
            <li>• After saving, go back to Seller KPIs and click "Sync Sales Data" to update grades</li>
            <li>• Grade scale: A (100%+), B (80-99%), C (60-79%), D (40-59%), F (<40%)</li>
        </ul>
    </div>
</div>

<script>
// Apply bulk target to all input fields
function applyBulkTarget() {
    const bulkAmount = document.getElementById('bulkTargetAmount').value;

    if (!bulkAmount || bulkAmount <= 0) {
        alert('Please enter a valid target amount');
        return;
    }

    const inputs = document.querySelectorAll('.target-input');
    inputs.forEach(input => {
        input.value = bulkAmount;
    });

    alert(`Applied RM ${parseFloat(bulkAmount).toLocaleString()} to all ${inputs.length} sellers`);
}
</script>
@endsection
